/*
 * This file is part of TraceProto.
 * Copyright 2004 Eric Hope
 * skippysaurus@skippylair.net
 *
 * TraceProto is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * TraceProto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TraceProto; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <libnet.h>
#include <pcap.h>
#include <errno.h>
#include <sys/time.h>
#include <stdio.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <netdb.h>
#include <ctype.h>
#include <unistd.h>

#include "traceproto.h"
#include "config.h"

int main( int argc, char * argv[] )
{

	int cmd_arg, i;
	char * opt_string;

/*
 * regularize the program name
 */
	if ( reg_name( argv[ 0 ] ) == 0 )
	{
		printf ( "setup error, unable to parse program name\n" );
		tixe ( tixe_cleanup, 1 );
	}

/*
 * set the defaults
 */
	behavior.continuous = NO;
	behavior.packets_per_hop = 3;
	behavior.src_port_incr = 1;	// src port
	behavior.dst_port_incr = 0;	// dst port
	behavior.min_src_port = 10240;
	behavior.rndm_src_port = YES;
	behavior.min_dst_port = 80;
	behavior.max_ttl = 30;			// hops
	behavior.min_ttl = 1;			// hop(s)
	behavior.wait_timeout = 5;		// seconds
	behavior.wait_between_packets = 100;	// milliseconds
	behavior.protocol = "tcp";
	behavior.tcp_resets = YES;
	behavior.account_level = TP_ACCOUNT_FULL;
	behavior.report = report_std;
	behavior.do_audit = NO;
	behavior.do_audit_exit = NO;
	behavior.do_skip = NO;
	behavior.output_style = TP_STD_OUTPUT;
	behavior.payload_size = 12;
	behavior.libnet_resolve_choice = LIBNET_RESOLVE;
	behavior.hop_incr_unit = 1;
	behavior.filter_text = "( host %s and dst port %d and src port %d ) or ( icmp[12:2] == %d )";

	packet.protocol_number = IPPROTO_TCP;
	packet.payload = 0;
	packet.frag_bit = 0;
	packet.tcp_flags = TH_SYN;

	state.tcp_h  = LIBNET_PTAG_INITIALIZER;
	state.udp_h  = LIBNET_PTAG_INITIALIZER;
	state.icmp_h = LIBNET_PTAG_INITIALIZER;
	state.ip_h   = LIBNET_PTAG_INITIALIZER;
	state.incr_error = NO;
	state.target_response = NO;
	state.continuous_count = ( int ) NULL;

	/*
	 * If called as "traceroute" try to be a drop-in replacement.
	 * Currently we aren't an exact copy, but most of the differences
	 * are a matter of different command line flags for the original
	 * features.  We can more or less produce the same output behavior. 
	 */
	if ( strncmp ( state.prog, "traceroute", 10 ) == 0 )
	{
		behavior.output_style = TP_CLASSIC_OUTPUT;
		behavior.protocol = "udp";
		behavior.min_dst_port = 32768 + 666;
		behavior.account_level = TP_ACCOUNT_NONE;
		behavior.src_port_incr = 0;	// src port
		behavior.dst_port_incr = 1;	// dst port
	}

/*
 * get command line args
 */
	while ( ( cmd_arg = getopt ( argc, argv,
			"cCvnhRfp:i:I:D:r:t:k:o:a:s:S:H:d:M:m:w:W:p:P:"
		) ) != EOF )
	{
		switch ( cmd_arg )
		{
		case 'c':
			behavior.continuous = YES;
			behavior.continuous_accounting = NO;
			break;
		case 'C':
			behavior.continuous = YES;
			behavior.continuous_accounting = YES;
			break;
		case 'I':
			state.continuous_count = atoi ( optarg );
			behavior.continuous = YES;
			break;
		case 'H':
			behavior.packets_per_hop = atoi ( optarg );
			break;
		case 'i':
			/*
			 * controls the port number on the two ends
			 * possiblities are src/dst and incr/decr/static
			 */
			opt_string = optarg;
			for ( i = strlen ( opt_string ); i >= 0; i-- )
			{
				switch ( opt_string [ i ] )
				{
				case 's':
					behavior.src_port_incr = -1;
					break;
				case 'S':
					behavior.src_port_incr = 1;	// default
					break;
				case 'd':
					behavior.dst_port_incr = -1;
					break;
				case 'D':
					behavior.dst_port_incr = 1;
					break;
				case 'n':
					behavior.src_port_incr = 0;
					break;
				case 'N':
					behavior.dst_port_incr = 0;	// default
					break;
				case '\0':
					break;
				default:
					state.incr_error = YES;
					break;
				}
			}
			break;
		case 'd':
			behavior.min_dst_port = atoi( optarg );
			break;
		case 'D':
			behavior.max_dst_port = atoi( optarg );
			break;
		case 'm':
			behavior.min_ttl = atoi ( optarg );
			break;
		case 'M':
			behavior.max_ttl = atoi( optarg );
			break;
		case 'n':
			behavior.libnet_resolve_choice = LIBNET_DONT_RESOLVE;
			break;
		case 'w':
			/* how long to wait to see if a response turns up */
			behavior.wait_timeout = atoi ( optarg );
			break;
		case 'W':
			/* how long to wait after a response ( or timeout )
			 * before sending the next probe */
			behavior.wait_between_packets = atoi ( optarg );
			break;
		case 's':
			behavior.min_src_port = atoi ( optarg );
			behavior.rndm_src_port = NO;
			break;
		case 'S':
			behavior.max_src_port = atoi ( optarg );
			break;
		case 'p':
			behavior.protocol = optarg;
			break;
		case 'a':
			behavior.account_level = atoi ( optarg );
			break;
		case 'k':
			/* a list of hops ( ie ttl numbers ) to skip */
			behavior.skip_str = optarg;
			behavior.do_skip = YES;
			break;
		case 'h':
			usage ( state.prog );
			tixe ( tixe_cleanup, 0 );
			break;
		case 'v':
			version( state.prog );
			tixe ( tixe_cleanup, 1 );
			break;
		case 'o':
			/* what should the printout look like */
			opt_string = optarg;
			if ( opt_string [ 0 ] == 's' )
			{
				behavior.output_style = TP_STD_OUTPUT;
				behavior.report = report_std;
				break;
			} else if ( opt_string [ 0 ] == 'g' ) {
				behavior.output_style = TP_GRAPHIC_OUTPUT;
				behavior.report = report_graphic;
				break;
			} else if ( opt_string [ 0 ] == 'c' ) {
				behavior.output_style = TP_CLASSIC_OUTPUT;
				behavior.report = report_classic;
				break;
			} else if ( opt_string [ 0 ] == 'n' ) {
				behavior.output_style = TP_NO_OUTPUT;
				behavior.report = report_none;
				break;
			} else if ( opt_string [ 0 ] == 'm' ) {
				behavior.output_style = TP_MIN_OUTPUT;
				behavior.report = report_minimum;
				break;
			} else if ( opt_string [ 0 ] == 'p' ) {
				behavior.output_style = TP_SCRIPT_OUTPUT;
				behavior.report = report_scriptable;
				break;
			} else {
				printf ( "invalid output style: %c\n", * opt_string );
				usage ( state.prog );
				tixe ( tixe_cleanup, 1 );
			}
			break;
		case 't':
			/* the tcp SYN/ACK/FIN/etc flags */
			opt_string = optarg;
			packet.tcp_flags = parse_flags ( opt_string );
			break;
		case 'P':
			behavior.payload_size = atoi ( optarg );
			break;
		case 'f':
			packet.frag_bit = TP_DONT_FRAG;
			break;
		case 'R':
			/* start at the distant end and trace consecutively
			 * closer until we get here */
			behavior.hop_incr_unit = -1;
			break;
		default:
			usage( state.prog );
			tixe ( tixe_cleanup, 1 );
			break;
		}
	}
	behavior.target = argv[optind];

/*
 * set the max ports and starting ports
 */
	if ( ! behavior.max_src_port )
		behavior.max_src_port = behavior.min_src_port
					+ ((( behavior.max_ttl - behavior.min_ttl )
					+ 1 )
					* behavior.packets_per_hop );

	if ( ! behavior.max_dst_port )
		behavior.max_dst_port = behavior.min_dst_port
					+ ((( behavior.max_ttl - behavior.min_ttl )
					+ 1 )
					* behavior.packets_per_hop );

	if ( behavior.src_port_incr == -1 )
		packet.src_port = behavior.max_src_port - behavior.src_port_incr;
	else
		packet.src_port = behavior.min_src_port - behavior.src_port_incr;

	if ( behavior.dst_port_incr == -1 )
		packet.dst_port = behavior.max_dst_port - behavior.dst_port_incr;
	else
		packet.dst_port = behavior.min_dst_port - behavior.dst_port_incr;

	if ( behavior.rndm_src_port == YES )
	{
		/* this needs to be random, just not very random */
		srand ( ( int ) getpid() );
		behavior.rndm_src_port = abs ( rand() % 256 );
		behavior.min_src_port += behavior.rndm_src_port;
		behavior.max_src_port += behavior.rndm_src_port;
	}

/*
 * sanity checks
 */

	/* we really do need to know where to trace to... */
	if ( behavior.target == NULL )
	{
		printf ( "invalid target\n" );
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

	if ( state.incr_error != NO )
	{
		printf ( "invalid incriment option specified with -i\n" );
		usage ( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

	/* limit the number of probes per hop */
	if ( behavior.packets_per_hop <= 0 || behavior.packets_per_hop > 10 )
	{
		printf("packets per hop must be between 1 and 10\n");
		behavior.packets_per_hop = 3;
	}

//	if ( behavior.continuous == YES && behavior.wait_between_packets < 200 )
//	{
//		behavior.wait_between_packets = 200;
//	}

	/* limits on the continuous count */
	if ( state.continuous_count > 2048 )
	{
		printf ( "warning: unusually large iteration number set with -I\n" );
	}

	/* set the protocol */
	if ( strncmp( behavior.protocol, "udp", 3 ) == 0 )
	{
		behavior.protocol = "udp";
		packet.protocol_number = IPPROTO_UDP;
	}
	else if ( strncmp( behavior.protocol, "tcp", 3 ) == 0 )
	{
		behavior.protocol = "tcp";
		packet.protocol_number = IPPROTO_TCP;
	}
	else if ( strncmp( behavior.protocol, "icm", 3 ) == 0 )
	{
		behavior.protocol = "icmp";
		packet.protocol_number = IPPROTO_ICMP;
	}
	else
	{
		printf("error: invalid protocol specified with -p\n");
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

	if ( packet.dst_port <= 0 || packet.dst_port > 65535 )
	{
		printf("error: illegal destination port specified with -d\n");
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

	if ( packet.src_port <= 0 || packet.dst_port > 65535 )
	{
		printf("error: illegal source port specified with -s\n");
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

	if ( behavior.max_ttl <= 0 )
	{
		printf("error: illegal maximum time to live value specified with -m\n");
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	} else if ( behavior.max_ttl > 256 ) {
		printf("warning: using an unusually high max ttl value\n");
	}

	if ( behavior.min_ttl <= 0 )
	{
		printf("error: illegal minimum time to live value specified with -n\n");
		behavior.min_ttl = 1;
	} else if ( behavior.min_ttl > behavior.max_ttl ) {
		printf("error: minimum ttl (-n) must be less than maximum ttl (-m)\n");
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

	if ( behavior.max_src_port < behavior.min_src_port )
	{
		printf ( "error: maximum src port ( specified with -S )\n" );
		printf ( "\tmust be greater than the minimum src port\n" );
		printf ( "\t( specified with -s or default 10240 )\n" );
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

	if ( behavior.wait_timeout <= 0 )
	{
		printf("error: illegal timeout specified with -w\n");
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	} else if ( behavior.wait_timeout > 300 ) {
		printf("warning: using and unusually high wait timeout specified with -w\n");
	}

	if ( behavior.wait_between_packets < 0 )
	{
		printf("error: illegal timeout specified with -z\n");
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	} else if ( behavior.wait_between_packets > 10000 ) {
		printf("warning: using an unusually high wait time between packets\n");
	}

	if ( packet.src_port == 0 )
	{
		printf("warning: using source port 0\n");
	}

	if ( behavior.account_level > TP_ACCOUNT_FULL )
	{
		behavior.account_level = TP_ACCOUNT_FULL;
	}
	else if ( behavior.account_level < TP_ACCOUNT_NONE )
	{
		behavior.account_level = TP_ACCOUNT_NONE;
	}

	/* there will be no overflows here, thank you. */
	if ( strlen ( behavior.filter_text )
			+ strlen ( behavior.target )
			+ 12 > FILTERSIZE )
	{
		printf ( "error: bpf filter size limit exceeded.  "
			"Specify a shorter target name\n" );
		tixe ( tixe_cleanup, 1 );
	}

	/*
	 * note that the default payload is 12 bytes due to a udp
	 * peculiarity.  less than 12 bytes and no udp response
	 * is returned.  Don't know if its the stack, libnet, pcap
	 * or sonething else completely.
	 */
	if ( behavior.payload_size != 0 )
	{
		if ( behavior.payload_size < 0 || behavior.payload_size > 8192 )
		{
			printf ( "illegal payload size\n" );
			usage ( state.prog );
			tixe ( tixe_cleanup, 1 );
		}
		packet.payload = ( char * ) malloc ( behavior.payload_size );
		if ( packet.payload == NULL )
		{
			printf ( "error allocating payload memory for payload\n" );
			tixe ( tixe_cleanup, 1 );
		}
		tixe_cleanup.payload_free = YES;
		memset ( packet.payload, '\0', behavior.payload_size );
	}

/*
 * DNS stuff
 */
	behavior.packed_target_reverse = gethostbyname ( behavior.target );
	if ( behavior.packed_target_reverse == NULL )
	{
		printf ( "error resolving target address %s\n", behavior.target );
		tixe ( tixe_cleanup, 1 );
	}

	behavior.target_reverse = inet_ntoa (
			*(struct in_addr *)( behavior.packed_target_reverse->h_addr_list[0]));

/*
 * initialize the skips list
 */
	if ( behavior.do_skip == YES && parse_skips ( behavior.skip_str ) != 0 )
	{
		fprintf ( stderr, "error with option k arguements\n" );
		usage ( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

/*
 * set up accounting
 */
	state.account_hops = behavior.max_ttl + 1;
	state.hop_record = ( struct hop_record * ) calloc (
						state.account_hops,
						sizeof ( struct hop_record ) );
	if ( state.hop_record == NULL )
	{
		perror ( "memory allocation error" );
		tixe ( tixe_cleanup, 1 );
	}
	tixe_cleanup.hop_record_free = YES;

/*
 * start packet
 */
	state.packet = libnet_init (
		LIBNET_RAW4,	    /* inject type */
		NULL,		   /* device */
		(u_char *) state.error_buff );
	if (!state.packet)
	{
		printf("Error: %s\n", state.error_buff);
		tixe ( tixe_cleanup, 1 );
	}
	tixe_cleanup.libnet_cleanup = YES;

	packet.packed_target = libnet_name2addr4(
		state.packet, behavior.target, LIBNET_RESOLVE );
	if ( packet.packed_target == -1 )
	{
		printf("error resolving destination (%s): %s\n",
			behavior.target, libnet_geterror( state.packet ) );
		usage( state.prog );
		tixe ( tixe_cleanup, 1 );
	}

	packet.packed_src = libnet_get_ipaddr4( state.packet );
	if ( packet.packed_src == -1 )
	{
		printf("error resolving source: %s\n",
			libnet_geterror( state.packet ) );
		tixe ( tixe_cleanup, 1 );
	}

	libnet_seed_prand( state.packet );

/*
 * do the pcap stuff and the initial filter
 */
	state.psocket = pcap_open_live( "any",
				SNAPLEN,
				NO_PROMISC,
				( behavior.wait_timeout * 1000 ),
				state.pc_error );
	if ( state.psocket == NULL )
	{
		printf("pcap error: %s\n", state.pc_error);
		tixe ( tixe_cleanup, 1 );
	}
	tixe_cleanup.pcap_cleanup = YES;

	/*
	 * this is a specific weirdness.  See the README for more details.
	 *
	 * The pcap_fileno is a bit borrowed from Michael Toren
	 * and his tcptraceroute.  Appearently the pcap documentation
	 * is blatently wrong here and the pcap_fileno actually returns
	 * the socket fd, not the fd of the logfile.
	 *
	 * So now you have a choice.  You can use my crude kludge to
	 * break the encapsulation, or you can use the "legit" but
	 * badly mis-documented library function....see the configure
	 * options.
	 */
#ifdef USE_PCAP_FILENO
	state.fake_psocket = ( struct fake_pcap * )
		malloc ( sizeof ( struct fake_pcap ) );
	if ( state.fake_psocket == NULL )
	{
		printf ( "error allocating pcap_fileno memory\n" );
		tixe ( tixe_cleanup, 1 );
	}
	tixe_cleanup.fake_psocket_free = YES;

	state.fake_psocket->fd  = pcap_fileno ( state.psocket );
#else
	state.fake_psocket = ( struct fake_pcap * ) state.psocket;
#endif /* USE_PCAP_FILENO */

/*
 * set up the initial filter
 */
	if ( do_filter() != 0 )
	{
		tixe ( tixe_cleanup, 1 );
	}

/*
 * drop the root privileges if possible
 */
	if ( setuid ( getuid ( ) ) != 0 )
	{
		printf ( "error dropping privileges\n" );
		tixe ( tixe_cleanup, 1 );
	}
	if ( setgid ( getgid ( ) ) != 0 )
	{
		printf ( "error dropping privileges\n" );
		tixe ( tixe_cleanup, 1 );
	}

/*
 * set up the signals
 */
	signal ( SIGINT, ctrl_c );
	signal ( SIGTSTP, ctrl_z );

/*
 * ttl setup depending on if we're tracing forward or backwards
 */
	if ( behavior.hop_incr_unit == 1 )
		state.current_hop = behavior.min_ttl;
	else
		state.current_hop = behavior.max_ttl;
/*
 * MAIN LOOP STARTS HERE

 * loop through, building the ip
 * and sending it
 */
	behavior.report( TP_OUT_HEADER, NULL, 0 );


	/*
	 * conditional: ( A && B ) || C
	 */
	for ( state.packets_this_hop = 0;

		( state.current_hop >= behavior.min_ttl
		&&
		state.current_hop <= behavior.max_ttl )
		||
		state.packets_this_hop < behavior.packets_per_hop;

		++( state.packets_this_hop ) )
	{
		/*
		 * reset if we've sent enough probes for this ttl
		 */
		if ( state.packets_this_hop >= behavior.packets_per_hop )
		{
			state.current_hop += behavior.hop_incr_unit;
			state.packets_this_hop = 0;
		}

		/*
		 * skip the hops marked on the cmd line
		 */
		if ( behavior.do_skip == YES && behavior.skips[ state.current_hop ] == YES )
		{
			state.current_hop += behavior.hop_incr_unit;
			state.packets_this_hop = -1;
			continue;
		}

		state.packet_match = TP_PACKET_NO;
		state.low_ttl = NO;

/*
 * incriment the src and/or dst ports
 */
		packet.src_port += behavior.src_port_incr;
		packet.dst_port += behavior.dst_port_incr;

		build_packet();

/*
 * build the ip
 * (this should probably be rolled in build_packet() )
 */
		packet.ip_id = libnet_get_prand(LIBNET_PRu16);
		state.ip_h = libnet_build_ipv4(
			LIBNET_IPV4_H,			/* IP hdr len */
			0,				/* tos */
			packet.ip_id,			/* frag id */
			packet.frag_bit,		/* frag offset */
			state.current_hop,		/* ttl */
			packet.protocol_number,
			0,				/* checksum */
			packet.packed_src,		/* packed source */
			packet.packed_target,		/* packed target */
			NULL,				/* payload */
			0,				/* payload size */
			state.packet,
			state.ip_h );
/*
 * and change the filter
 */
		if ( do_filter() != 0 )
		{
			tixe ( tixe_cleanup, 1 );
		}

/*
 * spit out the hop number if this is the first probe this hop
 * then start timer and send the packet
 */
		if ( state.packets_this_hop == 0 )
			behavior.report( TP_OUT_HOP_NUMBER, ( struct in_addr * ) NULL, 0 );

		gettimeofday ( & state.start_time, NULL );

		if ( libnet_write( state.packet ) == -1 )
		{
			printf("error at send: %s\n",
				libnet_geterror( state.packet ) );
			tixe ( tixe_cleanup, 1 );
		}

/*
 * wait for a packet to get
 * through the filter, timeout
 * after the wait_timeout
 */
		while ( state.packet_match == TP_PACKET_NO )
		{
			state.packet_wait.tv_sec  = behavior.wait_timeout;
			state.packet_wait.tv_usec = 0;
			FD_ZERO( & ( state.wheel ) );
			FD_SET( state.fake_psocket->fd, & ( state.wheel ) );

			if ( select( ( state.fake_psocket->fd ) + 1,
					& ( state.wheel ),
					NULL,
					NULL,
					& state.packet_wait ) > 0 )
			{
				/*
				 * if we get a bite, note the time,
				 * test the response, and report it if OK
				 */
				gettimeofday ( & ( state.end_time ), NULL );

				state.capture_buf = pcap_next(
						state.psocket,
						& ( state.psock_hdr ) );

				state.trip_time = diff_time (
					& ( state.start_time ),
					& ( state.end_time ) );

				if ( parse_packet( state.capture_buf ) > 0 )
					account_packet ( state.trip_time );

				if ( packet.protocol_number == IPPROTO_TCP
					&& behavior.tcp_resets == YES
					&& state.packet_match == TP_PACKET_DONE )
				{
					if ( send_tcp_reset ( ) != 0 )
						printf("error at send: %s\n",
							libnet_geterror( state.packet ) );
				}

				/* blank the buffer to prevent confusion */
				memset ( (char *) state.capture_buf, '\0', sizeof ( state.capture_buf ) );
				state.trip_time = ( double ) 0.0;
			} else {
				/*
				 * if we get here,  no one wanted to talk to us for this probe
				 */
				behavior.report( TP_OUT_HOP_INFO, ( struct in_addr * ) NULL, TP_TYPE_NR );
				state.packet_match = TP_PACKET_TIMEOUT;
				account_packet ( 0 );
			}

			/*
			 * if there was an interupt during the wait
			 */
			if ( behavior.do_audit == YES || behavior.do_audit_exit == YES )
			{
				hop_audit();
			}
		} // end of while ( packet_match == TP_PACKET_NO ) loop

/*
 * an acceptable response in hand,
 * quit and clean up, or start over
 */
		/*
		 * in technical terms, this is "icky"
		 * it does however, seem to operate correctly
		 *
		 * conditional: ( ( A || B ) && C && D ) || ( E && F && G )
		 * I told you it was icky...
		 */
		if ( ( ( state.target_response == YES
					||
					state.current_hop == behavior.max_ttl )
				&&
				state.packets_this_hop >= behavior.packets_per_hop - 1
				&&
				behavior.hop_incr_unit == 1 )
			||
			( behavior.hop_incr_unit == -1
				&&
				state.current_hop == behavior.min_ttl
				&&
				state.packets_this_hop >= behavior.packets_per_hop - 1 ) )
		{
			/*
			 * If we get here, we're at the end of a run
			 * either with one or more responses from the target
			 * or because we've reached the ttl limit.
			 */
			state.current_hop = behavior.max_ttl + 1;

			/*
			 * if we're supposed to be continuous,
			 * reset the loop and start again
			 */
			if ( state.continuous_count != (int) NULL )
			{
				state.continuous_count--;
				if ( state.continuous_count == 0 )
					behavior.continuous = NO;
			}
			if ( behavior.continuous == YES )
			{
				state.packet_match = TP_PACKET_NO;
				state.target_response = NO;

				if ( behavior.hop_incr_unit == 1 )
					state.current_hop = behavior.min_ttl;
				else
					state.current_hop = behavior.max_ttl;

				/*
				 * setting packets_this_hop to -1 allows the
				 * reset if at the beginning of the loop to function
				 */
				state.packets_this_hop = -1;

				if ( behavior.continuous_accounting == YES )
					hop_audit();
			}
		}

		/*
		 * after a run pause for -W usecs
		 * (at least I think it usecs)
		 */
		if ( behavior.wait_between_packets != 0 )
			usleep ( behavior.wait_between_packets * 1000 );

/*
 * reset the ports as needed
 */
		if ( packet.src_port >= behavior.max_src_port && behavior.src_port_incr == 1 )
			packet.src_port = behavior.min_src_port - 1;
		else if ( packet.src_port <= behavior.min_src_port && behavior.src_port_incr == -1 )
			packet.src_port = behavior.max_src_port + 1;

		if ( packet.dst_port >= behavior.max_dst_port && behavior.dst_port_incr == 1)
			packet.dst_port = behavior.min_dst_port - 1;
		else if ( packet.dst_port <= behavior.min_dst_port && behavior.dst_port_incr == -1)
			packet.dst_port = behavior.max_dst_port + 1;

	} // end of the main send/recv loop

/*
 * yahoo, we're done
 * 
 * spit out the stats, sweep up after ourselves, and close up shop
 * last packet out please shut off the lights
 */
	hop_audit();

	tixe ( tixe_cleanup, 0 );	// cleanup and exit, status 0

	exit ( 0 ); // we should never get here
}

