/*
	**
	** psfe.c 
	**
	** Postscript display frontend to network traffic visualiser
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: psfe.c,v 1.4 1999/04/07 23:07:08 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <signal.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <glib.h>

#include "psfe.h"
#include "util.h"
#include "report.h"

static char			rcsid[] = "$Id: psfe.c,v 1.4 1999/04/07 23:07:08 dmiller Exp $";

#define WIDTH					595.0
#define HEIGHT					842.0

#define TITLE_X_CENTER		(WIDTH / 2.0)
#define TITLE_Y_CENTER		(HEIGHT - 120.0)

#define TIME_X_CENTER		(WIDTH / 2.0)
#define TIME_Y_CENTER		(TITLE_Y_CENTER - 27.0)

#define GRAPH_X_CENTER		(WIDTH / 2.0)
#define GRAPH_Y_CENTER		((HEIGHT - 100.0) / 2.0)
#define GRAPH_RADIUS			(WIDTH * 0.25)

#define LABEL_SEPERATION	0.8
#define LABEL_RADIUS			(GRAPH_RADIUS + LABEL_SEPERATION)

/* Used for drawing of peer lines */
typedef struct 
{
	host_t	*host;
	float		x_pos;
	float		y_pos;
	float		rotation_percent;
} visible_host_t;

/* Prototypes */
GHashTable *build_visible_hosts(float x_pos, float y_pos, float radius, host_t *h, unsigned long n_hosts);
void format_time(time_t t, char *buf, size_t len);
static void draw_peers(FILE *out, host_t *host, GHashTable *visible_hosts);
static void draw_hosts_polygon(FILE *out, float x_pos, float y_pos, float radius, unsigned long n_hosts);
static void label_hosts_polygon(FILE *out, float x_pos, float y_pos, float radius, const char *font, float size, host_t *h, unsigned long n_hosts);
static void print_header(FILE *out);
static void draw_string(FILE *out, const char *font, float size, float x_pos, float y_pos, const char *string);
static void format_hostname(host_t *host, char *s, size_t s_len);

/* Displays summary of network traffic */
void ps_display_summary(report_t *r, const char *output_file, 
								float base_line_width, float max_line_width, 
								int suppress_title, const char *title, 
								int suppress_times)
{
	host_t			*h;
	peer_t			*p;
	char				start_time[80];
	char				finish_time[80];
	char				time_message[256];
	GHashTable		*visible_hosts;
	u_int32_t		n_hosts;
	u_int64_t		max_sent;
	FILE				*out;

	if (output_file == NULL)
	{
		setlinebuf(stdout);
		out = stdout;
	} else
	{
		/* Open output file */
		out = fopen(output_file, "w");
		if (out == NULL)
		{
			fprintf(stderr, "Couldn't open file '%s' for writing.\n", output_file);
			return;
		}
	}

	/* Count how many hosts we have */
	n_hosts = 0;
	h = r->hosts;
	max_sent = 0;
	while(h != NULL)
	{
		p = h->peers;
		while (p != NULL)
		{
			if (p->bytes_sent > max_sent)
				max_sent = p->bytes_sent;
			
			p = p->next;
		}
		h = h->next;
		n_hosts++;
	}
	
	/* Print postscript headers */
	print_header(out);
	fprintf(out, "/MaxSent { %f } def\n", (float)max_sent);
	fprintf(out, "/MaxWidth { %f } def\n", max_line_width);
	fprintf(out, "/MinWidth { %f } def\n", base_line_width);

	if (suppress_title == 0)
	{
		if (title == NULL)
		{
		/* Draw page heading */
			fprintf(out, "%% Page heading\n");
			draw_string(out, "Helvetica-Bold", 30, TITLE_X_CENTER, TITLE_Y_CENTER, 
							"traffic-vis report");					
		} else
		{
			fprintf(out, "%% Page heading\n");
			draw_string(out, "Helvetica-Bold", 30, TITLE_X_CENTER, TITLE_Y_CENTER, 
							title);					
		}
		
		if (suppress_times == 0)
		{
			/* Draw start and finish times */
			fprintf(out, "%% Start and finish times\n");
			format_time(r->summary_start, start_time, sizeof(start_time));
			format_time(r->summary_finish, finish_time, sizeof(finish_time));
			g_snprintf(time_message, sizeof(time_message), "%s to %s", start_time, finish_time);
			draw_string(out, "Helvetica", 8, TIME_X_CENTER, TIME_Y_CENTER, 
							time_message);
		}
	}
	
	/* Label hosts polygon */
	label_hosts_polygon(out, GRAPH_X_CENTER, GRAPH_Y_CENTER, 
								LABEL_RADIUS, "Helvetica", 7.0, r->hosts, n_hosts);

	/* Build table of visible hosts (used in peer drawing */
	visible_hosts = build_visible_hosts(GRAPH_X_CENTER, GRAPH_Y_CENTER, 
													GRAPH_RADIUS, r->hosts, n_hosts);

	/* Draw peer relationships */
	h = r->hosts;
	while(h != NULL)
	{
		/* Draw peer ralationships */
		draw_peers(out, h, visible_hosts);
		
		/* Move on to next peer entry */
		h = h->next;
	}

	/* Draw hosts polygon */
	/* We do this last so it is drawn on top of peer lines */
	/* draw_hosts_polygon((WIDTH / 2.0), (HEIGHT / 2.0), RADIUS, n_hosts); */

	/* Terminate postscript page */
	fprintf(out, "showpage\n");
	fprintf(out, "%%%%EndPage\n");
}

static void draw_peers(FILE *out, host_t *host, GHashTable *visible_hosts)
{
	peer_t		*p;
	visible_host_t	*src;
	visible_host_t	*dst;
	
	fprintf(out, "%% Peer lines %s\n", host->hostname);

	/* Save graphics state */
	fprintf(out, "gsave\n");

	p = host->peers;
	while(p != NULL)
	{
		src = g_hash_table_lookup(visible_hosts, (gpointer)&(p->src->ip_addr));
		dst = g_hash_table_lookup(visible_hosts, (gpointer)&(p->dst->ip_addr));

		if ((src != NULL) && (dst != NULL))
		{
			fprintf(out, "%f slw\n", (float)p->bytes_sent);
			fprintf(out, "newpath\n");
			fprintf(out, "%f %f moveto\n", src->x_pos, src->y_pos);
			fprintf(out, "%f %f lineto\n", dst->x_pos, dst->y_pos);
			fprintf(out, "stroke\n");
			fprintf(out, "\n");
		}

		p = p->next;
	}

	/* Restore graphics state */
	fprintf(out, "grestore\n");
	fprintf(out, "\n");
}

static void draw_hosts_polygon(FILE *out, float x_pos, float y_pos, float radius, unsigned long n_hosts)
{
	unsigned long 	c;
	float				face_theta;
	
	fprintf(out, "%% Hosts polygon\n");

	/* Calculate rotation per side */
	face_theta = (2.0 * M_PI) / (float)n_hosts;

	/* Start drawing and move into position */
	fprintf(out, "0 setlinewidth\n");
	fprintf(out, "newpath\n");
	fprintf(out, "%f %f moveto\n", 
		x_pos + radius * sin(-0.5 * face_theta),
		y_pos + radius * cos(-0.5 * face_theta));
	
	/* Draw a line segment connecting each vertex */
	for(c = 0; c < n_hosts; c++)
	{
		fprintf(out, "%f %f lineto\n",
				  x_pos + radius * sin(((float)c + 0.5) * face_theta), 
				  y_pos + radius * cos(((float)c + 0.5) * face_theta));
	}
	
	/* Color in polygon edge */
	fprintf(out, "stroke\n");
	fprintf(out, "\n");
}

static void label_hosts_polygon(FILE *out, float x_pos, float y_pos, float radius, const char *font, float size, host_t *h, unsigned long n_hosts)
{
	unsigned long 	c;
	float				face_theta;
	
	fprintf(out, "%% Host polygon labels\n");

	/* Calculate rotation per side */
	face_theta = (2.0 * M_PI) / (float)n_hosts;

	/* Set up font */
	fprintf(out, "%f /%s GetFont\n", size, font);
	
	for(c = 0; c < n_hosts; c++)
	{
		char					hostname[80];

		format_hostname(h, hostname, sizeof(hostname));
		
		/* Set up drawing and move to position */
		fprintf(out, "newpath\n");
		fprintf(out, "gsave\n");
		fprintf(out, "%f %f moveto\n", 
			x_pos + radius * sin((float)c * face_theta), 
			y_pos + radius * cos((float)c * face_theta));

		/* Reverse direction for names on the left hand side */
		if (c > (n_hosts / 2))
			fprintf(out, "%f (%s) RightRotatedText\n", 270.0 - ((360.0 / (float)n_hosts) * (float)c), hostname);
		else
			fprintf(out, "%f (%s) LeftRotatedText\n", 90.0 - ((360.0 / (float)n_hosts) * (float)c), hostname);

		fprintf(out, "grestore\n");
		fprintf(out, "\n");
		
		h = h->next;
	}
}

GHashTable *build_visible_hosts(float x_pos, float y_pos, float radius, host_t *h, unsigned long n_hosts)
{
	unsigned long 	c;
	float				face_theta;
	GHashTable		*visible_hosts;
	visible_host_t	*vh;
	
	/* Calculate rotation per side */
	face_theta = (2.0 * M_PI) / (float)n_hosts;

	/* Allocate new hash table */
	visible_hosts = g_hash_table_new((GHashFunc)host_hash, (GCompareFunc)host_compare);
	
	for(c = 0; c < n_hosts; c++)
	{
		/* Create new table entry */
		vh = util_zmalloc(sizeof(*vh));
		vh->host = h;
		
		/* Calculate X position of host on edge of polygon */
		/* by interpolation between adjacent polygon vertices */
		vh->x_pos = radius * sin((float)(c - 0.5) * face_theta);
		vh->x_pos += radius * sin((float)(c + 0.5) * face_theta);
		vh->x_pos /= 2;
		vh->x_pos += x_pos;

		/* Calculate Y position of host on edge of polygon */
		vh->y_pos = radius * cos((float)(c - 0.5) * face_theta);
		vh->y_pos += radius * cos((float)(c + 0.5) * face_theta);
		vh->y_pos /= 2;
		vh->y_pos += y_pos;
		
		/* Calculate rotation of host (used to draw arrow heads */
		vh->rotation_percent = (100.0 / (float)n_hosts) * (float)c;

		/* Add info to hash table */
		g_hash_table_insert(visible_hosts, (gpointer)&(vh->host->ip_addr), (gpointer)vh);
		
		h = h->next;
	}

	return(visible_hosts);
}

static void print_header(FILE *out)
{
	fprintf(out, "\
%%!PS-Adobe-2.0\n\
%%%%Creator: traffic-vis 0.33\n\
%%%%Orientation: Portrait\n\
%%%%EndComments\n\
%%%%PageTrailer\n\
%%%%Page: 0 0\n\
\n\
%% Sets up font\n\
%% Takes font size, font name from stack\n\
/GetFont { findfont exch scalefont setfont } def\n\
\n\
%% Gets the extent of a string\n\
%% Takes string from stack, leaves four coordinates\n\
/GetExtents { gsave newpath 0 0 moveto false charpath flattenpath pathbbox grestore } def\n\
\n\
%% Centers around a string\n\
%% Takes string from stack\n\
/CenterText { dup GetExtents 3 2 roll sub 2 div neg 3 1 roll exch sub 2 div neg exch rmoveto show } def\n\
\n\
%% Left justified rotated text\n\
%% Takes string from stack\n\
/LeftRotatedText { dup 3 1 roll GetExtents 3 2 roll sub 2 div neg 3 1 roll pop pop 0 exch 3 2 roll rotate rmoveto show } def\n\
\n\
%% Right justified rotated text\n\
%% Takes string from stack\n\
/RightRotatedText { dup 3 1 roll GetExtents 3 2 roll sub 2 div neg 3 1 roll exch sub neg exch 3 2 roll rotate rmoveto show } def\n\
\n\
%% Line width used for peer lines\n\
/slw { MaxSent div MaxWidth MinWidth sub mul MinWidth add setlinewidth } def\n\
\n\
");
}

static void draw_string(FILE *out, const char *font, float size, float x_pos, float y_pos, const char *string)
{
	fprintf(out, "%f /%s GetFont\n", size, font);
	fprintf(out, "newpath\n");
	fprintf(out, "%f %f moveto\n", x_pos, y_pos);
	fprintf(out, "(%s) CenterText\n", string);
	fprintf(out, "\n");
}

static void format_hostname(host_t *host, char *s, size_t s_len)
{
	static struct	in_addr i;

	i.s_addr = host->ip_addr;
	
	if (host->hostname != NULL)
		g_snprintf(s, s_len, "%s", host->hostname);
	else	
		g_snprintf(s, s_len, "Unknown (%s)", inet_ntoa(i));
}

void format_time(time_t t, char *buf, size_t len)
{
	struct tm	*tm;
	
	tm = localtime(&t);
	
	strftime(buf, len, "%d-%b-%Y %H:%M:%S %Z", tm);
}

