/*
	**
	** traffic-sort.c
	**
	** Main module for report sort modules for traffic-vis suite
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: traffic-sort.c,v 1.5 1999/04/07 23:07:09 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <time.h>
#include <sys/types.h>

#include "util.h"
#include "report.h"
#include "read-report.h"
#include "write-report.h"
#include "sort-report.h"
#include "limit-report.h"

static char			rcsid[] = "$Id: traffic-sort.c,v 1.5 1999/04/07 23:07:09 dmiller Exp $";

#ifdef HAVE_GETOPT_LONG
/* Commandline options */
static struct option long_options[] =
{
	{"input", 1, NULL, 'i'},
	{"output", 1, NULL, 'o'},
	{"limit-hosts", 1, NULL, 'L'},
	{"limit-peers", 1, NULL, 'K'},
	{"host-sort", 1, NULL, 'H'},
	{"peer-sort", 1, NULL, 'P'},
	{"version", 0, NULL, 'V'},
	{"help", 0, NULL, 'h'},
	{NULL, 0, NULL, 0}
};
#endif /* HAVE_GETOPT_LONG */

/* Prototypes */
void usage(void);
void version(void);
int main(int argc, char **argv);

void usage(void)
{
	fprintf(stderr, "Usage: traffic-sort [OPTION]\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Sorts and/or limits the length of a network traffic report created by\n");
	fprintf(stderr, "traffic-collector. Limiting of output is done after sorting.\n");

	fprintf(stderr, "\n");

#ifdef HAVE_GETOPT_LONG
	fprintf(stderr, "  -i, --input  file      Read input from file instead of stdin.\n");
	fprintf(stderr, "  -o, --output file      Write output to file instead of stdout.\n");
	fprintf(stderr, "  -H, --host-sort   key  Sorts hosts (see below for sort keys).\n");
	fprintf(stderr, "  -P, --peer-sort   key  Sorts peers (see below for sort keys) \n");
	fprintf(stderr, "  -L, --limit-hosts n    Limit the number of hosts in output to 'n'\n");
	fprintf(stderr, "  -K, --limit-peers n    Limit the number of peers in output to 'n'\n");
	fprintf(stderr, "  -V, --version          Print program version.\n");
	fprintf(stderr, "  -h, --help             Display this help text.\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Host sort keys:\n");
	fprintf(stderr, "  N, none         Do not sort\n");
	fprintf(stderr, "  b, bytes        Total bytes (sent + received)\n");
	fprintf(stderr, "  s, bsent        Bytes sent\n");
	fprintf(stderr, "  r, breceived    Bytes received\n");
	fprintf(stderr, "  p, packets      Total packets (sent + received)\n");
	fprintf(stderr, "  S, psent        Packets sent\n");
	fprintf(stderr, "  R, preceived    Packets received\n");
	fprintf(stderr, "  C, connects     Total TCP connections (sent + received)\n");
	fprintf(stderr, "  c, csent        TCP connections sent\n");
	fprintf(stderr, "  l, creceived    TCP connections received\n");
	fprintf(stderr, "  P, peers        Number of peers\n");
	fprintf(stderr, "  h, hostname     Hostname\n");
	fprintf(stderr, "  I, ipaddr       IP address\n");
	fprintf(stderr, "  F, firstseen    When host was first noticed\n");
	fprintf(stderr, "  L, lastseen     When host was last noticed\n");
	fprintf(stderr, "  X, random       Random order\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Peer sort keys:\n");
	fprintf(stderr, "  N, none         Do not sort\n");
	fprintf(stderr, "  b, bytes        Total bytes (sent + received)\n");
	fprintf(stderr, "  p, packets      Total packets (sent + received)\n");
	fprintf(stderr, "  C, connects     Total TCP connections (sent + received)\n");
	fprintf(stderr, "  X, random       Random order\n");
	
#else /* HAVE_GETOPT_LONG */
	fprintf(stderr, "  -i file  Read input from file instead of stdin.\n");
	fprintf(stderr, "  -o file  Write output to file instead of stdout.\n");
	fprintf(stderr, "  -H key   Sorts hosts (see below for sort keys).\n");
	fprintf(stderr, "  -P key   Sorts peers (see below for sort keys) \n");
	fprintf(stderr, "  -L n     Limit the number of hosts in output to 'n'\n");
	fprintf(stderr, "  -K n     Limit the number of peers in output to 'n'\n");
	fprintf(stderr, "  -V       Print program version.\n");
	fprintf(stderr, "  -h       Display this help text.\n");

	fprintf(stderr, "Host sort keys:\n");
	fprintf(stderr, "  b   Total bytes (sent + received)\n");
	fprintf(stderr, "  s   Bytes sent\n");
	fprintf(stderr, "  r   Bytes received\n");
	fprintf(stderr, "  p   Total packets (sent + received)\n");
	fprintf(stderr, "  S   Packets sent\n");
	fprintf(stderr, "  R   Packets received\n");
	fprintf(stderr, "  C   Total TCP connections (sent + received)\n");
	fprintf(stderr, "  c   TCP connections sent\n");
	fprintf(stderr, "  l   TCP connections received\n");
	fprintf(stderr, "  P   Number of peers\n");
	fprintf(stderr, "  h   Hostname\n");
	fprintf(stderr, "  I   IP address\n");
	fprintf(stderr, "  F   When host was first noticed\n");
	fprintf(stderr, "  L   When host was last noticed\n");
	fprintf(stderr, "  X   Random order\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Peer sort keys:\n");
	fprintf(stderr, "  b  Total bytes (sent + received)\n");
	fprintf(stderr, "  p  Total packets (sent + received)\n");
	fprintf(stderr, "  C  Total TCP connections (sent + received)\n");
	fprintf(stderr, "  X  Random order\n");
#endif /* HAVE_GETOPT_LONG */

	fprintf(stderr, "\n");

	fprintf(stderr, "A minus '-' sign before a sort key reverses the sort order.\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Please report bugs to dmiller@ilogic.com.au\n");
}

void version(void)
{
	fprintf(stderr, "traffic-sort %s\n", VERSION);
	
	/* Avoid stupid compiler warning */
	(void)rcsid;
}

int main(int argc, char **argv)
{
	report_t			*r;

	int 				c;
	extern char 	*optarg;
	
	/* Commandline options */
	static char		*output_file = NULL;
	static char		*input_file = NULL;
	static int		host_sort_mode = SORT_NONE;
	static int		peer_sort_mode = SORT_NONE;
	static int		host_reverse = 0;
	static int		peer_reverse = 0;
	static int		max_hosts = 0;
	static int		max_peers = 0;
	
	if (argc == 1)
	{
		usage();
		exit(0);
	}

	/* Fetch commandline options */
	while (1)
	{
#ifdef HAVE_GETOPT_LONG
		c = getopt_long (argc, argv, "i:o:H:P:L:K:hV?", long_options, NULL);
#else /* HAVE_GETOPT_LONG */
		c = getopt (argc, argv, "i:o:H:P:L:K:hV?");
#endif /* HAVE_GETOPT_LONG */
		if (c == -1)
			break;
		
		switch (c)
		{
			case 'i':
				input_file = util_strdup(optarg);
				break;
				
			case 'o':
				output_file = util_strdup(optarg);
				break;
				
			case 'H':
				if (optarg[0] == '-')
				{
					host_reverse = 1;
					host_sort_mode = parse_host_sort_mode(optarg + 1);
				} else
				{
					host_sort_mode = parse_host_sort_mode(optarg);
				}
				
				if (host_sort_mode == -1)
				{
					fprintf(stderr, "Invalid host sort mode.\n\n");
					usage();
					exit(-1);
				}
				break;
			
			case 'P':
				if (optarg[0] == '-')
				{
					peer_reverse = 1;
					peer_sort_mode = parse_peer_sort_mode(optarg + 1);
				} else
				{
					peer_sort_mode = parse_peer_sort_mode(optarg);
				}
				
				if (peer_sort_mode == -1)
				{
					fprintf(stderr, "Invalid peer sort mode.\n\n");
					usage();
					exit(-1);
				}
				break;

			case 'L':
				max_hosts = atoi(optarg);
				break;
			
			case 'K':
				max_peers = atoi(optarg);
				break;
			
			case 'V':
				version();
				exit(0);
				
			case 'h':
			case '?':
				usage();
				exit(0);
				
			default:
				fprintf(stderr, "Invalid commandline options.\n\n");
				usage();
				exit(-1);
		}
	}

	util_seed_rng();
	
	r = parse_report(input_file);
	
	if (host_sort_mode != SORT_NONE)
		r->hosts = sort_hosts(r->hosts, host_sort_mode, host_reverse);

	if (peer_sort_mode != SORT_NONE)
		sort_all_peers(r->hosts, peer_sort_mode, peer_reverse);

	if (max_hosts != 0)
	{
		limit_hosts_by_number(&(r->hosts), max_hosts);
		trim_peers(r->hosts);
	}
	
	if (max_peers != 0)
		limit_peers_by_number(r->hosts, max_peers);
	
	write_report(r, output_file);

	return(0);
}
