//@HEADER
// ************************************************************************
// 
//            NOX: An Object-Oriented Nonlinear Solver Package
//                 Copyright (2002) Sandia Corporation
// 
//            LOCA: Library of Continuation Algorithms Package
//                 Copyright (2005) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
// 
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or 
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source$
//  $Author$
//  $Date$
//  $Revision$
// ************************************************************************
//@HEADER
                                                                                
// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation 
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u 
 * --- - k * u**2 = 0
 * dx2
 *
 * subject to @ x=0, u=1
 */

#ifndef _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H
#define _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H

#include <fstream>

// Forward Declarations
class Epetra_Comm;
class Epetra_Map; 
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell the evaluate routine what objects to fill
enum FillType {F_ONLY, MATRIX_ONLY, ALL}; 

// Finite Element Problem Class
class FiniteElementProblem { 

public:

  // Constructor
  FiniteElementProblem(int NumGlobalElements, Epetra_Comm& Comm, 
		       double s = 1.0, std::ofstream* file = NULL);
  
  // Destructor
  ~FiniteElementProblem();

  // Evaluates the function (F) and/or the Jacobian using the solution 
  // values in solnVector.
  bool evaluate(FillType f, const Epetra_Vector *solnVector, 
		Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix,
		double jac_coeff = 1.0, 
		double mass_coeff = 0.0);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the FiniteElementProblem class.
  Epetra_Vector& getSolution();
  
  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the FiniteElementProblem class.
  Epetra_CrsMatrix& getJacobian();

  // Set a bifurcation parameter in the application physics
  bool set(string label, double value);

  //! Print solution to output file
  void printSolution(const Epetra_Vector& x_, double conParam);

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

  // Private to prohibit copying
  FiniteElementProblem(const FiniteElementProblem&);

  // Private to prohibit copying
  FiniteElementProblem& operator=(const FiniteElementProblem&);

private:

  FillType flag;
  Epetra_Map *StandardMap; 
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Epetra_Vector *initialSolution;
  Epetra_Vector *rhs;
  Epetra_CrsGraph *AA;
  Epetra_CrsMatrix *A;
  Epetra_Comm *Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes
  int NumMyElements;      // Number of elements owned by this process
  int NumGlobalElements;  // Total Number of elements

 public:

  double factor;          // factor used on nonlinear term
  double leftBC;          // value of the essential BC at x = 0.
  double rightBC;         // value of the essential BC at x = 1.
  double scale;           // scale factor to test parameter scaling

private:

  //! Stores pointer to output file stream
  std::ofstream* outputFile;

  //! Epetra_Map for printing solution to output file
  Epetra_Map* outputMap;

  //! Epetra_Vector for printint solution to output file
  Epetra_Vector* outputVector;

  //! Importer for printing solution to ouput file
  Epetra_Import* outputImporter;

};
#endif




