// @HEADER
// ************************************************************************
//
//               Rapid Optimization Library (ROL) Package
//                 Copyright (2014) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact lead developers:
//              Drew Kouri   (dpkouri@sandia.gov) and
//              Denis Ridzal (dridzal@sandia.gov)
//
// ************************************************************************
// @HEADER

#ifndef ROL_EXPERIMENTDESIGN_OBJECTIVE_H
#define ROL_EXPERIMENTDESIGN_OBJECTIVE_H

#include "ROL_Reduced_Objective_SimOpt.hpp"
#include "ROL_Vector.hpp"

namespace ROL {

template <class Real>
class ExperimentDesignObjective;

/*
  Reduced Hessian for the OED problem.
  This is a utility class, called by ExperimentDesignObjective.
*/
template <class Real>
class ReducedHessian : public LinearOperator<Real> {
private:
  const ExperimentDesignObjective<Real> edo_;
  Teuchos::RCP<const Vector<Real> > w_;

public:

  ReducedHessian(const ExperimentDesignObjective<Real> & edo, Teuchos::RCP<const Vector<Real> > w) : edo_(edo), w_(w) {}

  void apply( Vector<Real> &hv, const Vector<Real> &v, Real &tol ) const {
    Real mytol(1e-8);
    Teuchos::RCP<Vector<Real> > Bv = edo_.getConstraintVec()->clone();
    Teuchos::RCP<Vector<Real> > AiBv = edo_.getStateVec()->clone();
    Teuchos::RCP<Vector<Real> > QAiBv = edo_.getObservationVec()->clone();
    Teuchos::RCP<Vector<Real> > WQAiBv = (edo_.getObservationVec()->dual()).clone();
    Teuchos::RCP<Vector<Real> > QtWQAiBv = (edo_.getStateVec()->dual()).clone();
    Teuchos::RCP<Vector<Real> > AitQtWQAiBv = (edo_.getConstraintVec()->dual()).clone();

    Teuchos::RCP<Vector<Real> > BtAitQtWQAiBv = (edo_.getControlVec()->dual()).clone();

    edo_.getConstraint()->applyJacobian_2(*Bv, v, *(edo_.getStateVec()), *(edo_.getControlVec()), mytol);
    edo_.getConstraint()->applyInverseJacobian_1(*AiBv, *Bv, *(edo_.getStateVec()), *(edo_.getControlVec()), mytol);
    edo_.applyObserveOp(*QAiBv, *AiBv);
    edo_.applyWeightOp(*WQAiBv, *QAiBv, *w_);
    edo_.applyAdjointObserveOp(*QtWQAiBv, *WQAiBv);
    edo_.getConstraint()->applyInverseAdjointJacobian_1(*AitQtWQAiBv, *QtWQAiBv, *(edo_.getStateVec()), *(edo_.getControlVec()), mytol);
    edo_.getConstraint()->applyAdjointJacobian_2(*BtAitQtWQAiBv, *AitQtWQAiBv, *(edo_.getStateVec()), *(edo_.getControlVec()), mytol);

    edo_.getObjective()->hessVec_22(hv, v, *(edo_.getStateVec()), *(edo_.getControlVec()), mytol);
    hv.plus(*BtAitQtWQAiBv);
   
  }

};


/** @ingroup func_group
    \class ROL::ExperimentDesignObjective
    \brief Provides the interface to evaluate objective functions
           used for optimal experimental design (OED).

    ROL's objective function used for optimal experimental design
    relies on the SimOpt functional interface and the Elementwise
    vector interface.  It adds four virtual functions:

    \li #applyObserveOp -- apply observation operator;
    \li #applyAdjointObserveOp -- apply adjoint of the observation operator; and
    \li #applyWeightOp -- apply weight operator.

    ---
*/
template <class Real>
class ExperimentDesignObjective : public Objective<Real> {
private:
  Teuchos::RCP<Objective_SimOpt<Real> > obj_;            // objective function used for the conventional inverse problem 
  Teuchos::RCP<EqualityConstraint_SimOpt<Real> > con_;   // constraint function used for the conventional inverse problems
  Teuchos::RCP<Vector<Real> > state_;                    // state vector, used for cloning
  Teuchos::RCP<Vector<Real> > adjoint_;                  // adjoint vector, used for cloning
  Teuchos::RCP<Vector<Real> > control_;                  // control vector, used for cloning
  Teuchos::RCP<Vector<Real> > constraint_;               // constraint vector, used for cloning
  Teuchos::RCP<Vector<Real> > observation_;              // observation vector, used for cloning
  std::vector<Teuchos::RCP<Vector<Real> > > training_;   // training-set vectors used in OED
  Teuchos::RCP<Vector<Real> > rand01_;                   // a vector of 0 and 1 entries occurring with probability 1/2

  Real cgabstol_;   // CG absolute tolerance to solve reduced-Hessian subproblems
  Real cgreltol_;   // CG relative tolerance to solve reduced-Hessian subproblems
  int  cgmaxiter_;  // max number of CG iterations to solve reduced-Hessian subproblems

  Real sigma_;      // standard deviation of the noise in data
  Real beta_;       // sparsity regularization factor

public:
  ExperimentDesignObjective(const Teuchos::RCP<Objective_SimOpt<Real> > &obj,
                            const Teuchos::RCP<EqualityConstraint_SimOpt<Real> > &con,
                            const Teuchos::RCP<Vector<Real> > &state,
                            const Teuchos::RCP<Vector<Real> > &adjoint,
                            const Teuchos::RCP<Vector<Real> > &control,
                            const Teuchos::RCP<Vector<Real> > &constraint,
                            const Teuchos::RCP<Vector<Real> > &observation,
                            const std::vector<Teuchos::RCP<Vector<Real> > > &training,
                            const Teuchos::RCP<Vector<Real> > &rand01,
                            const Teuchos::RCP<Teuchos::ParameterList> &parlist) :
    obj_(obj), con_(con), state_(state), adjoint_(adjoint),
    control_(control), constraint_(constraint), observation_(observation),
    training_(training), rand01_(rand01) {
    // get problem parameters
    cgabstol_  = parlist->sublist("Problem").get("OED CG Absolute Tolerance", 1e10);
    cgreltol_  = parlist->sublist("Problem").get("OED CG Relative Tolerance", 1e-4);
    cgmaxiter_ = parlist->sublist("Problem").get("OED CG Iteration Limit", 1000);
    sigma_     = parlist->sublist("Problem").get("OED Noise Standard Deviation", 1e-2);
    beta_      = parlist->sublist("Problem").get("OED Sparsity Regularization", 1e-2);
  }

  Real value( const Vector<Real> &x, Real &tol ) {
    Teuchos::RCP<Vector<Real> > Mtrain = (control_->dual()).clone();
    Teuchos::RCP<Vector<Real> > CinvMtrain = control_->clone();
    Teuchos::RCP<Vector<Real> > Vx = observation_->dual().clone();
    Teuchos::RCP<Vector<Real> > QtVx = state_->dual().clone();
    Teuchos::RCP<Vector<Real> > AitQtVx = constraint_->dual().clone();
    Teuchos::RCP<Vector<Real> > BtAitQtVx = control_->dual().clone();
    Teuchos::RCP<Vector<Real> > CinvBtAitQtVx = control_->clone();

    Real mytol(1e-8);
    // Initialize sum of bias, variance and sparse regularization.
    Real sumBVR(0);
    // Norm computation for the bias term.
    // Comment out, for now.
    /*
    int numTraining = static_cast<int>(training_.size());
    for (int i=0; i<numTraining; ++i) {
      obj_->hessVec_22(*Mtrain, *(training_[i]), *state_, *control_, tol);
      applyInverseReducedHessian(*CinvMtrain, *Mtrain, x);
      sumBVR += CinvMtrain->dot(*CinvMtrain);
    }
    */
    // Trace estimation for the variance term.
    Vx->set(*rand01_);
    Vx->applyBinary(Elementwise::Multiply<Real>(), x);
    applyAdjointObserveOp(*QtVx, *Vx);
    con_->applyInverseAdjointJacobian_1(*AitQtVx, *QtVx, *state_, *control_, mytol);
    con_->applyAdjointJacobian_2(*BtAitQtVx, *AitQtVx, *state_, *control_, mytol);
    applyInverseReducedHessian(*CinvBtAitQtVx, *BtAitQtVx, x);
    sumBVR += (sigma_*sigma_)*CinvBtAitQtVx->dot(*CinvBtAitQtVx);
    // Sparse regularization term.
    sumBVR += beta_*x.reduce(ROL::Elementwise::ReductionSum<Real>());
    return sumBVR;
  }

  void gradient( Vector<Real> &g, const Vector<Real> &x, Real &tol ) {
    g.scale(0.0);
  }

  void hessVec( Vector<Real> &hv, const Vector<Real> &v, const Vector<Real> &x, Real &tol ) {
    hv.zero();
  }

  virtual void applyObserveOp(Vector<Real> &obsv, const Vector<Real> &v) const {
    obsv.set(v);
  }

  virtual void applyAdjointObserveOp(Vector<Real> &aobsv, const Vector<Real> &v) const {
    aobsv.set(v);
  }

  virtual void applyWeightOp(Vector<Real> &weightv, const Vector<Real> &v, const Vector<Real> &w) const {
    weightv.set(v.dual());
    weightv.applyBinary(Elementwise::Multiply<Real>(), w);
  }

  // Access functions.
  Teuchos::RCP<Vector<Real> > getStateVec() const { return state_; }
  Teuchos::RCP<Vector<Real> > getAdjointVec() const { return adjoint_; }
  Teuchos::RCP<Vector<Real> > getControlVec() const { return control_; }
  Teuchos::RCP<Vector<Real> > getObservationVec() const { return observation_; }
  Teuchos::RCP<Vector<Real> > getConstraintVec() const { return constraint_; }
  Teuchos::RCP<Objective_SimOpt<Real> > getObjective() const { return obj_; }
  Teuchos::RCP<EqualityConstraint_SimOpt<Real> > getConstraint() const { return con_; }

private:

  void applyInverseReducedHessian(Vector<Real> &ihv, const Vector<Real> &v, const Vector<Real> &w) const {
    int iter(0);
    int flag(0);
    ConjugateGradients<Real> cg(cgabstol_, cgreltol_, cgmaxiter_, false);
    ReducedHessian<Real> reducedHessian(*this, Teuchos::rcpFromRef(w));
    cg.run(ihv, reducedHessian, v, reducedHessian, iter, flag);
std::cout << "iter = " << iter << std::endl;
std::cout << "flag = " << flag << std::endl;
  }

}; // class ExperimentDesignObjective

} // namespace ROL

#endif
