// vtape.c - Virtual Tape Support Routines
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "emu/defs.h"
#include "emu/vtape.h"

// ********************************************************************

// TPS Format - Tape Data, SIMH Format

int tps_Read(VMT_TAPE *vmt, uint8 *data, int32 blksz)
{
	uint32 blksz32, tblksz32, wc;
	int    rc, pos;

	vmt->errCode = 0;

	if (blksz < 0) {
		if (vmt->posFlag == MT_BOT)
			return vmt->posFlag;
		pos = vmt->mtAddr - sizeof(blksz32);
		lseek(vmt->dpFile, pos, SEEK_SET);
		rc = read(vmt->dpFile, &tblksz32, sizeof(tblksz32));
		if ((rc == sizeof(tblksz32)) && (tblksz32 > 0)) {
			wc = (tblksz32 + 1) & ~1; // Word-aligned length
			pos -= wc + sizeof(tblksz32);
			lseek(vmt->dpFile, pos, SEEK_SET);
			rc = read(vmt->dpFile, &blksz32, sizeof(blksz32));
			if ((rc == sizeof(blksz32)) && (blksz32 == tblksz32)) {
				if ((rc = read(vmt->dpFile, data, wc)) == wc) {
					// Record is valid now.
					vmt->mtAddr  = pos;
					vmt->posFlag = MT_CUR;
					return blksz32;
				}
			}
		} else if (rc == sizeof(tblksz32)) {
			vmt->mtAddr  = pos;
			vmt->posFlag = (pos > 0) ? MT_MARK : MT_BOT;
		}
		if (rc < 0)
			vmt->errCode = errno;
	} else if ((rc = lseek(vmt->dpFile, vmt->mtAddr, SEEK_SET)) >= 0) {
		rc = read(vmt->dpFile, &blksz32, sizeof(blksz32));
		if ((rc == sizeof(blksz32)) && (blksz32 > 0)) {
#ifdef DEBUG
			if (blksz < blksz32)
				dbg_Printf("%s: Read(%d): Too large record: %d bytes (limit %d bytes)\n",
					vmt->Format->Name, vmt->mtAddr, blksz32, blksz);
#endif /* DEBUG */
			wc = (blksz32 + 1) & ~1; // Word-aligned length
			if ((rc = read(vmt->dpFile, data, wc)) == wc) {
				rc = read(vmt->dpFile, &tblksz32, sizeof(tblksz32));
				if ((rc == sizeof(tblksz32)) && (blksz32 == tblksz32)) {
#ifdef DEBUG
					dbg_Printf("%s: Read(%d): Read %d bytes of data.\n",
						vmt->Format->Name, vmt->mtAddr, blksz32);
#endif /* DEBUG */
					// Record is valid now.
					vmt->mtAddr  += wc + sizeof(blksz32) + sizeof(tblksz32);
					vmt->posFlag  = MT_CUR;
					return blksz32;
				}
#ifdef DEBUG
				dbg_Printf("%s: Read(%d): Mismatch block sizes: %d != %d\n",
					vmt->Format->Name, vmt->mtAddr, blksz32, tblksz32);
#endif /* DEBUG */
			}
		} else if (rc <= 0) {
			if (rc < 0) {
				vmt->errCode = errno;
#ifdef DEBUG
				dbg_Printf("%s: Read(%d): %s\n",
					vmt->Format->Name, vmt->mtAddr, strerror(errno));
#endif /* DEBUG */
			} else {
#ifdef DEBUG
				dbg_Printf("%s: Read(%d): ** End of Tape **\n",
					vmt->Format->Name, vmt->mtAddr);
#endif /* DEBUG */
				vmt->posFlag = MT_EOT;
			}
		} else if (blksz32 == 0) {
#ifdef DEBUG
			dbg_Printf("%s: Read(%d): ** Tape Mark **\n",
				vmt->Format->Name, vmt->mtAddr);
#endif /* DEBUG */
			vmt->mtAddr += sizeof(blksz32);
			vmt->posFlag = MT_MARK;
		}
	} else {
		vmt->errCode = errno;
#ifdef DEBUG
		dbg_Printf("%s: Seek(%d): %s\n",
			vmt->Format->Name, vmt->mtAddr, strerror(errno));
#endif /* DEBUG */
	}

	// Return with I/O error or tape mark encounters.
	return vmt->errCode ? MT_ERROR : vmt->posFlag;
}

int tps_Write(VMT_TAPE *vmt, uint8 *data, int32 blksz)
{
	int wc, idx, rc;

	vmt->errCode = 0;

	if (vmt->Flags & VMT_WRLOCK) {
		vmt->errCode = EACCES;
		return MT_ERROR;
	}

	if (blksz < 0) {
#ifdef DEBUG
		dbg_Printf("%s: Reverse Not Supported.\n",
			vmt->Format->Name);
#endif /* DEBUG */
		vmt->errCode = -1;
	} else if ((rc = lseek(vmt->dpFile, vmt->mtAddr, SEEK_SET)) >= 0) {
		if ((rc = write(vmt->dpFile, &blksz, sizeof(blksz))) == sizeof(blksz)) {
			wc = (blksz + 1) & ~1; // Word-aligned length
			if (blksz < wc) data[blksz] = '\0';
			if ((rc = write(vmt->dpFile, data, wc)) == wc) {
				rc = write(vmt->dpFile, &blksz, sizeof(blksz));
				if (rc == sizeof(blksz)) {
					// Record is valid now.
					vmt->mtAddr  += wc + (sizeof(blksz) * 2);
					vmt->posFlag  = MT_CUR;
					vmt->errCode  = 0;
					return blksz;
				}
			}
		}
	} else {
		vmt->errCode = errno;
#ifdef DEBUG
		dbg_Printf("%s: (W) Seek(%s): %s\n",
			vmt->Format->Name, vmt->mtAddr, strerror(errno));
#endif /* DEBUG */
	}

	// if no space left on device means the end of tape.
	if (vmt->errCode == ENOSPC) {
		vmt->errCode = 0;
		vmt->posFlag = MT_EOT;
		return MT_EOT;
	}

	// Return with I/O error.
	return MT_ERROR;
}

int tps_Mark(VMT_TAPE *vmt)
{
	uint32 zero = 0; // Block length
	int    rc;

	if (vmt->Flags & VMT_WRLOCK) {
		vmt->errCode = EACCES;
		return MT_ERROR;
	}

	if ((rc = write(vmt->dpFile, &zero, sizeof(zero))) < sizeof(zero)) {
		if (rc < 0) {
			vmt->errCode = errno;
#ifdef DEBUG
			dbg_Printf("%s: Mark(%d): %s\n",
				vmt->Format->Name, vmt->mtAddr, strerror(errno));
#endif /* DEBUG */
		}
	} else {
		vmt->mtAddr += sizeof(zero);
		return MT_OK;
	}

	return MT_ERROR;
}

int tps_Skip(VMT_TAPE *vmt, int32 dir)
{
	uint32 blksz;
	int count = 0;
	int pos, rc;

	vmt->posFlag = MT_CUR;
	if (dir < 0) {
		if (vmt->posFlag == MT_BOT)
			return vmt->posFlag;
		pos = vmt->mtAddr - sizeof(blksz);
		lseek(vmt->dpFile, pos, SEEK_SET);
		read(vmt->dpFile, &blksz, sizeof(blksz));
#ifdef DEBUG
		dbg_Printf("%s: (Backward) Skipping %d (%08X at %08X) bytes of record.\n",
			vmt->Format->Name, blksz, blksz, vmt->mtAddr);
#endif /* DEBUG */
		if (blksz == 0) {
			vmt->mtAddr  = pos;
			vmt->posFlag = MT_MARK;
		} else
			vmt->mtAddr -= (blksz + (sizeof(blksz) * 2));
	} else {
		lseek(vmt->dpFile, vmt->mtAddr, SEEK_SET);
		rc = read(vmt->dpFile, &blksz, sizeof(blksz));
		if (rc == sizeof(blksz)) {
#ifdef DEBUG
			dbg_Printf("%s: (Forward) Skipping %d (%04X at %08X) bytes of record.\n",
				vmt->Format->Name, blksz, blksz, vmt->mtAddr);
#endif /* DEBUG */
			if (blksz == 0) {
				vmt->mtAddr += sizeof(blksz);
				vmt->posFlag = MT_MARK;
			} else
				vmt->mtAddr += (blksz + (sizeof(blksz) * 2));
		} else if (rc == 0)
			vmt->posFlag = MT_EOT;
	}

	return vmt->posFlag;
}

int tps_Rewind(VMT_TAPE *vmt)
{
	uint32 pos;

	if (vmt == NULL)
		return VMT_NODESC;

	if ((pos = lseek(vmt->dpFile, 0, SEEK_SET)) < 0) {
		vmt->errCode = errno;
		return VMT_IOERROR;
	}

	vmt->mtAddr  = pos;
	vmt->posFlag = MT_BOT;

	return VMT_OK;
}

VMT_FORMAT vmt_Formats[] =
{
//	{ "tpc", "Tape data with Counts",   VMT_FMT_TPC },
	{ "tps", "Tape data, SIMH Format",  VMT_FMT_TPS,
     tps_Read, tps_Write, tps_Mark, tps_Skip, tps_Rewind	},
	{ NULL }  // Null Terminator
};

// ********************************************************************

VMT_FORMAT *vmt_GetFormat(char *fmtName)
{
	int idx;

	// Looking for disk format by name.
	for (idx = 0; vmt_Formats[idx].Name; idx++)
		if (!strcasecmp(fmtName, vmt_Formats[idx].Name))
			return &vmt_Formats[idx];

	// Format not found - return nothing.
	return NULL;
}

int vmt_OpenTape(VMT_TAPE *vmt)
{
	VMT_FORMAT *fmt;
	int        umode;

	// Check any errors first.
	if (vmt == NULL)
		return VMT_NODESC;
	if (vmt->fileName == NULL)
		return VMT_NONAME;
	if (vmt->fmtName == NULL) {
		char *p = strrchr(vmt->fileName, '.');
		if (p == NULL)
			return VMT_NOFMT;
		vmt->fmtName = p + 1;
	}
	if ((fmt = vmt_GetFormat(vmt->fmtName)) == NULL)
		return VMT_NOFMT;
	vmt->Format = fmt;

	// Attempt to open a disk file.
	umode = (vmt->Flags & VMT_WRLOCK) ? O_RDONLY : O_RDWR|O_CREAT;
	if ((vmt->dpFile = open(vmt->fileName, umode, 0700)) < 0) {
		vmt->errCode = errno;
		return VMT_OPENERR;
	}
	vmt->Flags |= VMT_OPENED;

	return VMT_OK;
}

int vmt_CloseTape(VMT_TAPE *vmt)
{
	if (vmt == NULL)
		return VMT_NODESC;

	close(vmt->dpFile);
	vmt->Flags  &= ~VMT_OPENED;
	vmt->dpFile  = 0;

	return VMT_OK;
}
