#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qpopupmenu.h>
#include <qtimer.h>
#include <qtextview.h>
#include <qimage.h>

#include <tulip/SuperGraph.h>
#include <tulip/Iterator.h>
#include <tulip/SelectionProxy.h>
#include "tulip/LayoutProxy.h"
#include <tulip/GlGraph.h>

#include "tulip/MouseInterface.h"

#include "tulip/GlGraphWidget.h"

using namespace std;
using namespace Tulip;

GlGraphWidget::GlGraphWidget(QWidget *parent, const char *name):
  QGLWidget(parent, name), GlGraphStrategy(),
  output(NULL),
  drawTimer(new QTimer(this)),
  mouse(NULL)
{
  TRACE_EXEC();
  connect(drawTimer, SIGNAL(timeout()), this, SLOT(drawGraph()));
  glGraph = new GlGraph(*this);
  setFocusPolicy(QWidget::StrongFocus);
}

GlGraphWidget::GlGraphWidget(const GlGraphWidget &w, QWidget *parent, const char *name):
  QGLWidget(parent, name, &w),
  GlGraphStrategy(),
  output(NULL),
  drawTimer(new QTimer(this)),
  mouse(NULL)
{
  TRACE_EXEC();
  connect(drawTimer, SIGNAL(timeout()), this, SLOT(drawGraph()));
  glGraph = new GlGraph(*this, *w.getGlGraph());
  setFocusPolicy(QWidget::StrongFocus);
}

GlGraphWidget::~GlGraphWidget() {
  delete drawTimer;
  if (glGraph != NULL) delete glGraph;
}

void GlGraphWidget::setGlGraph(GlGraph *g) {
  TRACE_EXEC();
  setUpdatesEnabled(false);
  if (glGraph != NULL) delete glGraph;
  glGraph = g;
  setUpdatesEnabled(true);
  redraw();
}

GlGraph *GlGraphWidget::getGlGraph() const {
  return glGraph;
}

void GlGraphWidget::setQTextView(QTextView *qt) {
  output = qt;
}
//==================================================
//GlGraphStrategy
//==================================================
void GlGraphWidget::MakeCurrent() {
  TRACE_EXEC();
  makeCurrent();
}

void GlGraphWidget::UpdateGL() {
  TRACE_EXEC();
  updateGL();
}

void GlGraphWidget::redraw() {
  TRACE_EXEC();
  assert (glGraph!=NULL);
  glGraph->redraw();
}

void GlGraphWidget::setDoubleBuffering(bool b) {
  TRACE_EXEC();
  QGLFormat tmpFormat;
  tmpFormat=format();
  //  tmpFormat.setDirectRendering(b);
  tmpFormat.setDoubleBuffer(b);
  setFormat(tmpFormat);
  setAutoBufferSwap(b);
}

bool GlGraphWidget::timerIsActive() {
  return drawTimer->isActive();
}

int GlGraphWidget::timerStart(int msec, bool sshot) {
  return drawTimer->start(msec, sshot);
}

void GlGraphWidget::timerStop() {
  drawTimer->stop();
}

void GlGraphWidget::mPaint(GlGraph *g) {
  if (mouse != NULL) mouse->mPaint(this);
}

void GlGraphWidget::outputSetText(const string &s) {
  if (output != NULL) {
    QString qs = QString(s.c_str());
    output->setText(qs);
  }
}

const string GlGraphWidget::outputGetText() {
  return string((output==NULL) ? string("") : string(output->text().utf8()));
}

void GlGraphWidget::update(set<Observable *>::iterator itb) {
  TRACE_EXEC();
}

void GlGraphWidget::emitNodeClicked(const node &n) {emit nodeClicked(n);}
void GlGraphWidget::emitEdgeClicked(const edge &e) {emit edgeClicked(e);}

//==================================================
void GlGraphWidget::setSuperGraph(SuperGraph *sg) {
  TRACE_EXEC();
  if (glGraph == NULL) return;
  setUpdatesEnabled(false);
  glGraph->setSuperGraph(sg);
  glGraph->init();
  toUpdate=true;
  setUpdatesEnabled(true);
  redraw();
}

SuperGraph *GlGraphWidget::getSuperGraph() const {
  if (glGraph != NULL) 
    return glGraph->getSuperGraph();
  else 
    return NULL;
}

MouseInterface *GlGraphWidget::getMouse() const {return mouse;}
void GlGraphWidget::setMouse(MouseInterface *m) {mouse = m;}

//==================================================
//QGLWidget
//==================================================
QImage GlGraphWidget::grabFrameBuffer(bool withAlpha) {
  bool incState = glGraph->isIncrementalRendering();
  glGraph->setIncrementalRendering(false);
  this->updateGL();
  QImage img = QGLWidget::grabFrameBuffer(withAlpha);
  glGraph->setIncrementalRendering(incState);
  return img;
}

//==================================================
//QGLWidget slots
//==================================================
void GlGraphWidget::initializeGL() {
  TRACE_EXEC();

  makeCurrent();
  QGLFormat tmpFormat;
  tmpFormat=format();
  tmpFormat.setDirectRendering(true);
  tmpFormat.setDoubleBuffer(true);
  setFormat(tmpFormat);
  setAutoBufferSwap(true);

  if (glGraph != NULL) glGraph->initializeGL();
}

void GlGraphWidget::paintGL() {
  TRACE_EXEC();
  assert(glGraph != NULL);

  makeCurrent();
  //hack for qt with directed rendering(DRI) activated, without, edges are not displayed. 
  //Mysterious bug.
  {
    int winX, winY, winW, winH;
    GLint *vp = new GLint[4];
    glGraph->getWinParameters(&winX, &winY, &winW, &winH, &vp);
    glViewport(winX,winY,winW,winH);
    glGetIntegerv(GL_VIEWPORT, vp);
    delete[] vp;
  }
  glGraph->paintGL();
}

void GlGraphWidget::resizeGL(int w, int h) {
  TRACE_EXEC();
  if (glGraph != NULL) glGraph->resizeGL(w,h);
}
//==================================================
void GlGraphWidget::drawGraph() {
  TRACE_EXEC();
  if (glGraph != NULL) glGraph->drawGraph();
}

void GlGraphWidget::deleteElement(int x, int y) {
  if (glGraph == NULL) return;
  drawTimer->stop();
  bool result;
  AtomType type;
  node tmpNode;
  edge tmpEdge;
  Observable::holdObservers();
  result = glGraph->doSelect(x, y, type, tmpNode, tmpEdge);
  bool strahler=glGraph->isViewStrahler();
  glGraph->setViewStrahler(false);
  if(result==true) {
    switch(type) {
    case NODE: glGraph->getSuperGraph()->delNode(tmpNode); break;
    case EDGE: glGraph->getSuperGraph()->delEdge(tmpEdge); break;
    }
  }
  glGraph->setViewStrahler(strahler);
  redraw();
  Observable::unholdObservers();
}
//==========================================================
void GlGraphWidget::setContextCoord(int x, int y) {
  xContext = x;
  yContext = y;
}

void GlGraphWidget::contextDel() {
  deleteElement(xContext,yContext);
}

void GlGraphWidget::contextSelect() {
  if (glGraph == NULL) return;
  Observable::holdObservers();
  bool result;
  AtomType type;
  node tmpNode;
  edge tmpEdge;
  SelectionProxy *elementSelected=getProxy<SelectionProxy>(glGraph->getSuperGraph(),"viewSelection");
  elementSelected->setAllNodeValue(false);
  elementSelected->setAllEdgeValue(false);
  result=glGraph->doSelect(xContext,yContext,type,tmpNode,tmpEdge);
  if (result==true) {
    switch(type) {
    case NODE: elementSelected->setNodeValue(tmpNode, true); break;
    case EDGE: elementSelected->setEdgeValue(tmpEdge, true); break;
    }
  }
  redraw();
  Observable::unholdObservers();
  
}

void GlGraphWidget::contextProperties() {
  node tmpNode;
  edge tmpEdge;
  AtomType type;
  
  if (glGraph->doSelect(xContext, yContext, type, tmpNode, tmpEdge)) {
    switch(type) {
    case NODE: emitNodeClicked(tmpNode); break;
    case EDGE: emitEdgeClicked(tmpEdge); break;
    }
  }
}

void GlGraphWidget::contextAddRemoveSelection() {
  if (glGraph == NULL) return;
  Observable::holdObservers();
  bool result;
  AtomType type;
  node tmpNode;
  edge tmpEdge;
  SelectionProxy *elementSelected=getProxy<SelectionProxy>(glGraph->getSuperGraph(), "viewSelection");
  result=glGraph->doSelect(xContext, yContext, type, tmpNode, tmpEdge);
  if (result==true) {
    switch(type) {
    case NODE: elementSelected->setNodeValue(tmpNode, !elementSelected->getNodeValue(tmpNode)); break;
    case EDGE: elementSelected->setEdgeValue(tmpEdge, !elementSelected->getEdgeValue(tmpEdge)); break;
    }
    redraw();
  }
  Observable::unholdObservers();
}

void GlGraphWidget::delSelection() {
  if (glGraph == NULL) return;
  Iterator<node> *itN=glGraph->getSuperGraph()->getNodes();
  Iterator<edge> *itE=glGraph->getSuperGraph()->getEdges();
  list<node> tmpListN;
  list<edge> tmpListE;
  SelectionProxy *elementSelected=getProxy<SelectionProxy>(glGraph->getSuperGraph(),"viewSelection");
  bool strahler=glGraph->isViewStrahler();
  glGraph->setViewStrahler(false);
  Observable::holdObservers();
  for(; itE->hasNext();) {
    edge ite=itE->next();
    if (elementSelected->getEdgeValue(ite)==true)   
      tmpListE.push_back(ite);
  } delete itE;
  for(; itN->hasNext();) {
    node itv=itN->next();
    if (elementSelected->getNodeValue(itv)==true)
      tmpListN.push_back(itv);
  } delete itN;
  while (!tmpListE.empty()) {
    glGraph->getSuperGraph()->delEdge(tmpListE.front());
    tmpListE.pop_front();
  }
  while (!tmpListN.empty()) {
    glGraph->getSuperGraph()->delNode(tmpListN.front());
    tmpListN.pop_front();
  }
  glGraph->setViewStrahler(strahler);
  redraw();
  Observable::unholdObservers();
}

//=====================================================
// Events 
//=====================================================
void GlGraphWidget::mousePressEvent(QMouseEvent *e) {
    if (mouse != NULL) mouse->mPressEvent(this, e);
}
//================================================
void GlGraphWidget::mouseMoveEvent(QMouseEvent *e) {
  if (mouse != NULL) mouse->mMoveEvent(this, e);
}

void GlGraphWidget::mouseReleaseEvent(QMouseEvent *e) {
  if (mouse != NULL) mouse->mReleaseEvent(this ,  e);
}

void GlGraphWidget::keyPressEvent(QKeyEvent *e) {
  if (mouse != NULL) mouse->keyPressEvent(this, e);
}

void GlGraphWidget::keyReleaseEvent(QKeyEvent *e) {
  if (mouse != NULL) mouse->keyReleaseEvent(this, e);
}

void GlGraphWidget::wheelEvent(QWheelEvent *e) {
  if (mouse != NULL) mouse->wheelEvent(this, e);
}
