//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 26/09/2001
 This program is free software; you can redistribute it and/or modify  *
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifndef TULIP_PROPERTIESPROXY_H
#define TULIP_PROPERTIESPROXY_H

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif

#include "Observable.h"
#include "Types.h"
#include "Property.h"
#include "PropertyProxyContainer.h"
#include "GraphIterator.h"

#include "tulipconf.h"
//=============================================================

class PProxy
{
public:
  virtual ~PProxy(){}
  ///
  virtual void erase(const node)=0;
  virtual void erase(const edge)=0;
};

/**
   This class is used to store a property of a graph.
   A property is a couple of two functions:
   - One from the set of nodes to a set of Tnode value
   - One from the set of edges to a set of Tedge value

   A PropertyProxy can be connected or not to a Property.
   In the first case it can be seen as buffer beetween the property and the user.
   In the second case it is only a memory area for storing data.
   A PropertyProxy is an observable, so it can be observe by others objects.
*/
template <class Tnode, class Tedge> class PropertyProxy:public PProxy,public Observable
{
public:
  ///
  ~PropertyProxy() ;
  //=============================================================
  ///Reset the proxy and notify the observers
  void reset();
  //=============================================================
  ///Change the property associated to the cahce and make a reset
  void changeCurrentProperty(Property< Tnode ,Tedge > *c, const std::string &s);
  //================================================================
  ///Return the name of this property
  std::string getName();
  //=============================================================
  /** 
     Return the node default value of the property proxy
     warnning: If the type is a pointer it can produce big memory
     leak if the user doesn't manage the memory himself
   */
  typename Tnode::RealType getNodeDefaultValue();
  typename Tedge::RealType getEdgeDefaultValue();
  //=============================================================
  /**
     Return the value associated to the node n in this property.
     If the value is in the cahce it is done in constant time.
     Else if it is a computed property, it calls the associated property
     If there is no value and no algorithms return default value
     depending of the type.
  */
  typename Tnode::RealType & getNodeValue(const node n );
  //=============================================================
  /**
     Return the value associated to the edge e in this property.
     If the value is in the cahce it is done in constant time.
     Else if it is a computed property call the associated algorithm
     If there is no vlaue and no algorithms return default value
     depending of the type.
  */
  typename Tedge::RealType & getEdgeValue(const edge e);
  //=============================================================
  ///Set the value of the node n and notify the observers
  void setNodeValue(const node n,const typename Tnode::RealType v);
  virtual void setNodeValue_handler(const node n){};
  //=============================================================
  ///Set the value of the edge e and notify the observers
  void setEdgeValue(const edge e,const typename Tedge::RealType v);
  virtual void setEdgeValue_handler(const edge e){};
  //=============================================================
  ///Set the value of all nodes and notify the observers
  void setAllNodeValue(const typename Tnode::RealType v);
  virtual void setAllNodeValue_handler(){};
  //============================================================
  ///Set the value of all edges and notify the observers
  void setAllEdgeValue(const typename Tedge::RealType v);
  virtual void setAllEdgeValue_handler(){};
  //==============================================================
  ///Reset the cache and restart the computation of this property.
  bool recompute(std::string &erreurMsg);
  //==============================================================
  ///Enable to free memory in case of pointer type
  virtual void reset_handler(){};
  //=================================================================================
  virtual void recompute_handler(){};
  //=================================================================================
  void erase(const node n);
  //=================================================================================
  void erase(const edge e);
  //=================================================================================
  virtual PropertyProxy<Tnode,Tedge>& operator =(PropertyProxy<Tnode,Tedge> &proxy);
  //=================================================================================
  ///Enable to clone part of sub_class
  virtual void clone_handler(PropertyProxy<Tnode,Tedge> &){};

  //=================================================================================protected:


protected:

  typedef STL_EXT_NS::hash_map<node, typename Tnode::RealType> NodeProperties;
  typedef STL_EXT_NS::hash_map<edge, typename Tedge::RealType> EdgeProperties;

  STL_EXT_NS::hash_map<node, typename Tnode::RealType> nodeProperties;
  STL_EXT_NS::hash_map<edge, typename Tedge::RealType> edgeProperties;

  typename Tnode::RealType nodeDefaultValue;
  typename Tedge::RealType edgeDefaultValue;
  Property< Tnode , Tedge > *currentProperty;
  SuperGraph *superGraph;
  std::string name;
  PProxy *propertyProxy;
  bool edgeValueSetup,nodeValueSetup;

  PropertyContext context;
  //=============================================================
  ///Init the cache and init the object with the information contained in context.
  PropertyProxy(const PropertyContext *context);

};

#include "./cxx/PropertyProxy.cxx"

#endif
