#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dirent.h>

#include "gzstream.h"

#include "tulip/TlpTools.h"
#include "tulip/SuperGraphImpl.h"
#include "tulip/SubGraph.h"
#include "tulip/Reflect.h"

using namespace std;

#ifndef _TULIP_LIB_DIR
#define _TULIP_LIB_DIR "/usr/local/lib/"
#endif

static const char PATH_DELIMITER=':';
static const char *TULIP_PLUGINS_PATH_VARIABLE="TULIP_PLUGINS_PATH";

string TlpTools::TulipLibDir;
string TlpTools::TulipPluginsPath;

//=========================================================
void TlpTools::initTulipLib() {
  char *getEnvTlp;
  getEnvTlp=getenv("TLPDIR");
  if (getEnvTlp==0)
    TulipLibDir=string(_TULIP_LIB_DIR);
  else
    TulipLibDir=string(getEnvTlp);
  getEnvTlp=getenv(TULIP_PLUGINS_PATH_VARIABLE);
  if (getEnvTlp!=0)
    TulipPluginsPath=string(getEnvTlp);
  TulipPluginsPath=TulipLibDir+"/tlp/plugins"+PATH_DELIMITER+TulipPluginsPath;
}
//=========================================================
istream *TlpTools::getIgzstream(const char *name, int open_mode) {
  return new igzstream(name, open_mode);
}

ostream *TlpTools::getOgzstream(const char *name, int open_mode) {
  return new ogzstream(name, open_mode);
}
//=========================================================
// accepts only file names ending with ".so"; used by TemplateFactory.cxx for directory filtering
int __tulip_selectSO(const struct dirent *ent) {
  const char *SO = ".so";
  int idx = strlen(ent->d_name) - 3;
  if (idx < 0) return 0;
  
  for (int i=0; i<3; ++i) {
    if ((ent->d_name[idx + i]) != SO[i]) return 0;
  }
  return 1;
}
//=========================================================
TemplateFactory<ClusteringFactory,Clustering,ClusterContext > TlpTools::clusteringFactory;
TemplateFactory<ImportModuleFactory,ImportModule,ClusterContext > TlpTools::importFactory;
TemplateFactory<ExportModuleFactory,ExportModule,ClusterContext > TlpTools::exportFactory;
//=========================================================
SuperGraph * TlpTools::newSuperGraph(){
  return new SuperGraphImpl();
}
//=========================================================
SuperGraph * TlpTools::inducedSubGraph(SuperGraph *sg, const std::set<node> &setNode,string name) {
  SelectionProxy *sel1 =getLocalProxy<SelectionProxy>(sg,"sub graph build tmp");
  sel1->setAllNodeValue(false);
  sel1->setAllEdgeValue(false);
  Iterator<node> *itN=sg->getNodes();
  for (;itN->hasNext();) {
    node itn=itN->next();
    if (setNode.find(itn)!=setNode.end()) {
      sel1->setNodeValue(itn,true);
    }
  }delete itN;
  Iterator<edge> *itE=sg->getEdges();
  for (;itE->hasNext();) {
    edge ite=itE->next();
    if (setNode.find(sg->source(ite))!=setNode.end() && setNode.find(sg->target(ite))!=setNode.end()) {
      sel1->setEdgeValue(ite,true);
    }
  }delete itE;
  SubGraph * tmpSubGraph;
  tmpSubGraph = sg->addView(name,sel1);
  sg->getPropertyProxyContainer()->delLocalProxy("sub graph build tmp");
  return tmpSubGraph->getAssociatedSuperGraph();
}
//=========================================================
SuperGraph * TlpTools::newSubGraph(SuperGraph *sg,string name) {
  SelectionProxy *sel1 =getLocalProxy<SelectionProxy>(sg,"sub graph build tmp");
  sel1->setAllNodeValue(false);
  sel1->setAllEdgeValue(false);
  SubGraph * tmpSubGraph;
  tmpSubGraph = sg->addView(name,sel1);
  sg->getPropertyProxyContainer()->delLocalProxy("sub graph build tmp");
  return tmpSubGraph->getAssociatedSuperGraph();
}
//=========================================================
SuperGraph * TlpTools::newCloneSubGraph(SuperGraph *sg,string name) {
  SelectionProxy *sel1 =getLocalProxy<SelectionProxy>(sg,"sub graph build tmp");
  sel1->setAllNodeValue(true);
  sel1->setAllEdgeValue(true);
  SubGraph * tmpSubGraph;
  tmpSubGraph = sg->addView(name,sel1);
  sg->getPropertyProxyContainer()->delLocalProxy("sub graph build tmp");
  return tmpSubGraph->getAssociatedSuperGraph();
}
//=========================================================
SuperGraph * TlpTools::importGraph(const string &alg, DataSet &dataSet, PluginProgress *plugProgress)
{
  if (!TlpTools::importFactory.exists(alg)) {
    cerr << "libtulip: " << __FUNCTION__ << ": import plugin \"" << alg << "\" doesn't exists (or is not loaded)" << endl;
    return NULL;
  }
  
  SuperGraph *newSuperGraph=new SuperGraphImpl();
  ClusterContext tmp;
  tmp.superGraph=newSuperGraph;
  tmp.dataSet = &dataSet;
  PluginProgress *tmpProgress;
  bool deletePluginProgress=false;
  if (plugProgress==0) {
    tmpProgress=new PluginProgressDefault();
    deletePluginProgress=true;
  }
  else tmpProgress=plugProgress;
  tmp.pluginProgress=tmpProgress;
  ImportModule *newImportModule=TlpTools::importFactory.getObject(alg, tmp);
  assert(newImportModule!=0);
  if (!newImportModule->import("")) {
    delete newSuperGraph;
    newSuperGraph=0;
  }
  if (deletePluginProgress) delete tmpProgress;
  delete newImportModule;
  dataSet = *tmp.dataSet;
  return newSuperGraph;
}
//=========================================================
bool TlpTools::exportGraph(SuperGraph *sg,ostream &os, const string &alg, DataSet &dataSet, PluginProgress *plugProgress)
{
  if (!TlpTools::exportFactory.exists(alg)) {
    cerr << "libtulip: " << __FUNCTION__ << ": export plugin \"" << alg << "\" doesn't exists (or is not loaded)" << endl;
    return false;
  }
  
  bool result;
  bool deletePluginProgress=false;
  ClusterContext tmp;
  tmp.superGraph=sg;
  tmp.dataSet=&dataSet;
  PluginProgress *tmpProgress=NULL;
  if (plugProgress==0) {
    tmpProgress=new PluginProgressDefault();
    deletePluginProgress=true;
  }
  else tmpProgress=plugProgress;
  tmp.pluginProgress=tmpProgress;
  ExportModule *newExportModule=TlpTools::exportFactory.getObject(alg, tmp);
  assert(newExportModule!=0);
  result=newExportModule->exportGraph(os,sg);

  if (deletePluginProgress) delete tmpProgress;
  delete newExportModule;
  return result;
}
//=========================================================
bool TlpTools::clusterizeGraph(SuperGraph *sg,string &errorMsg,DataSet *dataSet, const string &alg,PluginProgress *plugProgress)
{
  if (!TlpTools::clusteringFactory.exists(alg)) {
    cerr << "libtulip: " << __FUNCTION__ << ": cluster plugin \"" << alg << "\" doesn't exists (or is not loaded)" << endl;
    return false;
  }
  
  bool result;
  bool deletePluginProgress=false;
  ClusterContext tmp;
  tmp.superGraph=sg;
  tmp.dataSet=dataSet;
  PluginProgress *tmpProgress;
  if (plugProgress==0) {
    tmpProgress=new PluginProgressDefault();
    deletePluginProgress=true;
  }
  else tmpProgress=plugProgress;
  tmp.pluginProgress=tmpProgress; 
  Clustering *newClustering=TlpTools::clusteringFactory.getObject(alg, tmp);
  if ((result=newClustering->check(errorMsg)))
    newClustering->run();
  delete newClustering;
  if (deletePluginProgress) delete tmpProgress;
  return result;
}
//=========================================================
static void loadPlugins(string dir,PluginLoader *plug)
{
  SizesProxy::factory.load(dir + "sizes", "Sizes",plug);
  IntProxy::factory.load(dir + "int", "Int",plug);
  LayoutProxy::factory.load(dir + "layout" , "Layout",plug);
  ColorsProxy::factory.load(dir + "colors" , "Colors",plug);
  MetricProxy::factory.load(dir + "metric" , "Metric",plug);
  StringProxy::factory.load(dir + "string" , "String",plug);
  SelectionProxy::factory.load(dir + "selection" , "Selection",plug);
  TlpTools::clusteringFactory.load(dir + "clustering" , "Cluster",plug);
  TlpTools::importFactory.load(dir + "import" , "Import Module",plug);
  TlpTools::exportFactory.load(dir + "export" , "Export Module",plug);
}


void TlpTools::loadPlugins(PluginLoader *plug)
{
  string::const_iterator begin=TulipPluginsPath.begin();
  string::const_iterator end=begin;
  while (end!=TulipPluginsPath.end())
    if ((*end)==PATH_DELIMITER) {
      if (begin!=end) 
	loadPlugins(string(begin,end)+"/",plug);
      ++end;
      begin=end;
    } else
      ++end;
  if (begin!=end) 
    loadPlugins(string(begin,end)+"/",plug);
}

//=========================================================
