#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <GL/glu.h>

#include <tulip/SelectionProxy.h>
#include <tulip/StringProxy.h>
#include <tulip/MetricProxy.h>
#include <tulip/MetaGraphProxy.h>
#include <tulip/IntProxy.h>
#include <tulip/LayoutProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/ColorsProxy.h>
#include <tulip/TlpTools.h>
#include "tulip/GlFonts.h"
#include "tulip/GlLines.h"
#include "tulip/GlGraph.h"
#include "tulip/FanError.h"
#include "tulip/Glyph.h"
#include "tulip/TLPPixmapGlyph.h"
#include "tulip/TLPPixmapFont.h"
#include "tulip/Rendu.h"
#include "tulip/DynamicFunctions.h"


//static FTFont* normalFont=0;
//static FTFont* selectFont=0;
static Rendu* myRender = new Rendu();
static const unsigned int TLPMAXDEPTH=10;
const Color colorSelect= Color(255, 102, 255, 255);
static float colorMat[4] = { 0.0, 0.0, 0.0, 1.0 };

//====================================================
inline void SetColor(const Color &c) {
  glColor4ub(c[0],c[1],c[2],c[3]);
}
inline void setMaterial(const Color &C) {
  colorMat[0]=((float)C[0])/255.0;
  colorMat[1]=((float)C[1])/255.0;
  colorMat[2]=((float)C[2])/255.0;
  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, colorMat);
}
//====================================================
void GlGraph::drawMetaNode(node n,unsigned int depth, bool labels) {
  SuperGraph *sp=_superGraph;
  _superGraph=elementMetaGraph->getNodeValue(n);
  initProxies();
  Coord maxL=elementLayout->getMax(_superGraph);
  Coord minL=elementLayout->getMin(_superGraph);
  Size  maxS=elementSize->getMax(_superGraph);
  double saveMatrix[16];
  for (unsigned int i=0;i<16;++i) saveMatrix[i]=modelviewMatrix[i];
  Coord translate= (elementLayout->getMax(_superGraph)+elementLayout->getMin(_superGraph))/-2.0;
  double max=maxS[0]+maxL[0]-minL[0];
  max=max>?maxS[1]+maxL[1]-minL[1];
  max=max>?maxS[2]+maxL[2]-minL[2];
  
  if (max<0.0001) max=1;
  max=1/max;
  glPushMatrix();
  glScalef(max,max,max);
  glTranslatef(translate[0],translate[1],translate[2]);
  glGetDoublev (GL_MODELVIEW_MATRIX, modelviewMatrix);

  Iterator<node> *itN=_superGraph->getNodes();
  while (itN->hasNext()) {
    node itn=itN->next();
    if (!labels)
      drawNode(itn,depth+1);
    else {
      drawLabel(itn);
    }
  } delete itN;

  Iterator<edge> *itE=_superGraph->getEdges();
  while (itE->hasNext()) {
    edge ite=itE->next();
    if (!labels)
      drawEdge(ite);
    else
      drawEdgeLabel(ite);
  } delete itE;
  
  glPopMatrix();
  _superGraph=sp;
  initProxies();
  for (unsigned int i=0;i<16;++i) modelviewMatrix[i]=saveMatrix[i];
}
//====================================================
float GlGraph::projectSize(const Coord& position, const Size& size) {
  bool project;  
  double x1Scr,y1Scr,zScr;
  double x2Scr,y2Scr;
  if (project=(gluProject(position[0] - size[0],
			  position[1] - size[1],
			  position[2] - size[2],
			  modelviewMatrix,projectionMatrix,viewportArray,&x1Scr,&y1Scr,&zScr) &&
	       gluProject(position[0] + size[0],
			  position[1] + size[1],
			  position[2] + size[2],
			  modelviewMatrix,projectionMatrix,viewportArray,&x2Scr,&y2Scr,&zScr)))  {
    if ( (x1Scr<0.0 && x2Scr<0.0) ||
	 (x1Scr>winW && x2Scr>winW) ||
	 (y1Scr<0.0 && y2Scr<0.0) ||
	 (y1Scr>winH && y2Scr>winH)) {
      return 0;
    }
  }
  else 
    return 0;
  return ((x1Scr-x2Scr)*(x1Scr-x2Scr)+(y1Scr-y2Scr)*(y1Scr-y2Scr));
}
//====================================================
void GlGraph::drawNode(node n, unsigned int depth) {
  if (depth>TLPMAXDEPTH) return;
  const Coord &nodeCoord = elementLayout->getNodeValue(n);
  const Size &nodeSize = elementSize->getNodeValue(n);
  if( nodeSize == Size(0,0,0) )	return;
  float lod= projectSize(nodeCoord,nodeSize);
  if (lod==0) return;
  if (lod < 8.1) {
    const Color &nodeColor = elementColor->getNodeValue(n);
    if (!elementSelected->getNodeValue(n)) {
      setMaterial(nodeColor);
    } else
      setMaterial(Color(255, 102, 255, 255));
    glPassThrough(n.id);
    glBegin(GL_POINTS);
     glVertex3f(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glEnd();
  }
  else {
    this->desactivateTexture();
    glPushMatrix();
    glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glScalef(nodeSize[0], nodeSize[1], nodeSize[2]);
    if (elementSelected->getNodeValue(n)) {
      assert(glIsList(selectionDL));
      glCallList(selectionDL);
    }
    glPassThrough(n.id);
    if (elementMetaGraph->getNodeValue(n) == 0) {
      glyphs.get(elementShape->getNodeValue(n))->draw(n);
    }
    else {
      setMaterial(colorSelect);
      assert(glIsList(metaGraphDL));
      glCallList(metaGraphDL);
      drawMetaNode(n,depth);
    }
    glPopMatrix();
  }
}
//====================================================
unsigned int GlGraph::drawNodes(unsigned int number, Iterator<node> *itN) {
  if (!itN->hasNext() || number==0) return 0;
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glMatrixMode(GL_MODELVIEW); 
  glEnable(GL_LIGHTING);
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_COLOR_MATERIAL);
  glDisable(GL_BLEND);
  initProxies();
  glPassThrough(-1.0);	  //mark beginning of nodes drawing for feedback mode
  while ((itN->hasNext()) && (number >0)) {
    number--;
    drawNode(itN->next(),0);
  }
  glPassThrough(-1.0);  //mark end of nodes drawing
  glPopAttrib();
  return tmp-number;
}
//====================================================
void GlGraph::drawPixmapFont(const string &str, const Coord &position, bool selected) {
  /*
  FTFont *bitmapFont;
  int rastPos[4];
  unsigned int border = labelsBorder;
  float x1, y1, z1, x2, y2, z2;
  if (selected) {
    bitmapFont=selectFont;
    glColor4ub(255,0,0,0);
  }
  else {
    bitmapFont=normalFont;
    glColor4ub(0,0,0,0);
  }
  glRasterPos3f(position[0],position[1],position[2]);
  bitmapFont->BBox( str.c_str(), x1, y1, z1, x2, y2, z2);
  glBitmap(0,0,0,0,-(x2+x1)/2+x1 - border ,-(y2+y1)/2+y1 - border,0);
  glGetIntegerv(GL_CURRENT_RASTER_POSITION,rastPos);
  if (!occlusionTest.addRectangle(RectangleInt2D(rastPos[0],rastPos[1],
						 rastPos[0]+(int)(x2-x1)+2*border,
						 rastPos[1]+(int)(y2-y1)+2*border))) {
    glRasterPos3f(position[0],position[1],position[2]);
    glBitmap(0,0,0,0,-(x2+x1)/2,-(y2+y1)/2,0);
    bitmapFont->Render(str.c_str());
  }
  */
}
//====================================================
void GlGraph::drawEdgeLabel(edge e) {
  string tmp=elementLabel->getEdgeValue(e);
  if (tmp.length()==0) return;
  const Coord & srcCoord = elementLayout->getNodeValue(_superGraph->source(e));
  const Coord & tgtCoord = elementLayout->getNodeValue(_superGraph->target(e));
  const LineType::RealType &bends = elementLayout->getEdgeValue(e);
  Coord position;
  if (bends.empty()) {
    position = (srcCoord+tgtCoord) / 2.0;
  }
  else {
    if (bends.size()%2 == 0)
      position = (bends[bends.size()/2-1]+bends[bends.size()/2]) / 2.0;
    else
      position = bends[bends.size()/2];
  }
  int rastPos[4];
  glRasterPos3f(position[0], position[1], position[2]);
  glGetIntegerv(GL_CURRENT_RASTER_POSITION,rastPos);
  if(occlusionTest.testRectangle(RectangleInt2D(rastPos[0] - labelsBorder - 5,
						rastPos[1] - labelsBorder - 5,
						rastPos[0] + labelsBorder + 5,
						rastPos[1] + labelsBorder + 5)))
    return;
  myRender->setString(tmp, VERBATIM);
  if (elementSelected->getEdgeValue(e))
    glColor4ub(255,0,0,255);
  else
    glColor4ub(0,0,0,255);
  DynamicFunctions* RenderContext = DynamicFunctions::create();
  RenderContext->setMode(TLPPIXMAP);
  float w,h;
  //  w_max = nodeSize.getW();
  myRender->getBoundingBox(50, h, w);
  if(!occlusionTest.addRectangle(RectangleInt2D(rastPos[0]-(int)(w/2.) - labelsBorder,
						rastPos[1]-(int)(h/2.)-labelsBorder,
						rastPos[0]+(int)(w/2.)+labelsBorder,
						rastPos[1]+(int)(h/2.)+labelsBorder))) {
    myRender->draw(50, w);
  }
  //drawPixmapFont(tmp,position,elementSelected->getEdgeValue(e));
}
  //====================================================
void GlGraph::drawLabel(node n) {
  const string &tmp=elementLabel->getNodeValue(n);
  if (tmp.length()==0) return;
  const Coord &nodeCoord = elementLayout->getNodeValue(n);
  const Size &nodeSize = elementSize->getNodeValue(n);

  Color fontColor;
  if (elementSelected->getNodeValue(n))
    fontColor.set(255,0,0,255);
  else
    fontColor.set(0,0,0,255);
  if (_FontsType==0) fontColor.set(0,0,0,255);
  if (_FontsType==2) fontColor.set(255,255,255,255);

  glColor4ub(fontColor.getR(),fontColor.getG(),fontColor.getB(),fontColor.getA());
  setMaterial(fontColor);

  //  double wi,he;
  int rastPos[4];
  float w_max = 300;
  float w,h;

  DynamicFunctions* RenderContext = DynamicFunctions::create();
  float div_w, div_h;
  if (tmp.length()>0) {
    switch(_FontsType){
    case 0:
      if (projectSize(nodeCoord,nodeSize) < 36.0) return;
      myRender->setString(tmp, VERBATIM);
      RenderContext->setMode(POLYGON);
      w_max = nodeSize.getW();
      myRender->getBoundingBox(w_max, h, w);
      glPushMatrix();
      glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
      div_w = nodeSize.getW()/w;
      div_h = nodeSize.getH()/h;
      if(div_h > div_w) 
	glScalef(-div_w, -div_w, 1);
      else 
	glScalef(-div_h, -div_h, 1);
      myRender->draw(w_max,w);
      glPopMatrix();
      break;
    case 1:
      glRasterPos3f(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
      glGetIntegerv(GL_CURRENT_RASTER_POSITION,rastPos);
      if(occlusionTest.testRectangle(RectangleInt2D(rastPos[0] - labelsBorder - 5,
						    rastPos[1] - labelsBorder - 5,
						    rastPos[0] + labelsBorder + 5,
						    rastPos[1] + labelsBorder + 5)))
	return;
      myRender->setString(tmp, VERBATIM);
      RenderContext->setMode(TLPPIXMAP);
      w_max = nodeSize.getW();
      myRender->getBoundingBox(w_max, h, w);
      if(!occlusionTest.addRectangle(RectangleInt2D(rastPos[0]-(int)(w/2.) - labelsBorder,
						    rastPos[1]-(int)(h/2.) - labelsBorder,
						    rastPos[0]+(int)(w/2.) + labelsBorder,
						    rastPos[1]+(int)(h/2.) + labelsBorder))) {
	myRender->draw(w_max, w);
      }
      break;
    case 2:
      if (projectSize(nodeCoord,nodeSize) < 36.0) return;
      myRender->setString(tmp, VERBATIM);
      RenderContext->setMode(TEXTURE);
      w_max = nodeSize.getW();
      myRender->getBoundingBox(w_max, h, w);
      glPushMatrix();
      glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
      div_w = nodeSize.getW()/w;
      div_h = nodeSize.getH()/h;
      if(div_h > div_w) 
	glScalef(-div_w, -div_w, 1);
      else 
	glScalef(-div_h, -div_h, 1);
      glEnable( GL_TEXTURE_2D);
      glEnable(GL_BLEND);
      glBlendFunc(GL_ONE_MINUS_DST_COLOR,GL_ONE_MINUS_SRC_COLOR);
      myRender->draw(w_max, w);
      glDisable(GL_BLEND);
      glDisable( GL_TEXTURE_2D);
      glPopMatrix();
      break;
    default:
      cerr << "GlGraph::DrawNodes unknown fonts" << endl;
      break;
    }
  }
}
//====================================================
unsigned int GlGraph::drawEdgeLabels(unsigned int number, Iterator<edge> *itE, bool mode) {
  if (!itE->hasNext() || number==0) return 0;
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glMatrixMode(GL_MODELVIEW); 
  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS );
  glDisable(GL_BLEND);
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  while ((itE->hasNext()) && (number >0)) {
    edge ite=itE->next();
    if (elementSelected->getEdgeValue(ite)==mode){
      number--;
      drawEdgeLabel(ite);
    }
  }
  glDepthFunc(GL_LEQUAL );
  glPopAttrib();
  return tmp-number;
}
//====================================================
unsigned int GlGraph::drawLabels(unsigned int number, Iterator<node> *itN, bool mode) {
  if (!itN->hasNext() || number==0) return 0;
  unsigned int tmp=number;

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glMatrixMode(GL_MODELVIEW); 
  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS );
  glDisable(GL_BLEND);
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  if(mode)
    myRender->setContext(tlp::TulipLibDir + "tlp/bitmaps/" + "fontb.ttf", 14, 0, 0, 255);
  else	
    myRender->setContext(tlp::TulipLibDir + "tlp/bitmaps/" + "font.ttf", 12, 255, 255, 255);
  while ((itN->hasNext()) && (number >0)) {
    node itv=itN->next();
    if (elementSelected->getNodeValue(itv)==mode){
      number--;
      drawLabel(itv);
    }
  }
  glDepthFunc(GL_LEQUAL );
  glPopAttrib();
  return tmp-number;
}
//=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
#define L3D_BIT (1<<9)
void GlGraph::drawEdge(const Coord &startNode, const Coord &finalNode,
                       const Coord &startPoint,const LineType::RealType &bends, const Coord &endPoint,
                       const Color &startColor, const Color &endColor, const Size &size, int shape, const bool selected) {
  const double width = selected ? 3.0 : 1.0;
  
  if (_edge3D) shape |= L3D_BIT;
  if (shape & L3D_BIT) {
    if (shape < (L3D_BIT+16)) {
      glEnable(GL_LIGHTING);
      glEnable(GL_COLOR_MATERIAL);
      if (selected) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    }
    else shape  = shape & ~L3D_BIT; //no 3D model defined, falling back to Line model
  }
  
  switch (shape) {
  case 0:GlLines::glDrawCurve(startPoint,bends,endPoint,width,0,startColor,endColor);break;
  case 1:GlLines::glDrawCurve(startPoint,bends,endPoint,width,1,startColor,endColor);break;
  case 2:GlLines::glDrawCurve(startPoint,bends,endPoint,width,2,startColor,endColor);break;
  case 3:GlLines::glDrawCurve(startPoint,bends,endPoint,width,3,startColor,endColor);break;
  case 4:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
  case 5:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
  case 6:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
  case 7:GlLines::glDrawBezierCurve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
  case 8:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
  case 9:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
  case 10:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
  case 11:GlLines::glDrawSplineCurve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
  case 12:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,0,startColor,endColor);break;
  case 13:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,1,startColor,endColor);break;
  case 14:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,2,startColor,endColor);break;
  case 15:GlLines::glDrawSpline2Curve(startPoint,bends,endPoint,10,width,3,startColor,endColor);break;
    //3D lines
  case L3D_BIT: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::PLAIN,
					 GlLines::LINEAR, startColor, endColor); break;
  case L3D_BIT+1: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DOT,
                                           GlLines::LINEAR, startColor, endColor); break;
  case L3D_BIT+2: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DASHED,
                                           GlLines::LINEAR, startColor, endColor); break;
  case L3D_BIT+3: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::ALTERNATE,
                                           GlLines::LINEAR, startColor, endColor); break;

  case L3D_BIT+4: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::PLAIN,
                                           GlLines::BEZIER, startColor, endColor); break;
  case L3D_BIT+5: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DOT,
                                           GlLines::BEZIER, startColor, endColor); break;
  case L3D_BIT+6: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DASHED,
                                           GlLines::BEZIER, startColor, endColor); break;
  case L3D_BIT+7: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::ALTERNATE,
                                           GlLines::BEZIER, startColor, endColor); break;

  case L3D_BIT+8: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::PLAIN,
                                           GlLines::SPLINE3, startColor, endColor); break;
  case L3D_BIT+9: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DOT,
                                           GlLines::SPLINE3, startColor, endColor); break;
  case L3D_BIT+10: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DASHED,
					    GlLines::SPLINE3, startColor, endColor); break;
  case L3D_BIT+11: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::ALTERNATE,
					    GlLines::SPLINE3, startColor, endColor); break;
    
  case L3D_BIT+12: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::PLAIN,
					    GlLines::SPLINE4, startColor, endColor); break;
  case L3D_BIT+13: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DOT,
					    GlLines::SPLINE4, startColor, endColor); break;
  case L3D_BIT+14: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::DASHED,
					    GlLines::SPLINE4, startColor, endColor); break;
  case L3D_BIT+15: GlLines::glDrawExtrusion(startNode, finalNode, startPoint, bends, endPoint, 10, size, GlLines::ALTERNATE,
					    GlLines::SPLINE4, startColor, endColor); break;

  default:
    glDisable(GL_LIGHTING);
    GlLines::glDrawCurve(startPoint,bends,endPoint,width,0,startColor,endColor);break;
  }
  
  if ((shape & L3D_BIT) && (shape < (L3D_BIT+16))) {
    if (selected) glPolygonMode(GL_FRONT, GL_FILL);
    glDisable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
  }
}
//====================================================
void GlGraph::drawEdge(edge ite) {
  const Coord & srcCoord = elementLayout->getNodeValue(_superGraph->source(ite));
  Coord tgtCoord = elementLayout->getNodeValue(_superGraph->target(ite));
  Coord srcAnchor, tgtAnchor, endLineAnchor, tmpAnchor;
  const Size &srcSize = elementSize->getNodeValue(_superGraph->source(ite));
  const Size &tgtSize = elementSize->getNodeValue(_superGraph->target(ite));
  const Size &edgeSize = elementSize->getEdgeValue(ite);
  Color C1,C2;
  const LineType::RealType &lCoord = elementLayout->getEdgeValue(ite);
  // set color
  if (isEdgeColorInterpolate()) {
    C1 = elementColor->getNodeValue(_superGraph->source(ite));
    C2 = elementColor->getNodeValue(_superGraph->target(ite));
  }
  else {
    C1 = C2 = elementColor->getEdgeValue(ite);
  }
  
  // set srcAnchor, tgtAnchor. tmpAnchor will be on the point just before tgtAnchor
  Glyph *sourceGlyph=glyphs.get(elementShape->getNodeValue(_superGraph->source(ite)));
  tmpAnchor = (lCoord.size() > 0) ? lCoord.front() : tgtCoord;
  srcAnchor = sourceGlyph->getAnchor(srcCoord, tmpAnchor, srcSize);

  Glyph *targetGlyph=glyphs.get(elementShape->getNodeValue(_superGraph->target(ite)));
  //this time we don't take srcCoord but srcAnchor to be oriented to where the line comes from
  tmpAnchor = (lCoord.size() > 0) ? lCoord.back() : srcAnchor;
  tgtAnchor = targetGlyph->getAnchor(tgtCoord, tmpAnchor, tgtSize);

  //draw Arrow
  if (_viewArrow) {
    GLfloat matrix[16];
    float sizeT = edgeSize.getD();
    makeArrowMatrix((GLfloat *)&matrix, tmpAnchor, tgtAnchor, 1.0, 1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glEnable(GL_LIGHTING);
    glPushMatrix();
    glMultMatrixf((GLfloat *)&matrix);
    glScalef(sizeT, sizeT, sizeT);
    setMaterial(C2);
    assert(glIsList(arrowDL));
    glCallList(arrowDL);
    glPopMatrix();
    if (elementSelected->getEdgeValue(ite)==true) {      
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glPushMatrix();
      glMultMatrixf(matrix);
      glScalef(sizeT+0.1, sizeT+0.1, sizeT+0.1);
      setMaterial(colorSelect);
      assert(glIsList(arrowDL));
      glCallList(arrowDL);
      glPopMatrix();
      glPolygonMode(GL_FRONT, GL_FILL);
    }
    endLineAnchor = tmpAnchor - tgtAnchor;
    float nrm = endLineAnchor.norm();
    if( nrm > 0.00000001f ) {
	    endLineAnchor *= sizeT / nrm;
	    endLineAnchor += tgtAnchor;
    } else {
	    endLineAnchor = tgtAnchor;
    }
    tgtCoord = tgtAnchor; //this defines in drawEdge the arrow head as being the final node
    glDisable(GL_LIGHTING);
  }
  else {
    endLineAnchor = tgtAnchor;
  }
  
  //draw Edges
  if (elementSelected->getEdgeValue(ite)==true) {
    float w,h,d;
    edgeSize.get(w, h, d);
    Size selSize(w+0.05, h+0.05, d);
    drawEdge(srcCoord, tgtCoord, srcAnchor, lCoord, endLineAnchor, colorSelect, colorSelect,
	     selSize, elementShape->getEdgeValue(ite), true);
  }
  
  drawEdge(srcCoord, tgtCoord, srcAnchor, lCoord, endLineAnchor, C1, C2, edgeSize, elementShape->getEdgeValue(ite));
}
//====================================================
unsigned int GlGraph::drawEdges(unsigned int number,Iterator<edge> *itE) {
  if (!itE->hasNext() || number==0) return 0;
  unsigned int tmp=number;
  glDisable(GL_LIGHTING);
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  initProxies();
  while ((itE->hasNext()) && (number>0)) {
    number--;
    drawEdge(itE->next());
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
//Fan rendering methods for nodes David Auber, David Duke
void GlGraph::drawFanNode(node n) {
  if (elementSelected->getNodeValue(n)) {
    drawNode(n,0);
    Iterator<edge> *itE=_superGraph->getOutEdges(n);
    for (;itE->hasNext();) {
      edge ite=itE->next();
      drawEdge(ite);
    }delete itE;
  } 
  else {
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA,GL_ONE);
    Coord centerCoord,lastValid,lastTested;
    Color centerColor,lastTestedColor,lastValidColor;
    int numberOfChildren;
    if ((numberOfChildren=_superGraph->outdeg(n))==0)  {
      if (_superGraph->indeg(n)==0)  drawNode(n,0);
      return;
    }
    centerCoord=elementLayout->getNodeValue(n);
    centerColor=elementColor->getNodeValue(n);
    //Initiallize the FanError Object
    int k;
    if (numberOfChildren<5) k=2; else k=0;
    FanError fanError(centerCoord,6.28/(0.7*numberOfChildren+k),3);

    //We strore coordinates in order to compute normals
    vector<Coord> neighboors(numberOfChildren);
    vector<Color> neighboorsColors(numberOfChildren);
    Iterator<node> *itChild=_superGraph->getOutNodes(n);
    for (int i=0;itChild->hasNext();++i) {
      node itchild=itChild->next();
      neighboors[i]=elementLayout->getNodeValue(itchild);
      neighboorsColors[i]=elementColor->getNodeValue(itchild);
    }delete itChild;
    glFrontFace(GL_CW);

    //compute normal of the center average of the normal of all faces.
    Coord centerNormal(0,0,0);
    for (int j=0;j<numberOfChildren-1;++j) {
      centerNormal+=(neighboors[j]-centerCoord)^(neighboors[j+1]-centerCoord);
    }
    centerNormal/=(numberOfChildren-1);

    //Compute and draw the FRONT fans
    unsigned int result;
    unsigned int nbElements;
    node itchild;
    itChild=_superGraph->getOutNodes(n);
    itchild=itChild->next();
    lastTested=lastValid=elementLayout->getNodeValue(itchild);
    lastTestedColor=lastValidColor=elementColor->getNodeValue(itchild);
    result=2;
    nbElements=0;
    for (int i=0;itChild->hasNext();) {
      fanError.reset();
      fanError.addVertex(lastTested);
      if (result==2) {
	nbElements=0;
	glBegin(GL_TRIANGLE_FAN);
	//	SetMaterial(centerColor);
	SetColor(centerColor);
	glNormal3f(centerNormal.getX(),centerNormal.getY(),centerNormal.getZ());
	glVertex3f(centerCoord.getX() , centerCoord.getY() , centerCoord.getZ());
	//	SetMaterial(lastTestedColor);
	SetColor(lastTestedColor);
	if (i+1<numberOfChildren) {
	  Coord resultCoord=(neighboors[i+1]-neighboors[i])^(centerCoord-neighboors[i]);
	  glNormal3f(resultCoord.getX(),resultCoord.getY(),resultCoord.getZ());
	}
	glVertex3f(lastTested.getX() , lastTested.getY() , lastTested.getZ());	
      }
      lastValid=lastTested;
      for (;itChild->hasNext();) {
	node itchild=itChild->next();
	i++;
	lastTested=elementLayout->getNodeValue(itchild);
	lastTestedColor=elementColor->getNodeValue(itchild);
	if ((result=fanError.addVertex(lastTested))!=0) break;
	lastValid=lastTested;
	lastValidColor=lastTestedColor;
	nbElements++;
      }
      //      SetMaterial(lastValidColor);
      SetColor(lastValidColor);

      if (i-1>0) {
	Coord resultCoord=(centerCoord-neighboors[i])^(neighboors[i-1]-neighboors[i]);
	glNormal3f(resultCoord.getX(),resultCoord.getY(),resultCoord.getZ());
      }
      glVertex3f(lastValid.getX() , lastValid.getY() , lastValid.getZ());
      if (result==2) glEnd();
      if ((nbElements==0) && (result==2)) {
	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);
	GlLines::glDrawLine(centerCoord,lastValid,1.0f,0,centerColor,lastValidColor);
	glEnable(GL_LIGHTING);
	glEnable(GL_BLEND);
      }
    }
    if (result!=2) glEnd();
    else {
      glDisable(GL_BLEND);
      glDisable(GL_LIGHTING);
      GlLines::glDrawLine(centerCoord,lastTested,1.0f,0,centerColor,lastTestedColor);
      glEnable(GL_LIGHTING);
      glEnable(GL_BLEND);
    }
    delete itChild;


    //We make the BACK drawing now
    glFrontFace(GL_CCW);
    centerNormal*=-1;
    itChild=_superGraph->getOutNodes(n);
    itchild=itChild->next();
    lastTested=lastValid=elementLayout->getNodeValue(itchild);
    lastTestedColor=lastValidColor=elementColor->getNodeValue(itchild);
    result=2;
    nbElements=0;
    for (int i=0;itChild->hasNext();) {
      fanError.reset();
      fanError.addVertex(lastTested);
      if (result==2) {
	nbElements=0;
	glBegin(GL_TRIANGLE_FAN);
	//SetMaterial(centerColor);
	SetColor(centerColor);
	glNormal3f(centerNormal.getX(),centerNormal.getY(),centerNormal.getZ());
	glVertex3f(centerCoord.getX() , centerCoord.getY() , centerCoord.getZ());
	//	SetMaterial(lastTestedColor);
	SetColor(lastTestedColor);
	if (i+1<numberOfChildren) {
	  Coord resultCoord=(centerCoord-neighboors[i])^(neighboors[i+1]-neighboors[i]);
	  glNormal3f(resultCoord.getX(),resultCoord.getY(),resultCoord.getZ());
	}
	glVertex3f(lastTested.getX() , lastTested.getY() , lastTested.getZ());	
      }
      lastValid=lastTested;
      for (;itChild->hasNext();) {
	node itchild=itChild->next();
	i++;
	lastTested=elementLayout->getNodeValue(itchild);
	lastTestedColor=elementColor->getNodeValue(itchild);
	if ((result=fanError.addVertex(lastTested))!=0) break;
	lastValid=lastTested;
	lastValidColor=lastTestedColor;
	nbElements++;
      }
      //      SetMaterial(lastValidColor);
      SetColor(lastValidColor);

      if (i-1>0) {
	Coord resultCoord=(neighboors[i-1]-neighboors[i])^(centerCoord-neighboors[i]);
	glNormal3f(resultCoord.getX(),resultCoord.getY(),resultCoord.getZ());
      }

      glVertex3f(lastValid.getX() , lastValid.getY() , lastValid.getZ());
      if (result==2) glEnd();
    }
    if (result!=2) glEnd();
    delete itChild;
  }

  glFrontFace(GL_CCW);
  glDisable(GL_BLEND);
}
//====================================================
unsigned int GlGraph::drawFanNodes(unsigned int number, Iterator<node> *itN) {
  unsigned int tmp=number;
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  initProxies();
  while ((itN->hasNext()) && (number >0)) {
    number--;
    node itv=itN->next();
    drawFanNode(itv);
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
