#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qpopupmenu.h>

#include <tulip/SuperGraph.h>
#include <tulip/SelectionProxy.h>

#include "../include/tulip/ClusterTree.h"

using namespace std;

//=======================================================
struct ClusterListViewItem: public QListViewItem {
  SuperGraph *_superGraph;
  ClusterListViewItem(SuperGraph * superGraph, QListViewItem *item) : QListViewItem(item), _superGraph(superGraph) {}
  ClusterListViewItem(SuperGraph * superGraph, QListView *item) : QListViewItem(item), _superGraph(superGraph) {}
  SuperGraph * getSuperGraph() const {
    return _superGraph;
  }
};
//=======================================================
/* 
 *  Constructs a ClusterTree which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 */
ClusterTree::ClusterTree(SuperGraph *rootGraph , QWidget* parent,  const char* name, WFlags fl)
  : ClusterTreeData(parent, name, fl),
    _currentSuperGraph(rootGraph),
    _rootSuperGraph(rootGraph),
    removeOperation(false),
    contextMenu(NULL)
{
  setCaption(trUtf8("Cluster Tree"));
  connect(treeView, SIGNAL(currentChanged(QListViewItem*)), SLOT(changeSuperGraph(QListViewItem*)));
  connect(treeView, SIGNAL(contextMenuRequested ( QListViewItem *, const QPoint &, int )),
	  SLOT(rightButtonClusterTree( QListViewItem *, const QPoint &, int )));
  update();
}
//=======================================================
ClusterTree::ClusterTree(QWidget* parent, const char* name, WFlags fl) :
  ClusterTreeData(parent, name,fl),
  _currentSuperGraph(NULL),
  _rootSuperGraph(NULL),
  removeOperation(false),
  contextMenu(NULL) {
  setCaption(trUtf8("Cluster Tree"));
  connect(treeView, SIGNAL(currentChanged(QListViewItem*)), SLOT(changeSuperGraph(QListViewItem*)));
  connect(treeView, SIGNAL(contextMenuRequested ( QListViewItem *, const QPoint &, int )),
	  SLOT(rightButtonClusterTree( QListViewItem *, const QPoint &, int )));
}
//=======================================================
/*  
 *  Destroys the object and frees any allocated resources
 */
ClusterTree::~ClusterTree() { 
  if (contextMenu != NULL) delete contextMenu;
}
//=======================================================
void ClusterTree::setRootSuperGraph(SuperGraph *sg) {
  treeView->clear();
  if (sg != NULL) {
    _rootSuperGraph = _currentSuperGraph = sg->getRoot();
    update();
  }
  else 
    _rootSuperGraph = _currentSuperGraph = 0;
}
//=======================================================
SuperGraph* ClusterTree::getSuperGraph() const {
  return _currentSuperGraph;
}
//=======================================================
static QListViewItem *findItemBySuperGraph(QListViewItem *item, const int id) {
  QListViewItem *i = item;
  if (((ClusterListViewItem *) i)->getSuperGraph()->getId() == id)
    return i;
  for (i = i->firstChild(); i != NULL; i = i->nextSibling()) {
    QListViewItem *tmp = findItemBySuperGraph(i, id);
    if (tmp != NULL) return tmp;
  }
  if (item->nextSibling() != NULL)
    return findItemBySuperGraph(item->nextSibling(), id);
  return NULL;
}
//=======================================================
void ClusterTree::setSuperGraph(SuperGraph *graph) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (graph==0) return;
  if (_currentSuperGraph==0 || findItemBySuperGraph(treeView->firstChild(), graph->getId())==0) {
    _currentSuperGraph=graph;
    update();
  }
  currentSuperGraphChanged(graph);
  emit supergraphChanged(_currentSuperGraph);
}
//=======================================================
void ClusterTree::currentSuperGraphChanged(const SuperGraph *graph) {
#ifdef QT_DEBUG
  //  cerr << __PRETTY_FUNCTION__ << endl;
  //Q_CHECK_PTR(graph);
#endif
  QListViewItem *item = findItemBySuperGraph(treeView->firstChild(), graph->getId());
  if (item != NULL) {
    disconnect(treeView, SIGNAL(currentChanged(QListViewItem*)), this, SLOT(changeSuperGraph(QListViewItem*)));
    treeView->setCurrentItem(item);
    connect(treeView, SIGNAL(currentChanged(QListViewItem*)), this, SLOT(changeSuperGraph(QListViewItem*)));    
    _currentSuperGraph=((ClusterListViewItem *)item)->getSuperGraph();
  }
}
//=======================================================
void ClusterTree::update() 
{  //cerr << __PRETTY_FUNCTION__ << endl;
  treeView->clear();
  //  cerr <<  "clear" << endl;
  if (_currentSuperGraph != 0) {
    _rootSuperGraph = _currentSuperGraph->getRoot();
    buildTreeView(treeView, _rootSuperGraph);
    currentSuperGraphChanged(_currentSuperGraph);
  }
}
//=======================================================
//Building of the cluster tree view
void ClusterTree::buildTreeView(QListView *item, SuperGraph *p) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  QListViewItem *tmpItem = new ClusterListViewItem(p, item);
  tmpItem->setText(0, QString(p->getAttribute<string>("name").c_str()));
  tmpItem->setExpandable(true);
  item->setOpen(tmpItem,true);
  Iterator<SuperGraph *> *itS=p->getSubGraphs();
  while (itS->hasNext())
    buildTreeView(tmpItem,itS->next());
  delete itS;
}
//=======================================================
void ClusterTree::buildTreeView(QListViewItem *item, SuperGraph *p) {
  QListViewItem *tmpItem=new ClusterListViewItem(p,item);
  tmpItem->setText(0, QString(p->getAttribute<string>("name").c_str()));
  tmpItem->setExpandable(true);
  treeView->setOpen(tmpItem,true);
  Iterator<SuperGraph *> *itS=p->getSubGraphs();
  while (itS->hasNext())
    buildTreeView(tmpItem,itS->next());
  delete itS;
}
//=======================================================
//Cluster Tree Structure modification
void ClusterTree::contextRemoveCluster() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (_currentSuperGraph == _rootSuperGraph) {
    QMessageBox::critical( 0, "Tulip Hierarchy Editor Remove Failed", QString("You cannot remove the root graph"));
    return;
  } 
  Observable::holdObservers();
  removeOperation = true;
  emit aboutToRemoveView(_currentSuperGraph);
  SuperGraph *father=_currentSuperGraph->getFather();
  father->delSubGraph(_currentSuperGraph);
  _currentSuperGraph=father;
  update();
  currentSuperGraphChanged(father);
  removeOperation = false;
  Observable::unholdObservers();
  emit supergraphChanged(father);
}
//=======================================================
void ClusterTree::contextRemoveAllCluster() {
  if (_currentSuperGraph == _rootSuperGraph) {
    QMessageBox::critical( 0, "Tulip Cluster Tree Editor Remove Failed",QString("You cannot remove the root cluster"));
    return;
  }
  Observable::holdObservers();
  removeOperation = true;
  emit aboutToRemoveAllView(_currentSuperGraph);
  _currentSuperGraph->getFather()->delAllSubGraphs(_currentSuperGraph);
  _currentSuperGraph=_rootSuperGraph;
  update();
  removeOperation = false;
  Observable::unholdObservers();
}
//=======================================================
void ClusterTree::contextCloneCluster() {  
  if (_currentSuperGraph == _rootSuperGraph) {
    QMessageBox::critical( 0, "Tulip Cluster Tree Editor Clone Failed",QString("You cannot clone the root cluster"));
    return;
  }
  bool ok;
  QString text = QInputDialog::getText( "Cluster name" , "Please enter the cluster name" ,
                                        QLineEdit::Normal,QString::null, &ok, this);
  if (ok) {
    SuperGraph *tmp=_currentSuperGraph->getFather()->addSubGraph();
    tmp->setAttribute("name",string(text.latin1()));
    Iterator<node> *itN=_currentSuperGraph->getNodes();
    while (itN->hasNext())
      tmp->addNode(itN->next());
    delete itN;
    Iterator<edge> *itE=_currentSuperGraph->getEdges();
    while (itE->hasNext())
      tmp->addEdge(itE->next());
    delete itE;
    update();
  }
}
//=======================================================
void ClusterTree::contextCloneSubgraphCluster() {
  bool ok;
  QString text = QInputDialog::getText( "Cluster name" , "Please enter the cluster name" ,
                                        QLineEdit::Normal,QString::null, &ok, this);
  if (ok) {
    SelectionProxy *sel1 = _currentSuperGraph->getLocalProperty<SelectionProxy>("tmpselect");
    sel1->setAllNodeValue(true);
    sel1->setAllEdgeValue(true);
    SuperGraph *tmp=_currentSuperGraph->addSubGraph(sel1);
    tmp->setAttribute("name",string(text.latin1()));
    _currentSuperGraph->delLocalProperty("tmpselect");
    update();
  }
}
//=======================================================
void ClusterTree::contextMoveUpCluster() {
  std::cerr << "Not Implemented" << __PRETTY_FUNCTION__ << std::endl;
  //  _clusterTree->moveUp(_currentSubGraph);
  update();
}
//=======================================================
void ClusterTree::contextRenameCluster() {
  //  std::cerr << "Not Implemented" << __PRETTY_FUNCTION__ << std::endl;
  bool ok;
  QString text = QInputDialog::getText( trUtf8("Cluster Name") ,  trUtf8("Please enter the cluster name"),
					QLineEdit::Normal, _currentSuperGraph->getAttribute<string>("name").c_str(), &ok, this);
  if (ok) {
    _currentSuperGraph->setAttribute("name",string(text.latin1()));
  }
  update();
}
//=======================================================
void ClusterTree::setContextMenu(QPopupMenu *menu) {
  if (contextMenu != NULL) delete contextMenu;
  contextMenu = menu;
}
//=======================================================
QPopupMenu *ClusterTree::getContextMenu() const {
  return contextMenu;
}
//=======================================================
void ClusterTree::rightButtonClusterTree(QListViewItem *item, const QPoint &p, int c) {
  if (item == NULL) return;
  if (contextMenu == NULL) {
    contextMenu = new QPopupMenu(this, "cluster_tree_context_menu");
    contextMenu->insertItem(trUtf8("Remove"), this, SLOT(contextRemoveCluster()));
    contextMenu->insertItem(trUtf8("Remove all"), this, SLOT(contextRemoveAllCluster()));
    //    contextMenu->insertItem(trUtf8("Move up"), this, SLOT(contextMoveUpCluster()));
    contextMenu->insertItem(trUtf8("Clone"), this, SLOT(contextCloneCluster()));
    contextMenu->insertItem(trUtf8("SubGraph Clone"), this, SLOT(contextCloneSubgraphCluster()));
    contextMenu->insertItem(trUtf8("Rename"), this, SLOT(contextRenameCluster()));
  }
  contextMenu->exec(p);
}
//=======================================================
void ClusterTree::changeSuperGraph(QListViewItem *item,const QPoint &p, int i) {
  changeSuperGraph(item);
}
//**********************************************************************
///Make the current visible graph equal to the subgraph induce by the 
///Selected partition
void ClusterTree::changeSuperGraph(QListViewItem *item) {
#ifndef NDEBUG
  //  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  _currentSuperGraph=((ClusterListViewItem *)item)->getSuperGraph();
  emit supergraphChanged(_currentSuperGraph);
}
//=======================================================
