//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 26/09/2001
 This program is free software; you can redistribute it and/or modify  *
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifndef TULIP_PROPERTIESPROXY_H
#define TULIP_PROPERTIESPROXY_H

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif

#include "Observable.h"
#include "Types.h"
#include "Property.h"
#include "PropertyManager.h"
#include "MutableContainer.h"
#include "ReturnType.h"
#include "tulipconf.h"
#include "MethodFactory.h"
#include "TemplateFactory.h"



/**
 * \addtogroup properties
 */ 
/*@{*/
//=============================================================
class PProxy: public Observable {
public:
  virtual ~PProxy(){}
  virtual void erase(const node)=0;
  virtual void erase(const edge)=0;
  virtual void copy(const node, const node, PProxy *)=0;
  virtual void copy(const edge, const edge, PProxy *)=0;
  virtual PProxy* clonePrototype(SuperGraph *, std::string)=0;

  //=================================================================================
  // Returns a string describing the type of the proxy.
  // i.e. "metagraph", "metric", "layout", "string", "int", "color", "size", ...
  virtual std::string getTypename() = 0;
  static  std::string getTypename( PProxy * );

  // Untyped accessors
  virtual std::string getNodeDefaultStringValue() = 0;
  virtual std::string getEdgeDefaultStringValue() = 0;
  virtual std::string getNodeStringValue( const node n ) = 0;
  virtual std::string getEdgeStringValue( const edge e ) = 0;
  virtual bool		  setNodeStringValue( const node n, const std::string & v ) = 0;
  virtual bool		  setEdgeStringValue( const edge e, const std::string & v ) = 0;
  virtual bool		  setAllNodeStringValue( const std::string & v ) = 0;
  virtual bool		  setAllEdgeStringValue( const std::string & v ) = 0;
};
/**
 * This class is used to store a property of a graph.
 * A property is a couple of two functions:
 * - One from the set of nodes to a set of Tnode value
 * - One from the set of edges to a set of Tedge value
 *
 * A PropertyProxy can be connected or not to a Property.
 * In the first case it can be seen as buffer beetween the property and the user.
 * In the second case it is only a memory area for storing data.
 * A PropertyProxy is an observable, so it can be observed by others objects.
 */
template <class Tnode, class Tedge> class PropertyProxy:public PProxy {
  friend class SuperGraphView;


public:
  static TemplateFactory< PropertyFactory<Property<Tnode,Tedge> >, Property<Tnode,Tedge>, PropertyContext > factory;
  PropertyProxy(SuperGraph *);
  ~PropertyProxy() ;

  /**
   * Reset the proxy and notify the observers, if the property is
   * a data property (not connected to a plug-in) all elements receive
   * the default value of the property else the value will be recomputed
   * from the plug-in.
   */
  void reset();
  /**
   * Handler function for inherited PropertyProxy. This function
   * is called after a reset operation.
   */
  virtual void reset_handler(){};
  /**
   * Return the name of this property
   */
  std::string getName();
  /** 
   * Return the node default value of the property proxy
   * warnning: If the type is a pointer it can produce big memory
   * leak if the user doesn't manage the memory himself
   */
  typename Tnode::RealType getNodeDefaultValue();
  /** 
   * Return the edge default value of the property proxy
   * warnning: If the type is a pointer it can produce big memory
   * leak if the user doesn't manage the memory himself
   */
  typename Tedge::RealType getEdgeDefaultValue();
  /**
   * Return the value associated to the node n in this property.
   * If the value is already fixed it is done in constant time.
   * else if it is a computed property (plug-in), it calls the associated property
   * If there is no value and no algorithms it returns the default value
   * depending of the type.
   */
  typename ReturnType<typename Tnode::RealType>::Value getNodeValue(const node n );
  /**
   * Return the value associated to the node n in this property.
   * If the value is already fixed it is done in constant time.
   * else if it is a computed property (plug-in), it calls the associated property
   * If there is no value and no algorithms it returns the default value
   * depending of the type.
   */
  typename ReturnType<typename Tedge::RealType>::Value getEdgeValue(const edge e);
  /**
   * Set the value of a node n and notify the observers of a modification.
   * Warning : When using computed property (plug-in), if one sets the value of
   * a node n, the plug-in won't be call for n.
   */
  virtual void setNodeValue(const node n,const typename Tnode::RealType v);
  /**
   * Handler function for inherited PropertyProxy. This function
   * is called after a setNodeValue operation.
   */
  virtual void setNodeValue_handler(const node n){};
  //=============================================================
  /**
   * Set the value of an edge and notify the observers of a modification.
   * Warning : When using computed property (plug-in), if one sets the value of
   * a edge e, the plug-in won't be call for e.
   */
  void setEdgeValue(const edge e,const typename Tedge::RealType v);
  /**
   * Handler function for inherited PropertyProxy. This function
   * is called after a setEdgeValue operation.
   */
  virtual void setEdgeValue_handler(const edge e){};
  //=============================================================
  /**
   * Set the value of all nodes and notify the observers
   */
  void setAllNodeValue(const typename Tnode::RealType v);
  /**
   * Handler function for inherited PropertyProxy. This function
   * is called after a setAllNodeValue operation.
   */
  virtual void setAllNodeValue_handler(){};
  /**
   * Set the value of all edges and notify the observers
   */
  void setAllEdgeValue(const typename Tedge::RealType v);
  /**
   * Handler function for inherited PropertyProxy. This function
   * is called after a setAllEdgeValue operation.
   */
  virtual void setAllEdgeValue_handler(){};
  //=================================================================================
  void erase(const node n);
  //=================================================================================
  void erase(const edge e);
  //=================================================================================
  virtual PropertyProxy<Tnode,Tedge>& operator =(PropertyProxy<Tnode,Tedge> &proxy);
  //=================================================================================
  ///Enable to clone part of sub_class
  virtual void clone_handler(PropertyProxy<Tnode,Tedge> &){};
  //=================================================================================
  bool compute(const std::string &algorithm, std::string &msg, const PropertyContext&);

  //=================================================================================
  virtual std::string getTypename();

  // Untyped accessors
  virtual std::string getNodeDefaultStringValue();
  virtual std::string getEdgeDefaultStringValue();
  virtual std::string getNodeStringValue( const node n );
  virtual std::string getEdgeStringValue( const edge e );
  virtual bool		  setNodeStringValue( const node n, const std::string & v );
  virtual bool		  setEdgeStringValue( const edge e, const std::string & v );
  virtual bool		  setAllNodeStringValue( const std::string & v );
  virtual bool		  setAllEdgeStringValue( const std::string & v );

protected:
  MutableContainer<bool> nodeComputed;
  MutableContainer<typename Tnode::RealType> nodeProperties;
  MutableContainer<bool> edgeComputed;
  MutableContainer<typename Tedge::RealType> edgeProperties;
  typename Tnode::RealType nodeDefaultValue;
  typename Tedge::RealType edgeDefaultValue;
  Property< Tnode , Tedge > *currentProperty;
  SuperGraph *superGraph;
  //=============================================================
};
/*@}*/
#include "./cxx/PropertyProxy.cxx"
#endif

