//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 26/09/2001
 This program is free software; you can redistribute it and/or modify  *
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/

#ifndef NDEBUG
#include <iostream>
#endif

template <class Tnode, class Tedge, class TPROPERTY>
TemplateFactory<PropertyFactory<TPROPERTY >, TPROPERTY, PropertyContext > PropertyProxy<Tnode,Tedge,TPROPERTY>::factory;

template <class Tnode, class Tedge, class TPROPERTY>
PropertyProxy<Tnode,Tedge,TPROPERTY>::PropertyProxy(SuperGraph *graph):
  nodeDefaultValue(Tnode::defaultValue()),
  edgeDefaultValue(Tedge::defaultValue()),
  currentProperty(0),
  superGraph(graph),
  circularCall(false){
  nodeComputed.setAll(false);
  edgeComputed.setAll(false);
  nodeProperties.setAll(nodeDefaultValue);
  edgeProperties.setAll(edgeDefaultValue);
}
//=============================================================
static bool 
isAncestor(SuperGraph *g1, SuperGraph *g2) {
  if(g1 == g2->getRoot())
    return true;
  SuperGraph *currentGraph = g2;
  while(currentGraph->getFather() != currentGraph) {
    if(currentGraph == g1)
      return true;
    currentGraph = currentGraph->getFather();
  }
  return false;
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
bool PropertyProxy<Tnode,Tedge,TPROPERTY>::compute(const std::string &algorithm, std::string &msg, const PropertyContext& context) {

  if(!isAncestor(this->superGraph, context.superGraph))
    return false;

#ifndef NDEBUG
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
#endif
  if(circularCall) {
#ifndef NDEBUG
    std::cerr << "Circular call of " << __PRETTY_FUNCTION__ << " " << msg << std::endl;
#endif
    return false;
  }
  Observable::holdObservers();
  circularCall = true;
  PropertyContext tmpContext(context);
  tmpContext.propertyProxy = this;
  TPROPERTY *tmpProperty = factory.getObject(algorithm, tmpContext);
  bool result;
  if (tmpProperty != 0) {
    result = tmpProperty->check(msg);
    if (result) {
      reset();
      currentProperty = tmpProperty;
      currentProperty->run();
    }
  }
  else {
    msg = "No algorithm available with this name";
    result=false;
  }
  circularCall = false;
  notifyObservers();
  Observable::unholdObservers();
  return result;
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
PropertyProxy<Tnode,Tedge,TPROPERTY>::~PropertyProxy() {
  if (currentProperty!=0) delete currentProperty;
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
void PropertyProxy<Tnode,Tedge,TPROPERTY>::reset() {
  Observable::holdObservers();
  reset_handler();
  if(currentProperty != 0) {
    delete currentProperty;
    currentProperty = 0;
  }
  nodeComputed.setAll(false);
  edgeComputed.setAll(false);
  nodeProperties.setAll(nodeDefaultValue);
  edgeProperties.setAll(edgeDefaultValue);
  notifyObservers();
  Observable::unholdObservers();
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
typename Tnode::RealType PropertyProxy<Tnode,Tedge,TPROPERTY>::getNodeDefaultValue() { 
  return nodeDefaultValue; 
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
typename Tedge::RealType PropertyProxy<Tnode,Tedge,TPROPERTY>::getEdgeDefaultValue() { 
  return edgeDefaultValue; 
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
typename ReturnType<typename Tnode::RealType>::Value PropertyProxy<Tnode,Tedge,TPROPERTY>::getNodeValue(const node n ) {
  if ((currentProperty!=0) && !nodeComputed.get(n.id) ) {
    nodeProperties.set(n.id,currentProperty->getNodeValue(n));
    nodeComputed.set(n.id,true);
  }
  return nodeProperties.get(n.id);
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
typename ReturnType<typename Tedge::RealType>::Value PropertyProxy<Tnode,Tedge,TPROPERTY>::getEdgeValue(const edge e) {
  if ((currentProperty!=0) && !edgeComputed.get(e.id) ) {
    edgeProperties.set(e.id,currentProperty->getEdgeValue(e));
    edgeComputed.set(e.id,true);
  }
  return edgeProperties.get(e.id);
} 
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
void PropertyProxy<Tnode,Tedge,TPROPERTY>::setNodeValue(const node n,const typename Tnode::RealType v) {
  nodeProperties.set(n.id,v);
  if (currentProperty != 0)
    nodeComputed.set(n.id,true);
  setNodeValue_handler(n);
  notifyObservers();
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
void PropertyProxy<Tnode,Tedge,TPROPERTY>::setEdgeValue(const edge e,const typename Tedge::RealType v) {
  edgeProperties.set(e.id,v);
  if (currentProperty != 0)
    edgeComputed.set(e.id,true);
  setEdgeValue_handler(e);
  notifyObservers();
}
//=============================================================
template <class Tnode, class Tedge, class TPROPERTY>
void PropertyProxy<Tnode,Tedge,TPROPERTY>::setAllNodeValue(const typename Tnode::RealType v) {
  nodeDefaultValue=v;
  nodeProperties.setAll(v);
  nodeComputed.setAll(true);
  setAllNodeValue_handler();
  notifyObservers();
}
//============================================================
template <class Tnode, class Tedge, class TPROPERTY>
void PropertyProxy<Tnode,Tedge,TPROPERTY>::setAllEdgeValue(const typename Tedge::RealType v) {
  edgeDefaultValue=v;
  edgeProperties.setAll(v);
  edgeComputed.setAll(true);
  setAllEdgeValue_handler();
  notifyObservers();
}
//==============================================================
template <class Tnode, class Tedge, class TPROPERTY>
void PropertyProxy<Tnode,Tedge,TPROPERTY>::erase(const node n) {
  nodeProperties.set(n.id,nodeDefaultValue);
}
//=================================================================================
template <class Tnode, class Tedge, class TPROPERTY>
void PropertyProxy<Tnode,Tedge,TPROPERTY>::erase(const edge e) {
  edgeProperties.set(e.id,edgeDefaultValue);
}
//=================================================================================
template <class Tnode, class Tedge, class TPROPERTY>
PropertyProxy<Tnode,Tedge,TPROPERTY>& PropertyProxy<Tnode,Tedge,TPROPERTY>::operator =(PropertyProxy<Tnode,Tedge,TPROPERTY> &proxy) {
  if (this!= &proxy) {
    //=============================================================
    //The backup is necessary, if a proxy is a function which use the value of "*this"
    //Future implementation should take into account : recursive or not
    //It will enable to presserve the backup cost in a lot of case.
    //=============================================================
    if (superGraph==0) superGraph=proxy.superGraph;
    MutableContainer<typename Tnode::RealType> backupNode;
    MutableContainer<typename Tedge::RealType> backupEdge;
    backupNode.setAll(proxy.nodeDefaultValue);
    backupEdge.setAll(proxy.edgeDefaultValue);
    Iterator<node> *itN=superGraph->getNodes();
    while (itN->hasNext()) {
      node itn=itN->next();
      if (proxy.superGraph->isElement(itn))
	backupNode.set(itn.id,proxy.getNodeValue(itn));
    } delete itN;
    Iterator<edge> *itE=superGraph->getEdges();
    while (itE->hasNext()) {
      edge ite=itE->next();
      if (proxy.superGraph->isElement(ite))
	backupEdge.set(ite.id,proxy.getEdgeValue(ite));
    } delete itE;
    //==============================================================*
    if (superGraph==proxy.superGraph) {
      reset();
      setAllNodeValue(proxy.getNodeDefaultValue());
      setAllEdgeValue(proxy.getEdgeDefaultValue());
    }
    itN=superGraph->getNodes();
    while (itN->hasNext()) {
      node itn=itN->next();
      if (proxy.superGraph->isElement(itn))
	  setNodeValue(itn,backupNode.get(itn.id));
    } delete itN;
    itE=superGraph->getEdges();
    while (itE->hasNext()) {
      edge ite=itE->next();
      if (proxy.superGraph->isElement(ite))
	setEdgeValue(ite,backupEdge.get(ite.id));
    } delete itE;
    clone_handler(proxy);
  }
  return *this;
}
//==============================================================
template <class Tnode, class Tedge, class TPROPERTY>
std::string PropertyProxy<Tnode,Tedge,TPROPERTY>::getTypename() {
  return PProxy::getTypename( this );
}
//==============================================================
// Untyped accessors
template <class Tnode, class Tedge, class TPROPERTY>
std::string PropertyProxy<Tnode,Tedge,TPROPERTY>::getNodeDefaultStringValue() {
  typename Tnode::RealType v = getNodeDefaultValue();
  return Tnode::toString( v );
}
template <class Tnode, class Tedge, class TPROPERTY>
std::string PropertyProxy<Tnode,Tedge,TPROPERTY>::getEdgeDefaultStringValue() {
  typename Tedge::RealType v = getEdgeDefaultValue();
  return Tedge::toString( v );
}
template <class Tnode, class Tedge, class TPROPERTY>
std::string PropertyProxy<Tnode,Tedge,TPROPERTY>::getNodeStringValue( const node inN ) {
  typename Tnode::RealType v = getNodeValue( inN );
  return Tnode::toString( v );
}
template <class Tnode, class Tedge, class TPROPERTY> std::string
PropertyProxy<Tnode,Tedge,TPROPERTY>::getEdgeStringValue( const edge inE ) {
  typename Tedge::RealType v = getEdgeValue( inE );
  return Tedge::toString( v );
}
template <class Tnode, class Tedge, class TPROPERTY>
bool PropertyProxy<Tnode,Tedge,TPROPERTY>::setNodeStringValue( const node inN, const std::string & inV ) {
  typename Tnode::RealType v;
  if( !Tnode::fromString(v,inV) )
    return false;
  setNodeValue( inN, v );
  return true;
}
template <class Tnode, class Tedge, class TPROPERTY>
bool PropertyProxy<Tnode,Tedge,TPROPERTY>::setEdgeStringValue( const edge inE, const std::string & inV ) {
  typename Tedge::RealType v;
  if( !Tedge::fromString(v,inV) )
    return false;
  setEdgeValue( inE, v );
  return true;
}
template <class Tnode, class Tedge, class TPROPERTY>
bool PropertyProxy<Tnode,Tedge,TPROPERTY>::setAllNodeStringValue( const std::string & inV ) {
  typename Tnode::RealType v;
  if( !Tnode::fromString(v,inV) )
    return false;
  setAllNodeValue( v );
  return true;
}
template <class Tnode, class Tedge, class TPROPERTY> bool
PropertyProxy<Tnode,Tedge,TPROPERTY>::setAllEdgeStringValue( const std::string & inV ) {
  typename Tedge::RealType v;
  if( !Tedge::fromString(v,inV) )
    return false;
  setAllEdgeValue( v );
  return true;
}
