#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tulip/SuperGraphAbstract.h"
#include "tulip/SelectionProxy.h"
#include "tulip/StlIterator.h"
#include "tulip/StableIterator.h"
#include "tulip/SuperGraphView.h"
#include "tulip/ConcatIterator.h"

using namespace std;

//=========================================================================
SuperGraphAbstract::SuperGraphAbstract(SuperGraph *father):father(father){
  if (father==0) father=this;
  propertyContainer=new PropertyManagerImpl(this);
}
//=========================================================================
SuperGraphAbstract::~SuperGraphAbstract() {
  delete propertyContainer;
}
//=========================================================================
void SuperGraphAbstract::clear() {
  StableIterator<SuperGraph *> itS(getSubGraphs());
  while(itS.hasNext())
    delAllSubGraphs(itS.next());
  StableIterator<node> itN(getNodes());
  while(itN.hasNext())
    delNode(itN.next());
}
//=========================================================================
SuperGraph *SuperGraphAbstract::addSubGraph(SelectionProxy *selectionProxy){
  SuperGraph *tmp = new SuperGraphView(this, selectionProxy);
  subgraphs.insert(tmp);
  return tmp;
}
//=========================================================================
void SuperGraphAbstract::delSubGraph(SuperGraph *toRemove) {
  Iterator<SuperGraph *> *itS = toRemove->getSubGraphs();
  while (itS->hasNext()) {
    SuperGraph *tmp = itS->next();
    subgraphs.insert(tmp);
    tmp->setFather(this);
  } delete itS;
  subgraphs.erase(toRemove);
  delete toRemove;
}
//=========================================================================
void SuperGraphAbstract::delAllSubGraphs(SuperGraph * toRemove) {
  StableIterator<SuperGraph *> itS(toRemove->getSubGraphs());
  while (itS.hasNext())
    toRemove->delAllSubGraphs(itS.next());
  subgraphs.erase(toRemove);
  delete toRemove;
}
//=========================================================================
SuperGraph* SuperGraphAbstract::getFather()const {
  return father;
}
//=========================================================================
SuperGraph* SuperGraphAbstract::getRoot()const {
  SuperGraph *result=getFather();
  while (result!=result->getFather()) 
    result=result->getFather();
  return result;
}
//=========================================================================
void SuperGraphAbstract::setFather(SuperGraph *sg) {
  father=sg;
}
//=========================================================================
Iterator<SuperGraph *> * SuperGraphAbstract::getSubGraphs() const {
  return new StlIterator<SuperGraph *, set<SuperGraph*>::const_iterator >(subgraphs.begin(),subgraphs.end());
}
//=========================================================================
node SuperGraphAbstract::getOneNode() const {
  node result;
  Iterator<node> *it = getNodes();
  if (it->hasNext())
    result = it->next();
  delete it;
  return result;
}
//=========================================================================
edge SuperGraphAbstract::getOneEdge() const {
  edge result;
  Iterator<edge> *it=getEdges();
  if (it->hasNext())
    result = it->next();
  delete it;
  return result;
}
//=========================================================================
node SuperGraphAbstract::getInNode(const node n,unsigned int i)const {
  assert(i<=indeg(n) && i>0);
  Iterator<node>*itN=getInNodes(n);
  node result;
  for(unsigned int j=i+1;j>1;--j){
    result=itN->next();
  }delete itN;
  return result;
}
//=========================================================================
node SuperGraphAbstract::getOutNode(const node n,unsigned int i)const {
  assert(i<=outdeg(n) && i>0);
  Iterator<node>*itN=getOutNodes(n);
  node result;
  for(unsigned int j=i+1; j>1; --j) {
    result=itN->next();
  } delete itN;
  return result;
}
//=========================================================================
unsigned int SuperGraphAbstract::deg(const node n) const {
  unsigned int deg=0;
  Iterator<edge> *it=getInOutEdges(n);
  while (it->hasNext()) {
    it->next();
    ++deg;
  } delete it;
  return deg;
}
//=========================================================================
unsigned int SuperGraphAbstract::indeg(const node n) const {
  unsigned int deg=0;
  Iterator<edge> *it=getInEdges(n);
  while (it->hasNext()) {
    it->next();
    ++deg;
  } delete it;
  return deg;
}
//=========================================================================
unsigned int SuperGraphAbstract::outdeg(const node n) const {
  unsigned int deg=0;
  Iterator<edge> *it=getOutEdges(n);
  while(it->hasNext()) {
    it->next();
    ++deg;
  } delete it;
  return deg;
}
//=========================================================================
node SuperGraphAbstract::source(const edge e) const {
  return getFather()->source(e);
}
//=========================================================================
node SuperGraphAbstract::target(const edge e) const {
  return getFather()->target(e);
}
//=========================================================================
node SuperGraphAbstract::opposite(const edge e, const node n)const {
  return ( (source(e)==n) ? target(e):source(e) );
}
//=========================================================================
void SuperGraphAbstract::reverse(const edge e) {
  notifyReverseEdge(this,e);
  getFather()->reverse(e);
}
//=========================================================================
edge SuperGraphAbstract::existEdge(const node n1, const node n2)const {
  Iterator<edge> *it = getOutEdges(n1);
  while (it->hasNext()) {
    edge e(it->next());
    if (opposite(e, n1) == n2) {
      delete it;
      return e;
    }
  } delete it;
  return edge();
}
//=========================================================================
PropertyManager *SuperGraphAbstract::getPropertyManager() {
  return propertyContainer;
}
//=========================================================================
bool SuperGraphAbstract::existProperty(const std::string&name) {
  return propertyContainer->existProperty(name);
}
//=========================================================================
bool SuperGraphAbstract::existLocalProperty(const std::string&name) {
  return propertyContainer->existLocalProperty(name);
}
//=========================================================================
PProxy* SuperGraphAbstract::getProperty(const string &str) {
  return propertyContainer->getProperty(str);
}
//=========================================================================
void SuperGraphAbstract::delLocalProperty(const std::string&name) {
  propertyContainer->delLocalProperty(name);
}
//=========================================================================
Iterator<std::string>* SuperGraphAbstract::getLocalProperties() {
  return propertyContainer->getLocalProperties();
}
//=========================================================================
Iterator<std::string>* SuperGraphAbstract::getInheritedProperties() {
  return propertyContainer->getInheritedProperties();
}
//=========================================================================
Iterator<std::string>* SuperGraphAbstract::getProperties() {
  return new ConcatIterator<std::string> (propertyContainer->getLocalProperties(),propertyContainer->getInheritedProperties());
}
//=========================================================================
unsigned int SuperGraphAbstract::numberOfNodes() const {
  unsigned int result=0;
  Iterator<node> *it=getNodes();
  while(it->hasNext()) {
    it->next();
    result++;
  } delete it;
  return result;
}
//=========================================================================
unsigned int SuperGraphAbstract::numberOfEdges() const {
  unsigned int result=0;
  Iterator<edge> *it=getEdges();
  while (it->hasNext()) {
    it->next();
    result++;
  } delete it;
  return result;
}
//=========================================================================
