//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 20/08/2001
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tulip/SuperGraph.h"
#include "tulip/SuperGraphImpl.h"
#include "tulip/SuperGraphView.h"
#include "tulip/SelectionProxy.h"
#include "tulip/GraphIterator.h"

using namespace std;

/* 
 * function to test if an edge e exist in the adjacency of a node
 */
bool existEdgeE(SuperGraph *g, const node n1, const node n2, edge e) {
  Iterator<edge> *it = g->getOutEdges(n1);
  while (it->hasNext()) {
    edge e1(it->next());
    if (e == e1) {
      delete it;
      return true;
    }
  } delete it;
  return false;
}
/* 
 * function to test the integrity of the graph structure
 */
bool integrityTest(SuperGraph *graph) {
  Iterator<edge> *itE = graph->getEdges();
  set<edge> edgesTest;
  while(itE->hasNext()) {
    edge e = itE->next();
    edgesTest.insert(e);
    if (!existEdgeE(graph, graph->source(e), graph->target(e), e)) {
      cerr << "edge do not exist in neighbood" << endl;
      delete itE;
      return false;
    }
  }
  delete itE;
  Iterator<node> *itN = graph->getNodes();
  while(itN->hasNext()) {
    node n(itN->next());
    unsigned int degree = 0;
    Iterator<edge> *it = graph->getInOutEdges(n);
    while (it->hasNext()) {
      edge e = it->next();
      bool found = edgesTest.find(e)!=edgesTest.end();
      if (graph->isElement(e)!=found) {
	cerr << "isElment function not valid" << endl;
	delete it;
	delete itN;
	return false;
      }
      if (!graph->isElement(e)) {
	cerr << "Adjacency edges are not valid" << endl;
	delete it;
	delete itN;
	return false;
      }
      degree++;
    } delete it;
    if (graph->deg(n) != degree) {
      cerr << "degree failed" << endl;
      return false;
    }
  }delete itN;
  return true;
}
//----------------------------------------------------------------
SuperGraphImpl::SuperGraphImpl():SuperGraphAbstract(this) {
  nbNodes=0;
  nbEdges=0;
  outDegree.setAll(0);
}
//----------------------------------------------------------------
SuperGraphImpl::~SuperGraphImpl() {
  notifyDestroy(this);
  removeObservers();
  for (Nodes::iterator i=nodes.begin();i!=nodes.end();++i) {
    i->deallocateAll();
  }
}
void SuperGraphImpl::clear() {
  SuperGraphAbstract::clear();
  nbNodes=0;
  nbEdges=0;
  outDegree.setAll(0);
}
//----------------------------------------------------------------
bool SuperGraphImpl::isElement(const node n) {
  return !nodeIds.is_free(n.id);
}
//----------------------------------------------------------------
bool SuperGraphImpl::isElement(const edge e) {
  return !edgeIds.is_free(e.id);
}
//----------------------------------------------------------------
node SuperGraphImpl::addNode() {
  node newNode(nodeIds.get());
  outDegree.set(newNode.id, 0);
  while (nodes.size()<=newNode.id){
    nodes.push_back(EdgeContainer());
  }
  assert(nodes[newNode.id].empty());
  nbNodes++;
  notifyAddNode(this,newNode);
  return newNode;
}
//----------------------------------------------------------------
void SuperGraphImpl::addNode(const node n) {
  cerr << "SuperGraphImpl::addNode(const node n), The "
       << "current version of Tulip doesn't manage this"
       << "operation on Graph Implementation (coming soon)." << endl;
}
//----------------------------------------------------------------
edge SuperGraphImpl::addEdge(const node s,const node t) {
  assert(isElement(s) && isElement(t));
  pair< node , node > tmp(s,t);
  outDegree.set(s.id, outDegree.get(s.id) + 1);
  edge newEdge(edgeIds.get());
  while (edges.size()<=newEdge.id){
    edges.push_back(tmp);
  }
  edges[newEdge.id] = tmp;
  nodes[s.id].push_back(newEdge);
  nodes[t.id].push_back(newEdge);
  nbEdges++;
  notifyAddEdge(this,newEdge);
  return newEdge;
}
//----------------------------------------------------------------
void SuperGraphImpl::addEdge(const edge e) {
  cerr << "SuperGraphImpl::addEdge(const edge e), The"
       << " current version of Tulip doesn't manage this"
       << "operation on Graph Implementation (coming soon)." << endl;
}
//----------------------------------------------------------------
void SuperGraphImpl::delNode(const node n) {
  //Warning, the current implementation doesn't manage the 
  //updating of properties for upper_subgraph in the case of 
  //Super Graph Implementation
  notifyDelNode(this, n);
  externRemove(n);
  unsigned int loopcount = 0;
  for(EdgeContainer::iterator i=nodes[n.id].begin(); i!=nodes[n.id].end(); ++i) {
    notifyDelEdge(this,*i);
    externRemove(*i); //del in all subgraph/properties and free the id
    node s = opposite(*i, n);
    if (s!=n) {
      removeEdge(nodes[s.id], *i);
      if (s == edges[(*i).id].first)
	outDegree.set(s.id,outDegree.get(s.id)-1);
    }
    else {
      loopcount++;
    }
  }
  nbEdges += loopcount / 2;
  nodes[n.id].clear();
}
//----------------------------------------------------------------
void SuperGraphImpl::delEdge(const edge e) {
  notifyDelEdge(this,e);
  assert(existEdgeE(this, source(e),target(e), e));
  if (!isElement(e)) {
    return;
  }
  //Warning, the current implementation doesn't manage the updating of 
  //properties for upper_subgraph in the case of Super Graph Implementation
  node s = source(e);
  node t = target(e);
  outDegree.set(s.id, outDegree.get(s.id)-1);
  externRemove(e);
  removeEdge(nodes[s.id], e);
  removeEdge(nodes[t.id], e);
}
//----------------------------------------------------------------
void SuperGraphImpl::delAllNode(const node n){delNode(n);}
//----------------------------------------------------------------
void SuperGraphImpl::delAllEdge(const edge e){delEdge(e);}
//----------------------------------------------------------------
void SuperGraphImpl::setEdgeOrder(const node n,const vector<edge> &v ) {
  //  cerr << __PRETTY_FUNCTION__ << "not tested function" << endl;
  if (v.size()==0) return;
  MutableContainer<int> isEle;
  isEle.setAll(0); 
  for (vector<edge>::const_iterator it=v.begin();it!=v.end();++it) {
    isEle.set(it->id, isEle.get(it->id)+1);
  }
  vector<edge>::const_iterator it2=v.begin();
  EdgeContainer currentOrder = nodes[n.id];
  for (unsigned int i=0; i<currentOrder.size(); ++i) {
    if ( isEle.get(currentOrder[i].id)>0 ) {
      isEle.set(currentOrder[i].id, isEle.get(currentOrder[i].id) -1);
      currentOrder[i] = *it2;
      ++it2;
    }
  }
}
//----------------------------------------------------------------
void SuperGraphImpl::swapEdgeOrder(const node n,const edge e1 , const edge e2) {
  //  cerr << __PRETTY_FUNCTION__ << " not tested function" << endl;
  if (e1==e2) return;
  EdgeContainer adjacency=nodes[n.id];
  unsigned int e1Pos=UINT_MAX,e2Pos=UINT_MAX;
  for (unsigned int i=0;i<deg(n);++i) {
    if (adjacency[i]==e1) e1Pos=i;
    if (adjacency[i]==e2) e2Pos=i;
    if (e1Pos!=UINT_MAX && e2Pos!=UINT_MAX) break;
  }
  assert(e1Pos!=UINT_MAX && e2Pos!=UINT_MAX);
  adjacency[e1Pos]=e2;
  adjacency[e2Pos]=e1;
}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getNodes()const
{return (new xSGraphNodeIterator(this));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getInNodes(const node n)const
{return (new xInNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getOutNodes(const node n)const
{return (new xOutNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getInOutNodes(const node n)const
{return (new xInOutNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getEdges()const
{return (new xSGraphEdgeIterator(this));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getInEdges(const node n)const
{return (new xInEdgesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getOutEdges(const node n)const
{return (new xOutEdgesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getInOutEdges(const node n)const
{return (new xInOutEdgesIterator(this,n));}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::deg(const node n) const {return nodes[n.id].size();}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::indeg(const node n) const {return nodes[n.id].size()-outDegree.get(n.id);}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::outdeg(const node n) const {return outDegree.get(n.id);}
//----------------------------------------------------------------
node SuperGraphImpl::source(const edge e)const{return edges[e.id].first;}
//----------------------------------------------------------------
node SuperGraphImpl::target(const edge e)const{return edges[e.id].second;}
//----------------------------------------------------------------
void SuperGraphImpl::reverse(const edge e) {
  assert(isElement(e));
  node src = edges[e.id].first;
  node tgt = edges[e.id].second;
  edges[e.id].first  = tgt;
  edges[e.id].second = src;
  outDegree.set(src.id, outDegree.get(src.id) - 1);
  outDegree.set(tgt.id, outDegree.get(tgt.id) + 1);
  notifyReverseEdge(this,e);
}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::numberOfEdges()const{return nbEdges;}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::numberOfNodes()const{return nbNodes;}
//----------------------------------------------------------------
void SuperGraphImpl::removeEdge(EdgeContainer &c, const edge e) {
  bool copy = false;
  EdgeContainer::iterator previous;
  for (EdgeContainer::iterator i=c.begin(); i!=c.end(); ++i) {
    edge e1 = *i;
    if (copy)
      *previous = e1;
    previous = i;
    if (e1 == e)  
      copy = true;
  }
  c.pop_back();
}
//----------------------------------------------------------------
void SuperGraphImpl::externRemove(const edge e) {
  assert(isElement(e));
  Iterator<SuperGraph *>*itS=getSubGraphs();
  while (itS->hasNext()) {
    SuperGraph *subgraph = itS->next();
    assert(subgraph != this);
    subgraph->delEdge(e);
  } delete itS;
  getPropertyManager()->erase(e);
  edgeIds.free(e.id);
  nbEdges--;
}
//----------------------------------------------------------------
void SuperGraphImpl::externRemove(const node n) {
  assert(isElement(n));
  Iterator<SuperGraph *>*itS=getSubGraphs();
  while (itS->hasNext()) {
    SuperGraph *subgraph = itS->next();
    assert(subgraph != this);
    subgraph->delNode(n);
  } delete itS;
  getPropertyManager()->erase(n);
  nodeIds.free(n.id);
  nbNodes--;
}
//----------------------------------------------------------------
