#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <GL/glu.h>

#include <tulip/SelectionProxy.h>
#include <tulip/StringProxy.h>
#include <tulip/MetricProxy.h>
#include <tulip/MetaGraphProxy.h>
#include <tulip/IntProxy.h>
#include <tulip/LayoutProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/ColorsProxy.h>
#include <tulip/TlpTools.h>
#include "tulip/GlRenderer.h"
#include "tulip/GlLines.h"
#include "tulip/GlGraph.h"
#include "tulip/Glyph.h"
#include "tulip/TLPPixmapGlyph.h"
#include "tulip/TLPPixmapFont.h"
#include "tulip/TextRenderer.h"
#include "tulip/DrawingTools.h"
#include "tulip/Curves.h"
#include "tulip/GlTools.h"
#include <tulip/Matrix.h>

using namespace std;
using namespace tlp;


//====================================================
static const unsigned int TLPMAXDEPTH=10;
static const Color COLORSELECT= Color(255, 102, 255, 255);

namespace {
  void drawPoint(const Color &color, const Coord &pos, float size) {
    setColor(color);
    glPointSize(size);
    glBegin(GL_POINTS);
    glVertex3f(pos[0], pos[1], pos[2]);
    glEnd();
  }
  //===============================================
  //Modify bends, start and endN to prevent bad edge drawing
  //===============================================
  void cleanVertices(vector<Coord> &vertices, Coord &startN, Coord &endN) {
    assert(vertices.size()>0);
    vector<Coord> result;
    result.push_back(vertices[0]);
    for(unsigned int i=1; i<vertices.size(); ++i) 
      if ((vertices[i] - vertices[i-1]).norm()>1E-4) {
	result.push_back(vertices[i]);
      }
    if (result.size() < 2) { //only one valid point for a line
      vertices.clear();
      return;
    }
    //Adjust tangent direction
    if ((startN - result[0]).norm()<1E-4) {
      startN = result[0] - (result[1] - result[0]);
    }
    if ((endN - result.back()).norm()<1E-4) {
      endN = result.back() + result.back() - result[result.size()-2];
    }
    vertices = result;
  }
}
//====================================================
void GlGraph::drawNode(node n, unsigned int depth) {
  if (depth>TLPMAXDEPTH) return;

  const Coord &nodeCoord = elementLayout->getNodeValue(n);
  const Size &nodeSize = elementSize->getNodeValue(n);
  float lod = projectSize(nodeCoord, nodeSize, transformMatrix, viewportArray);
  if (lod < 0) return; //node is not visible
  glPassThrough(n.id); //id of the node for the feed back mode
  
  if (lod < 10.0) { //less than four pixel on screen, we use points instead of glyphs
    glDisable(GL_LIGHTING);
    const Color &nodeColor = elementColor->getNodeValue(n);
    if (!elementSelected->getNodeValue(n)) {
      drawPoint(nodeColor, nodeCoord, sqrt(lod));
    }
    else {
      glStencilFunc(GL_ALWAYS, 1, 0xFFFF);
      drawPoint(COLORSELECT, nodeCoord, sqrt(lod) + 1);
      glStencilFunc(GL_LEQUAL, 2, 0xFFFF);
    }
    glEnable(GL_LIGHTING);
  }
  else { //draw a glyph or make recursive call for meta nodes
    this->desactivateTexture();
    glPushMatrix();
    glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glRotatef(elementRotation->getNodeValue(n), 0., 0., 1.);
    glScalef(nodeSize[0], nodeSize[1], nodeSize[2]);
    if (elementMetaGraph->getNodeValue(n) == 0) {
      glyphs.get(elementShape->getNodeValue(n))->draw(n);
    }
    else {
      assert(glIsList(metaGraphDL));
      glCallList(metaGraphDL); //cube
      drawMetaNode(n,depth);
    }
    if (elementSelected->getNodeValue(n)) {
      glStencilFunc(GL_ALWAYS, 1, 0xFFFF);
      assert(glIsList(selectionDL));
      glCallList(selectionDL);
      glStencilFunc(GL_LEQUAL, 2, 0xFFFF);
    }
    glPopMatrix();
  }
}
//====================================================
unsigned int GlGraph::drawNodes(unsigned int number, Iterator<node> *itN, unsigned int depth) {
  if (!itN->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp = number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glMatrixMode(GL_MODELVIEW); 
  glEnable(GL_LIGHTING);
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_LINE_SMOOTH);
  glDisable(GL_COLOR_MATERIAL);
  glEnable(GL_STENCIL_TEST);
  glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);
  glStencilFunc(GL_LEQUAL, 2, 0xFFFF);
  initProxies();
  glPassThrough(-1.0);	//mark beginning of nodes drawing for feedback mode
  while ((itN->hasNext()) && (number >0)) {
    --number;
    drawNode(itN->next(), depth);
  }
  glPassThrough(-1.0);  //mark end of nodes drawing
  glPopAttrib();
  return tmp-number;
}
//====================================================
void GlGraph::drawEdge(edge ite) {
  const node source = _superGraph->source(ite);
  const node target = _superGraph->target(ite);
  Coord srcCoord = elementLayout->getNodeValue(source);
  Coord tgtCoord = elementLayout->getNodeValue(target);
  const LineType::RealType &bends = elementLayout->getEdgeValue(ite);
  
  if (bends.size()==0 && (srcCoord - tgtCoord).norm() < 1E-4) return; //a loop without bends

  //take source and target information for edge clipping
  const Size &srcSize  = elementSize->getNodeValue(source);
  const Size &tgtSize  = elementSize->getNodeValue(target);
  double srcRot = elementRotation->getNodeValue(source);
  double tgtRot = elementRotation->getNodeValue(target);

  Size edgeSize; //the edge radius and arrow radius
  if (isEdgeSizeInterpolate()) {
    edgeSize[0] = std::min(srcSize[0], srcSize[1]) / 16.;
    edgeSize[1] = std::min(tgtSize[0], tgtSize[1]) / 16.;
    if (_viewArrow) {
      edgeSize[2] = std::min(tgtSize[0], tgtSize[1]) / 4.;
    }
  }
  else {
    edgeSize = elementSize->getEdgeValue(ite);
  }

  Color srcCol,tgtCol;
  if (isEdgeColorInterpolate()) {
    srcCol = elementColor->getNodeValue(source);
    tgtCol = elementColor->getNodeValue(target);
  }
  else {
    srcCol = tgtCol = elementColor->getEdgeValue(ite);
  }

  //if first bend is inside the glyph (srcCoord is an anchor point it must be  replaced the node position)
  //if last bend is in the glyph (Coord is an anchor point it should replace the node position)
  //Be carefull if there is only one node.
  //  if (bends.size() > 0) {
  //    Coord firstBend = bends.front();
  //    Coord lastBend  = bends.back();
  //  }
  
  // set srcAnchor, tgtAnchor. tmpAnchor will be on the point just before tgtAnchor
  Coord srcAnchor, tgtAnchor, endLineAnchor, tmpAnchor;

  //compute anchor, (clip line with the glyph)
  float lod;
  lod = projectSize(srcCoord, srcSize, transformMatrix, viewportArray); 
  if (lod > 10) { //clip edges with the glyph
    int srcGlyphId = 1; //cube outlined
    if (elementMetaGraph->getNodeValue(source)==0)
      srcGlyphId = elementShape->getNodeValue(source);
    Glyph *sourceGlyph = glyphs.get(srcGlyphId);
    tmpAnchor = (bends.size() > 0) ? bends.front() : tgtCoord;
    srcAnchor = sourceGlyph->getAnchor(srcCoord, tmpAnchor, srcSize, srcRot);
  } else {
    srcAnchor = srcCoord;
  }

  //compute anchor, (clip line with the glyph)
  lod = projectSize(tgtCoord, tgtSize, transformMatrix, viewportArray); 
  if (lod > 10) { //clip edges with the glyph
    int tgtGlyphId = 1; //cube outlined
    if (elementMetaGraph->getNodeValue(target)==0)
      tgtGlyphId = elementShape->getNodeValue(target);
    Glyph *targetGlyph = glyphs.get(tgtGlyphId);
    //this time we don't take srcCoord but srcAnchor to be oriented to where the line comes from
    tmpAnchor = (bends.size() > 0) ? bends.back() : srcAnchor;
    tgtAnchor = targetGlyph->getAnchor(tgtCoord, tmpAnchor, tgtSize, tgtRot);
  } 
  else {
    tgtAnchor = tgtCoord;
  }

  //draw Arrow
  if (_viewArrow) {
    GLfloat matrix[16];
    float sizeT = edgeSize.getD();
    sizeT = std::min(sizeT, (float)(tmpAnchor- tgtAnchor).norm()/2.0f);
    makeArrowMatrix((GLfloat *)&matrix, tmpAnchor, tgtAnchor, 1.0, 1.0, 1.0);
    glEnable(GL_LIGHTING);
    glPushMatrix();
    glMultMatrixf((GLfloat *)&matrix);
    glScalef(sizeT, sizeT, sizeT);
    setMaterial(tgtCol);
    assert(glIsList(arrowDL));
    glCallList(arrowDL);
    glPopMatrix();
    if (elementSelected->getEdgeValue(ite)==true) {
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glPushMatrix();
      glMultMatrixf(matrix);
      glScalef(sizeT+0.1, sizeT+0.1, sizeT+0.1);
      setMaterial(COLORSELECT);
      assert(glIsList(arrowDL));
      glCallList(arrowDL);
      glPopMatrix();
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    }
    endLineAnchor = tmpAnchor - tgtAnchor;
    float nrm = endLineAnchor.norm();
    if( nrm > 0.00000001f ) {
      endLineAnchor *= sizeT / nrm;
      endLineAnchor += tgtAnchor;
    } else {
      endLineAnchor = tgtAnchor;
    }
    tgtCoord = tgtAnchor; //this defines in drawEdge the arrow head as being the final node
    glDisable(GL_LIGHTING);
  }
  else {
    endLineAnchor = tgtAnchor;
  }
  
  //draw Edges
  if (elementSelected->getEdgeValue(ite)) {
    Size selSize(edgeSize[0]+0.05, edgeSize[1]+0.05, edgeSize[2]);
    drawEdge(srcCoord, tgtCoord, srcAnchor, bends, endLineAnchor, COLORSELECT, COLORSELECT, selSize, elementShape->getEdgeValue(ite), true);
  } 
  else  
    drawEdge(srcCoord, tgtCoord, srcAnchor, bends, endLineAnchor, srcCol, tgtCol, edgeSize, elementShape->getEdgeValue(ite));
}
//====================================================
unsigned int GlGraph::drawEdges(unsigned int number,Iterator<edge> *itE, unsigned int depth) {
  if (!itE->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp = number;
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  glDisable(GL_LIGHTING);
  glEnable(GL_STENCIL_TEST);
  glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
  glStencilFunc(GL_EQUAL, 3, 0xFFFF);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glDisable(GL_CULL_FACE);
  //  glEnable(GL_POLYGON_SMOOTH);
  initProxies();
  while ((itE->hasNext()) && (number>0)) {
    number--;
    drawEdge(itE->next());
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================

//=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
#define L3D_BIT (1<<9)
void GlGraph::drawEdge(const Coord &startNode, const Coord &finalNode,
                       const Coord &startPoint,const LineType::RealType &bends, const Coord &endPoint,
                       const Color &startColor, const Color &endColor, const Size &size, int shape, const bool selected) {
  bool drawLine = true;
  bool drawPoly = false;
  tlp::curveVisibility(startPoint, bends, endPoint, size, drawPoly, drawLine, transformMatrix, viewportArray);
  if (!drawLine && !drawPoly) {
    return;
  }
  
  //================================
  if (_edge3D)
    shape |= L3D_BIT;
  if (shape & L3D_BIT) {
    if (shape < (L3D_BIT+16)) {
      glEnable(GL_LIGHTING);
      glEnable(GL_COLOR_MATERIAL);
      //      if (selected) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    }
    else shape  = shape & ~L3D_BIT; //no 3D model defined, falling back to Line model
  }
  //================================
  vector<Coord> tmp(bends.size()+2);  
  tmp[0] = startPoint;
  for(unsigned int i=0;i<bends.size();++i)
    tmp[i+1] = bends[i];
  tmp[tmp.size()-1] = endPoint;

  Coord srcDir(startNode);
  Coord tgtDir(finalNode);
  
  cleanVertices(tmp, srcDir, tgtDir);
  if (tmp.size()<2) {
    return;
  }
  Color srcCol(startColor), tgtCol(endColor);
  /*
    if (!selected) {
    srcCol[3] = 128; //alpha for transparency
    tgtCol[3] = 128; //alpha for transparency
  } else {
    srcCol[3] = 255; //alpha for transparency
    tgtCol[3] = 255; //alpha for transparency
  }
  */
  switch (shape) {
  case POLYLINESHAPE:
    if (drawPoly) {
      tlp::polyQuad(tmp, srcCol, tgtCol, size[0], size[1], srcDir, tgtDir);
    }
    if (drawLine) {
      tlp::polyLine(tmp,srcCol,tgtCol);
    }
    break;
  case BEZIERSHAPE:
    if (drawPoly)
      tlp::bezierQuad(tmp, srcCol, tgtCol, size[0], size[1], srcDir, tgtDir);
    if (drawLine) 
      tlp::bezierLine(tmp, srcCol, tgtCol);
    break;
  case SPLINESHAPE:
    if (drawPoly)
      tlp::splineQuad(tmp, srcCol, tgtCol, size[0], size[1], srcDir, tgtDir);
    if (drawLine) 
      tlp::splineLine(tmp, srcCol, tgtCol);
    break;
    //3D lines
  case L3D_BIT + POLYLINESHAPE: 
    if (drawPoly)
      GlLines::glDrawExtrusion(srcDir, tgtDir, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
			       GlLines::LINEAR, startColor, endColor); 
    if (drawLine) tlp::polyLine(tmp,srcCol,tgtCol);
    break;
  case L3D_BIT + BEZIERSHAPE: GlLines::glDrawExtrusion(srcDir, tgtDir, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
                                           GlLines::BEZIER, startColor, endColor); break;
  case L3D_BIT + SPLINESHAPE: GlLines::glDrawExtrusion(srcDir, tgtDir, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
                                           GlLines::SPLINE3, startColor, endColor); break;
  default:
    if (drawPoly)
      tlp::polyQuad(tmp, srcCol, tgtCol, size[0], size[1], srcDir, tgtDir);
    if (drawLine) 
      tlp::polyLine(tmp,srcCol,tgtCol);
    break;
  }
  
  if ((shape & L3D_BIT) && (shape < (L3D_BIT+16))) {
    //    if (selected) glPolygonMode(GL_FRONT, GL_FILL);
    glDisable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
  }
}
//====================================================
//====================================================
// Drawing of labels
// .................
//====================================================
unsigned int GlGraph::drawEdgeLabels(unsigned int number, Iterator<edge> *itE, bool mode, unsigned int depth) {
  if (!itE->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp = number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS );
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  if(mode)
    fontRenderer->setContext(fontsPath + "fontb.ttf", 14, 0, 0, 255);
  else	
    fontRenderer->setContext(fontsPath + "font.ttf", 12, 255, 255, 255);
  while ((itE->hasNext()) && (number >0)) {
    edge ite=itE->next();
    if (elementSelected->getEdgeValue(ite)==mode){
      number--;
      drawEdgeLabel(ite, mode);
    }
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
unsigned int GlGraph::drawNodeLabels(unsigned int number, Iterator<node> *itN, bool mode, unsigned int depth) {
  if (!itN->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp = number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS );
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  if(mode)
    fontRenderer->setContext(fontsPath + "fontb.ttf", 12, 0, 0, 255);
  else	
    fontRenderer->setContext(fontsPath + "font.ttf", 10, 255, 255, 255);
  while ((itN->hasNext()) && (number >0)) {
    node itv = itN->next();
    if (elementSelected->getNodeValue(itv)==mode){
      --number;
      drawNodeLabel(itv, mode, depth);
    }
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
void GlGraph::drawPixmapFont(const string &str, const Color &col,  const Coord &position, bool selected, float width) {
  int rastPos[4];
  float w_max = 300;
  float w,h;


  glRasterPos3f(position[0], position[1], position[2]);
  glGetIntegerv(GL_CURRENT_RASTER_POSITION,rastPos);
  if(occlusionTest.testRectangle(RectangleInt2D(rastPos[0] - labelsBorder - 5,
						rastPos[1] - labelsBorder - 5,
						rastPos[0] + labelsBorder + 5,
						rastPos[1] + labelsBorder + 5)))
    return;

  fontRenderer->setMode(TLP_TLPPIXMAP);
  fontRenderer->setString(str, VERBATIM);
  glColor3ub(col[0], col[1], col[2]);
  fontRenderer->setColor(col[0], col[1], col[2]);
  //  w_max = width;
  fontRenderer->getBoundingBox(w_max, h, w);
  if(!occlusionTest.addRectangle(RectangleInt2D(rastPos[0]-(int)(w/2.0) - labelsBorder,
						rastPos[1]-(int)(h/2.0) - labelsBorder,
						rastPos[0]+(int)(w/2.0) + labelsBorder,
						rastPos[1]+(int)(h/2.0) + labelsBorder))) {
    fontRenderer->draw(w, w);
  }
}

//====================================================
void GlGraph::drawEdgeLabel(edge e, bool mode) {
  string tmp=elementLabel->getEdgeValue(e);
  if (tmp.length()==0) return;
  const Coord & srcCoord = elementLayout->getNodeValue(_superGraph->source(e));
  const Coord & tgtCoord = elementLayout->getNodeValue(_superGraph->target(e));
  const LineType::RealType &bends = elementLayout->getEdgeValue(e);
  Coord position;
  if (bends.empty()) {
    position = (srcCoord+tgtCoord) / 2.0;
  }
  else {
    if (bends.size()%2 == 0)
      position = (bends[bends.size()/2-1]+bends[bends.size()/2]) / 2.0;
    else
      position = bends[bends.size()/2];
  }

  Color fontColor;
  if (elementSelected->getEdgeValue(e))
    fontColor.set(255,0,0,255);
  else {
    fontColor = elementLabelColor->getEdgeValue(e);
  }
  drawPixmapFont(tmp, fontColor, position, elementSelected->getEdgeValue(e), 50);
}
//====================================================
void GlGraph::drawNodeLabel(node n, bool mode, unsigned int depth) {
  const Coord &nodeCoord = elementLayout->getNodeValue(n);
  const Size  &nodeSize  = elementSize->getNodeValue(n);
  if (_viewMetaLabel && elementMetaGraph->getNodeValue(n)!=0) {
    glPushMatrix();
    glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glRotatef(elementRotation->getNodeValue(n), 0., 0., 1.);
    glScalef(nodeSize[0], nodeSize[1], nodeSize[2]);
    drawMetaNode(n, depth, true, mode);
    glPopMatrix();
  }
  const string &tmp = elementLabel->getNodeValue(n);
  if (tmp.length() < 1) {
    return;
  }
  Color fontColor;
  if (mode)
    fontColor.set(255,0,0,255);
  else {
    fontColor = elementLabelColor->getNodeValue(n);
  }

  float w_max = 300;
  float w,h;
  float div_w, div_h;
  switch(_FontsType){
  case 0:
    if (projectSize(nodeCoord,nodeSize, transformMatrix, viewportArray) < 8.0) return;

    fontRenderer->setMode(TLP_POLYGON);
    fontRenderer->setString(tmp, VERBATIM);
    fontRenderer->setColor(fontColor[0], fontColor[1], fontColor[2]);
    //      w_max = nodeSize.getW()*50.0;
    fontRenderer->getBoundingBox(w_max, h, w);
    glPushMatrix();
    glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glRotatef(elementRotation->getNodeValue(n), 0., 0., 1.);
    div_w = nodeSize.getW()/w;
    div_h = nodeSize.getH()/h;
    if(div_h > div_w) 
      glScalef(-div_w, -div_w, 1);
    else 
      glScalef(-div_h, -div_h, 1);  
    fontRenderer->draw(w,w);
    glPopMatrix();
    break;
  case 1:
    drawPixmapFont(tmp, fontColor, nodeCoord, elementSelected->getNodeValue(n), nodeSize.getW());
    break;
  case 2:
    if (projectSize(nodeCoord,nodeSize, transformMatrix, viewportArray) < 8.0) return;

    fontRenderer->setMode(TLP_TEXTURE);
    fontRenderer->setString(tmp, VERBATIM);
    fontRenderer->setColor(fontColor[0], fontColor[1], fontColor[2]);
    //      w_max = nodeSize.getW();
    fontRenderer->getBoundingBox(w_max, h, w);
    glPushMatrix();
    glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glRotatef(elementRotation->getNodeValue(n), 0., 0., 1.);
    div_w = nodeSize.getW()/w;
    div_h = nodeSize.getH()/h;
    if(div_h > div_w) 
      glScalef(-div_w, -div_w, 1);
    else 
      glScalef(-div_h, -div_h, 1);
    glEnable( GL_TEXTURE_2D);
    glBlendFunc(GL_ONE_MINUS_DST_COLOR,GL_ONE_MINUS_SRC_COLOR);
    fontRenderer->draw(w, w);
    glDisable( GL_TEXTURE_2D);
    glPopMatrix();
    break;
  default:
    cerr << "GlGraph::DrawNodes unknown fonts" << endl;
    break;
  }
}
//====================================================
void GlGraph::drawMetaNode(node n,unsigned int depth, bool labels, bool selectedLabels) {
  SuperGraph *sp = _superGraph;
  _superGraph = elementMetaGraph->getNodeValue(n);
  string layoutNameBackup(layoutName);
  layoutName = "viewLayout";
  initProxies();
  pair<Coord, Coord> bboxes = tlp::computeBoundingBox(_superGraph, elementLayout, elementSize, elementRotation);
  //  cerr << bboxes.first << "/" << bboxes.second << endl;
  Coord maxC = bboxes.first;
  Coord minC = bboxes.second;
  double saveMatrix[16];
  for (unsigned int i=0;i<16;++i) saveMatrix[i] = modelviewMatrix[i];
  Coord translate= (maxC + minC)/ (-2.0);
  double dept  = maxC[2] - minC[2];
  double width  = maxC[0] - minC[0];
  double height = maxC[1] - minC[1];
  if (width<0.0001) width=1;
  if (height<0.0001) height=1;
  if (dept<0.0001) dept=1;
  glPushMatrix();
  glScalef(1.0/width, 1.0/height, 1.0/dept);
  glTranslatef(translate[0],translate[1],translate[2]);
  glGetDoublev (GL_MODELVIEW_MATRIX, modelviewMatrix);
  tlp::multMatrix(projectionMatrix, modelviewMatrix, transformMatrix);

  Iterator<node> *itN=_superGraph->getNodes();
  if (!labels)
    drawNodes(_superGraph->numberOfNodes(), itN , depth+1);
  else
    drawNodeLabels(_superGraph->numberOfNodes(), itN , selectedLabels, depth+1);
  delete itN;
  if (_displayEdges) {
    Iterator<edge> *itE=_superGraph->getEdges();
    if (!labels)
      drawEdges(_superGraph->numberOfEdges(), itE, depth+1);
    else
      drawEdgeLabels(_superGraph->numberOfEdges(), itE , selectedLabels, depth+1);
    delete itE;
  }
  glPopMatrix();
  _superGraph=sp;
  layoutName = layoutNameBackup;
  initProxies();
  for (unsigned int i=0;i<16;++i) modelviewMatrix[i]=saveMatrix[i];
  tlp::multMatrix(projectionMatrix, modelviewMatrix, transformMatrix);
}
