#if (QT_REL == 3)
#include <qevent.h>
#include <qtimer.h>
#include <qimage.h>
#else
#ifdef  _WIN32
// compilation pb workaround
#include <windows.h>
#endif
#include <QtCore/qtimer.h>
#include <QtGui/qevent.h>
#include <QtGui/qimage.h>
#include <QtGui/qtooltip.h>
#include "tulip/Qt3ForTulip.h"
#endif

#include "tulip/GlGraphWidget.h"

#include <tulip/SuperGraph.h>
#include <tulip/Iterator.h>
#include <tulip/SelectionProxy.h>
#include <tulip/LayoutProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/GlGraph.h>

#include "tulip/MouseInterface.h"

#include <tulip/GlHudRect4.h>
#include <tulip/GlHudRect.h>


using namespace std;
using namespace tlp;

QGLFormat GlInit() {
  QGLFormat tmpFormat;
  tmpFormat.setDirectRendering(true);
  tmpFormat.setDoubleBuffer(true);
  tmpFormat.setAccum(false);
  tmpFormat.setStencil(true);
  tmpFormat.setOverlay(false);
  tmpFormat.setDepth(true);
  tmpFormat.setRgba(true);
  tmpFormat.setAlpha(true);
  return tmpFormat;
}

//==================================================
GlGraphWidget::GlGraphWidget(QWidget *parent, const char *name):
  QGLWidget(GlInit(),parent),
  drawTimer(new QTimer(this)),
  mouse(0) {
#if (QT_REL == 3)
  setName(name);
#else
  setObjectName(name);
#endif

  //  cerr << __PRETTY_FUNCTION__ << endl;
  setAutoBufferSwap(false);

  TRACE_EXEC();
  makeCurrent();
  connect(drawTimer, SIGNAL(timeout()), this, SLOT(drawGraph()));
  setFocusPolicy(StrongFocus);

  composite = new GlADComposite();
}
//==================================================
GlGraphWidget::GlGraphWidget(const GlGraphWidget &w, QWidget *parent, const char *name):
  QGLWidget(GlInit(),parent),
  GlGraph(w),
  drawTimer(new QTimer(this)),
  mouse(0) {
#if (QT_REL == 3)
  setName(name);
#else
  setObjectName(name);
#endif

  TRACE_EXEC();
  makeCurrent();
  connect(drawTimer, SIGNAL(timeout()), this, SLOT(drawGraph()));
  setFocusPolicy(StrongFocus);

  composite = new GlADComposite();
}
//==================================================
GlGraphWidget::~GlGraphWidget() {
  delete drawTimer;
  if (composite != 0 ) {
    composite->reset(false);
    delete composite;
  }
  if (mouse != 0)
    delete mouse;
}
//==================================================
void GlGraphWidget::closeEvent(QCloseEvent *e) {
  emit closed(this);
}
//==================================================
void GlGraphWidget::makeCurrent() {
  //  cerr << __PRETTY_FUNCTION__ << " (" << (int)this << ")" << endl;
  assert(context()->isValid());
  QGLWidget::makeCurrent();
}
//==================================================
void GlGraphWidget::updateGL() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  QGLWidget::updateGL();
}
//==================================================
void GlGraphWidget::initializeGL() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  GlGraph::initializeGL();
}
//==================================================
void GlGraphWidget::redraw() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  draw();
}
//==================================================
bool GlGraphWidget::timerIsActive() {
  return drawTimer->isActive();
}
//==================================================
int GlGraphWidget::timerStart(int msec, bool sshot) {
  return drawTimer->start(msec, sshot);
}
//==================================================
void GlGraphWidget::timerStop() {
  drawTimer->stop();
}
//==================================================
void GlGraphWidget::mPaint() {
  makeCurrent();
  if (composite!=0) composite->draw(this);
  if (mouse != NULL) mouse->mPaint(this);
}
//==================================================
void GlGraphWidget::clickAt(int x, int y) {
  node tmpNode;
  edge tmpEdge;
  ElementType type;  
  if (doSelect(x, y, type, tmpNode, tmpEdge)) {
    switch(type) {
    case NODE: emit nodeClicked(getSuperGraph(), tmpNode); break;
    case EDGE: emit edgeClicked(getSuperGraph(), tmpEdge); break;
    }
  }
}
//==================================================
void GlGraphWidget::clickAt(const QPoint &pos) { 
  clickAt(pos.x(), pos.y()); 
}
//==================================================
void GlGraphWidget::setSuperGraph(SuperGraph *sg) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  TRACE_EXEC();
  GlGraph::setSuperGraph(sg);
  centerScene();
  toUpdate=true;
}
//==================================================
MouseInterface *GlGraphWidget::getMouse() const {
  return mouse;
}
//==================================================
void GlGraphWidget::setMouse(MouseInterface *m) {
  if (mouse != 0) delete mouse;
  mouse = m->clone();
}
//==================================================
//QGLWidget
//==================================================
QImage GlGraphWidget::grabFrameBuffer(bool withAlpha) {
  bool incState = isIncrementalRendering();
  setIncrementalRendering(false);
  this->updateGL();
  QImage img = QGLWidget::grabFrameBuffer(withAlpha);
  setIncrementalRendering(incState);
  return img;
}
//==================================================
//QGLWidget slots
//==================================================
void GlGraphWidget::paintGL() {
  TRACE_EXEC();
  makeCurrent();
  draw();
  if (!isIncrementalRendering())
     swapBuffers();
  else
    drawGraph();
}
//==================================================
void GlGraphWidget::resizeGL(int w, int h) {
  TRACE_EXEC();
  changeViewport(0,0,w,h);
}
//==================================================
void GlGraphWidget::drawGraph() {
  TRACE_EXEC();
  makeCurrent();
  if (drawPart()) timerStop();
  swapBuffers();
}
//=====================================================
// Events 
//=====================================================
void GlGraphWidget::mousePressEvent(QMouseEvent *e) {
  if (mouse != NULL) 
    mouse->mPressEvent(this, e);
  else {
    if (e->button()==Qt::LeftButton) {
      node tmpNode;
      edge tmpEdge;
      ElementType type;
      if (doSelect(e->x(), e->y(), type, tmpNode,tmpEdge)) {
	switch(type) {
	case NODE: emit nodeClicked(getSuperGraph(), tmpNode); break;
	case EDGE: emit edgeClicked(getSuperGraph(), tmpEdge); break;
	}
      }
    }
  }
}
//================================================
void GlGraphWidget::mouseMoveEvent(QMouseEvent *e) {
  if (mouse != NULL) mouse->mMoveEvent(this, e);
}

void GlGraphWidget::mouseReleaseEvent(QMouseEvent *e) {
  if (mouse != NULL) mouse->mReleaseEvent(this ,  e);
}

void GlGraphWidget::keyPressEvent(QKeyEvent *e) {
  if (mouse != NULL) mouse->keyPressEvent(this, e);
}

void GlGraphWidget::keyReleaseEvent(QKeyEvent *e) {
  if (mouse != NULL) mouse->keyReleaseEvent(this, e);
}

void GlGraphWidget::wheelEvent(QWheelEvent *e) {
  if (mouse != NULL) mouse->wheelEvent(this, e);
}

#if (QT_REL == 4)
// With Qt4 software/src/tulip/ElementTooltipInfo.cpp
// is no longer needed; the tooltip implementation must take place
// in the event() method inherited from QWidget.
bool GlGraphWidget::event(QEvent *e) {
  if (e->type() == QEvent::ToolTip) {
    node tmpNode;
    edge tmpEdge;
    ElementType type;
    QString tmp;
    QHelpEvent *he = static_cast<QHelpEvent *>(e);
    if (doSelect(he->pos().x(), he->pos().y(), type, tmpNode, tmpEdge)) {
      switch(type) {
      case NODE:
	QToolTip::showText(he->globalPos(),
			   QString("node: ") + tmp.setNum(tmpNode.id));
	break;
      case EDGE: 
	QToolTip::showText(he->globalPos(),
			   QString("edge: ") + tmp.setNum(tmpEdge.id));
	break;
      }
    }
  }
  return QWidget::event(e);
}
#endif

void GlGraphWidget::addGlAugmentedDisplay(GlAugmentedDisplay *augmentedDisplay, const char* key) {
  composite->addGlAugmentedDisplay(augmentedDisplay, key);
}

void GlGraphWidget::removeGlAugmentedDisplay(GlAugmentedDisplay *augmentedDisplay) {
  composite->deleteGlAugmentedDisplay(augmentedDisplay);
}

void GlGraphWidget::removeGlAugmentedDisplay(const char* key) {
  composite->deleteGlAugmentedDisplay(key);
}

GlAugmentedDisplay* GlGraphWidget::findGlAugmentedDisplay(const char* key) {
  return composite->findGlAugmentedDisplay(key);
}
