//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <sstream>
#include <GL/gl.h>
#include <GL/glu.h>

#include <tulip/BooleanProperty.h>
#include <tulip/StringProperty.h>
#include <tulip/DoubleProperty.h>
#include <tulip/GraphProperty.h>
#include <tulip/IntegerProperty.h>
#include <tulip/LayoutProperty.h>
#include <tulip/SizeProperty.h>
#include <tulip/ForEach.h>
#include "tulip/GlLines.h"
#include "tulip/GlGraph.h"
#include "tulip/Glyph.h"
#include "tulip/GlTools.h"

using namespace std;
using namespace tlp;

//=====================================================
//this is used to sort node or edges from a select buffer according to their depth
template<class Element>
struct lessElementZ : public std::binary_function<Element, Element, bool> {
  GLuint (*v)[4];
  MutableContainer<GLint> *mapping;
  bool operator()(Element e1, Element e2) {
    GLuint z1, z2;
    z1 = v[mapping->get(e1.id)][1]/2 + v[mapping->get(e1.id)][2]/2;
    z2 = v[mapping->get(e2.id)][1]/2 + v[mapping->get(e2.id)][2]/2;
    return z1 < z2;
  }
};
//=====================================================
//Selection d'objet dans le monde 3D
//=====================================================
void GlGraph::initDoSelect(GLint x, GLint y, 
			   GLint w,GLint h, 
			   const unsigned int nbPickableElements) {
  assert(nbPickableElements > 0);
  glPushAttrib(GL_ALL_ATTRIB_BITS); //save previous attributes
  glPushClientAttrib(GL_CLIENT_ALL_ATTRIB_BITS); //save previous attributes
  
  //Allocate memory to store the result oh the selection
  selectBuf = new GLuint[nbPickableElements][4];
  glSelectBuffer(nbPickableElements*4 , (GLuint *)selectBuf);
  //Activate Open Gl Selection mode
  glRenderMode(GL_SELECT);
  glInitNames();
  glPushName(INT_MAX);
  Vector<int, 4> viewport = _renderingParameters.getViewport();

  glMatrixMode(GL_PROJECTION);
  glPushMatrix(); //save previous projection matrix

  //initialize picking matrix
  glLoadIdentity();
  x += w/2;
  y =  viewport[3] - (y + h/2);
  gluPickMatrix(x, y, w, h, (GLint *)&viewport);
  initProjection(false);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix(); //save previous model view matrix
  initModelView();

  //update modelview matrix for clipping test
  glGetFloatv (GL_MODELVIEW_MATRIX, (GLfloat *)&modelviewMatrix);
  //update projection matrix for clipping test
  glGetFloatv (GL_PROJECTION_MATRIX, (GLfloat *)&projectionMatrix);
  transformMatrix = modelviewMatrix * projectionMatrix;
  
  glPolygonMode(GL_FRONT, GL_FILL);
  glDisable(GL_LIGHTING);
  glDisable(GL_BLEND);
  glDisable(GL_STENCIL_TEST);
  tlp::glTest(__PRETTY_FUNCTION__);
}
//=====================================================
void GlGraph::endSelect() {
  delete [] selectBuf;
  glPopClientAttrib();
  glPopAttrib();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
  glGetFloatv (GL_MODELVIEW_MATRIX, (GLfloat *)&modelviewMatrix);
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glGetFloatv (GL_PROJECTION_MATRIX, (GLfloat *)&projectionMatrix);
  transformMatrix = modelviewMatrix * projectionMatrix;
}
//=====================================================
bool GlGraph::doNodeSelect(const int x, const int y, 
			   const int width , const int height, 
			   vector<node> &vNode , bool ordered) {
  if (_graph==0 || _graph->numberOfNodes() == 0) return false;
  GLint hits;
  initDoSelect(x , y, width, height, _graph->numberOfNodes());
  makeNodeSelect(0);
  glFlush();
  hits = glRenderMode(GL_RENDER);
  if (hits <= 0) {
    endSelect(); 
    tlp::glTest(__PRETTY_FUNCTION__);
    return false;
  }
  MutableContainer<GLint> nodeToSelectBuffer;
  while(hits>0) {
    vNode.push_back(node(selectBuf[hits-1][3]));
    nodeToSelectBuffer.set(selectBuf[hits-1][3], hits -1);
    --hits;
  }
  if (ordered) {
    struct lessElementZ<node> comp;
    comp.v = selectBuf;
    comp.mapping = &nodeToSelectBuffer;
    sort(vNode.begin(), vNode.end(), comp);
  }
  endSelect();
  tlp::glTest(__PRETTY_FUNCTION__);
  return true;
}
//=====================================================
bool GlGraph::doEdgeSelect(const int x, const int y, 
			   const int width, const int height, 
			   vector<edge> &vEdge, bool ordered) {
  if (_graph==0 || _graph->numberOfEdges() == 0) return false;
  GLint hits;
  initDoSelect(x , y, width, height, _graph->numberOfEdges());
  makeEdgeSelect(0);
  glFlush();
  hits = glRenderMode(GL_RENDER);
  if (hits <= 0) {
    endSelect(); 
    tlp::glTest(__PRETTY_FUNCTION__);
    return false;
  }
  MutableContainer<GLint> edgeToSelectBuffer;
  while(hits>0) {
    vEdge.push_back(edge(selectBuf[hits-1][3]));
    edgeToSelectBuffer.set(selectBuf[hits-1][3], hits -1);
    --hits;
  }
  if (ordered) {
    struct lessElementZ<edge> comp;
    comp.v = selectBuf;
    comp.mapping = &edgeToSelectBuffer;
    sort(vEdge.begin(), vEdge.end(), comp);
  }
  endSelect();
  tlp::glTest(__PRETTY_FUNCTION__);
  return true;
}
//=====================================================
void GlGraph::doSelect(const int x, const int y, 
		       const int width ,const int height,
                       vector<node> &sNode, vector<edge> &sEdge) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << " x:" << x << ", y:" <<y <<", wi:"<<width<<", height:" << height << endl;
#endif

  int w = std::max(1 , width);
  int h = std::max(1 , height);
  doNodeSelect(x, y, w, h, sNode, false);
  doEdgeSelect(x, y, w, h, sEdge, false);

  tlp::glTest(__PRETTY_FUNCTION__);
  //  cerr << "bool GlGraph::doSelect End" << endl;
}
//=====================================================
bool GlGraph::doSelect(const int x, const int y, tlp::ElementType &type ,node &n, edge &e ) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  bool result;
  vector<node> tmpSetNode;
  //Check that line
  result = doNodeSelect(x-3, y-3, 6, 6, tmpSetNode);
  if (result){
    n = (*(tmpSetNode.begin()));
    type = tlp::NODE;
  }
  else { 
    type = tlp::EDGE;
    vector<edge> tmpSetEdge;
    result = doEdgeSelect(x-3, y-3, 6, 6, tmpSetEdge);
    if (result) 
      e = (*(tmpSetEdge.begin()));
  }
  tlp::glTest(__PRETTY_FUNCTION__);
  return result;
  //  cerr << "bool GlGraph::doSelect End" << endl;
}
//====================================lines================
//Construction des objets avec affectation de nom
//======================================================================
//Placement des sommets du graphe �l'index i=startIndex..startIndex+n-1
void GlGraph::makeNodeSelect(int startIndex) {
  glMatrixMode(GL_MODELVIEW);
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  node itv;
  forEach(itv, _graph->getNodes()) {
    const Size &nSize = elementSize->getNodeValue(itv);
    if( nSize == Size(0,0,0) )	continue;
    const Coord &tmpCoord = elementLayout->getNodeValue(itv);	  
    float lod = projectSize(tmpCoord, nSize, projectionMatrix, modelviewMatrix, _renderingParameters.getViewport());
    //    cerr << lod << endl;
    if (lod < 0) {
      //      cerr << ",";
      continue;
    }
    //    cerr << ".";
    glLoadName(itv.id);
    glPushMatrix();
    glTranslatef( tmpCoord.getX() , tmpCoord.getY() , tmpCoord.getZ() );
    glRotatef(elementRotation->getNodeValue(itv), 0., 0., 1.);
    glScalef(nSize.getW(), nSize.getH(), nSize.getD());
    //    if (elementGraph->getNodeValue(itv) == 0)
    glyphs.get(elementShape->getNodeValue(itv))->draw(itv);
    //    else
    //    glyphs.get(0)->draw(itv);
    glPopMatrix();
  }
  glPopAttrib();
  tlp::glTest(__PRETTY_FUNCTION__);
}
//=================================================================
//Placement des ar�es du graphe �l'index i=startIndex..startIndex+n-number_of_edges
void GlGraph::makeEdgeSelect(int startIndex) {
  if (!_renderingParameters.isDisplayEdges()) return;
  initProxies();
  glMatrixMode(GL_MODELVIEW);
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_CULL_FACE);
  glDisable(GL_LINE_SMOOTH);
  glDisable(GL_POLYGON_SMOOTH);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  static const Color tmpColor(255,255,255,100);
  Iterator<edge> *itE = _graph->getEdges();
  while (itE->hasNext()) {
    edge ite=itE->next();
    glLoadName(ite.id);
    const Coord &tmpCoord=elementLayout->getNodeValue(_graph->source(ite));
    const Coord &tmpCoord2=elementLayout->getNodeValue(_graph->target(ite));
    const LineType::RealType &lCoord=elementLayout->getEdgeValue(ite);
    drawEdge(tmpCoord, tmpCoord2, tmpCoord, tmpCoord2, lCoord,
             tmpColor, tmpColor, elementSize->getEdgeValue(ite),
	     elementShape->getEdgeValue(ite));
  } delete itE;
  glPopAttrib();
  tlp::glTest(__PRETTY_FUNCTION__);
}
//=====================================================
