from math import ceil
import logging

import cherrypy
try:
    from sqlobject.main import SelectResults
except ImportError:
    pass

import turbogears
from turbogears.decorator import weak_signature_decorator
from turbogears.view import variable_providers

log = logging.getLogger("turbogears.paginate")

def paginate(var_name, default_order='', limit=10,
            allow_limit_override=False, max_pages=5):
    def entangle(func):
        def decorated(func, *args, **kw):
            page = int(kw.pop('tg_paginate_no', 1))
            limit_ = int(kw.pop('tg_paginate_limit', limit))
            order = kw.pop('tg_paginate_order', default_order)
            reversed = kw.pop('tg_paginate_reversed', None)
            
            if not allow_limit_override:
                limit_ = limit
            
            log.debug("Pagination params: page=%s, limit=%s, order=%s, "
                      "reversed=%s", page, limit_, order, reversed)
            
            # get the output from the decorated function    
            output = func(*args, **kw)
            if not isinstance(output, dict):
                return output
            try:
                var_data = output[var_name]
            except KeyError:
                raise "Didn't get expected variable"
            
            if order and not default_order:
                raise "If you want to enable ordering you need " \
                      "to provide a default_order" 
            
            row_count = 0
            if isinstance(var_data, SelectResults):
                row_count = var_data.count()
                col = getattr(var_data.sourceClass.q, order, None)
                if default_order:
                    if col:
                        var_data = var_data.orderBy(col)
                    else:
                        raise "The order column (%s) doesn't exist" % order
                    if reversed:
                        var_data = var_data.reversed()
            elif isinstance(var_data, list):
                row_count = len(var_data)
            else:
                raise 'Variable is not a list or SelectResults'

            offset = (page-1) * limit_
            page_count = int(ceil(float(row_count)/limit_))

            # if it's possible display every page
            if page_count <= max_pages:
                pages_to_show = range(1,page_count+1)
            else:
                pages_to_show = _select_pages_to_show(page_count=page_count,
                                              current_page=page,
                                              max_pages=max_pages)
                
            # which one should we use? cherrypy.request.input_values or kw?
            #input_values = cherrypy.request.input_values.copy()
            input_values = kw.copy()
            input_values.pop('self', None)
            
            cherrypy.request.paginate = Paginate(current_page=page,
                                                 limit=limit_, 
                                                 pages=pages_to_show, 
                                                 page_count=page_count, 
                                                 input_values=input_values, 
                                                 order=order,
                                                 reversed=reversed)
                                                 
            # we replace the var with the sliced one
            endpoint = offset + limit_
            log.debug("slicing data between %d and %d", offset, endpoint)
            output[var_name] = var_data[offset:endpoint]

            return output
        return decorated
    return weak_signature_decorator(entangle)

def _paginate_var_provider(d): 
    # replaced cherrypy.thread_data for cherrypy.request
    # thanks alberto!
    paginate = getattr(cherrypy.request, 'paginate', None)
    if paginate:
        d.update(dict(paginate=paginate))
variable_providers.append(_paginate_var_provider)

class Paginate:
    """class for variable provider"""
    def __init__(self, current_page, pages, page_count, input_values, 
                 limit, order, reversed):
        self.pages = pages
        self.limit = limit
        self.page_count = page_count
        self.current_page = current_page
        self.input_values = input_values
        self.order = order
        self.reversed = reversed
        
        self.input_values.update(dict(tg_paginate_limit=limit,
                                      tg_paginate_order=order,
                                      tg_paginate_reversed=reversed))
        if current_page < page_count:
            self.input_values.update(dict(
                                tg_paginate_no=current_page+1,
                                tg_paginate_limit=limit))
            self.href_next = turbogears.url(cherrypy.request.path, input_values)
            self.input_values.update(dict(
                                tg_paginate_no=page_count,
                                tg_paginate_limit=limit))
            self.href_last = turbogears.url(cherrypy.request.path, input_values)
        else:
            self.href_next = None
            self.href_last = None
            
        if current_page > 1:
            self.input_values.update(dict(
                                tg_paginate_no=current_page-1,
                                tg_paginate_limit=limit))
            self.href_prev = turbogears.url(cherrypy.request.path, input_values)
            self.input_values.update(dict(
                                tg_paginate_no=1,
                                tg_paginate_limit=limit))
            self.href_first = turbogears.url(cherrypy.request.path, input_values)
        else:
            self.href_prev = None
            self.href_first = None
            
    def get_href(self, page, order=None, reverse_order=None):
        if order:
            if order == self.order:
                if self.reversed:
                    reversed = None
                else:
                    reversed = True
            else:
                reversed = None
                if reverse_order:
                    if reversed:
                        reversed = None
                    else:
                        reversed = True
        else:
            order = self.order
            reversed = self.reversed
        
        self.input_values.update(dict(tg_paginate_no=page, 
                                      tg_paginate_order=order,
                                      tg_paginate_reversed=reversed))
        
        return turbogears.url('', self.input_values)


def _select_pages_to_show(current_page, page_count, max_pages):
    pages_to_show = []
    
    if max_pages < 3:
        raise "The minimun value for max_pages on this algorithm is 3"

    if page_count <= max_pages:
        pages_to_show = range(1,page_count+1)
    
    pad = 0
    if not max_pages % 2:
        pad = 1
        
    start = current_page - (max_pages / 2) + pad
    end = current_page + (max_pages / 2)
    
    if start < 1:
        end = end + (start * -1) + 1
        start = 1

    if end > page_count:
        start = start - (end - page_count)
        end = page_count
        
    return range(start, end+1)
