/*
 * Copyright (c) Tomas Znamenacek
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

package net.sf.turkey;

import jargs.gnu.CmdLineParser;

import java.io.BufferedReader;
import java.io.InputStreamReader;

import java.awt.Toolkit;
import java.awt.datatransfer.StringSelection;

/**
 * The command-line interface for Turkey.
 */
class CLI {

	public void run(String[] args) {
		
		CmdLineParser parser = new CmdLineParser();
		
		CmdLineParser.Option autoOption = parser.addBooleanOption('a', "automate");

		CmdLineParser.Option helpOption = parser.addBooleanOption('h', "help");
		CmdLineParser.Option versionOption = parser.addBooleanOption('v', "version");
		CmdLineParser.Option copyOption = parser.addBooleanOption('c', "copy");
		CmdLineParser.Option htmlOption = parser.addBooleanOption('t', "html");
		
		CmdLineParser.Option sentencesOption = parser.addIntegerOption('s', "sentences");
		CmdLineParser.Option paragraphsOption = parser.addIntegerOption('p', "paragraphs");

		CmdLineParser.Option dictionaryOption = parser.addStringOption('d', "dictionary");

		try {
			parser.parse(args);
			
		} catch (CmdLineParser.OptionException e) {
			
			System.err.println(e.getMessage());
			printUsage();
			System.exit(1);
		}

		Boolean help = (Boolean)parser.getOptionValue(helpOption);
		Boolean version = (Boolean)parser.getOptionValue(versionOption);
		Boolean copy = (Boolean)parser.getOptionValue(copyOption);
		Boolean html = (Boolean)parser.getOptionValue(htmlOption);

		Integer sentences = (Integer)parser.getOptionValue(sentencesOption);
		Integer paragraphs = (Integer)parser.getOptionValue(paragraphsOption);
	
		String dictionary = (String)parser.getOptionValue(dictionaryOption);
		
		if ((help != null) && help.booleanValue()) {
			printUsage();
			System.exit(0);
		}

		if ((version != null) && version.booleanValue()) {
			System.out.print(getFileContents("VERSION"));
			System.exit(0);
		}

		Generator g = Generator.getInstance();

		if (sentences != null)
			g.setSentencesPerPar(sentences.intValue());
		if (paragraphs != null)
			g.setParagraphsTotal(paragraphs.intValue());
		if ((html != null) && html.booleanValue())
			g.setHTMLOutput(true);

		if ((dictionary != null) && dictionary.equals("help")) {
			System.out.println("Available dictionaries: " + g.getDictionaryNames());
			System.exit(0);
		}

		if (dictionary != null)
			g.selectDictionary(dictionary);

		if ((copy != null) && copy.booleanValue()) {
			setClipboard(g.getText());
			System.exit(0);
		}
		
		System.out.print(g.getText());
	}
	
	protected void printUsage() {

		System.out.println(getFileContents("cli-help.txt"));
	}
	
	protected void setClipboard(String s) {
		
		Toolkit.getDefaultToolkit().getSystemClipboard().setContents(
			new StringSelection(s), null);
	}
		
	/** Returns the contents of a file. */
	protected String getFileContents(String fileName) {

		try {
			BufferedReader r = new BufferedReader(new InputStreamReader(
				getClass().getResourceAsStream(fileName)));
		
			StringBuffer s = new StringBuffer();
			String line;
			
			while ((line = r.readLine()) != null)
				s.append(line + "\n");
			
			r.close();
			return s.toString();
			
		} catch (Exception e) {

			return "(could not read contents of " + fileName + ")";
		}
	}
}
