 /* twpsk:  PSK31 for Linux with a Motif interface
 * Copyright (C) 1999-2014 Ted Williams WA0EIR 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * Version: 4.1 - Feb 2014
 */

#include "twpskCB.h"

#include "GUI.h"
#include "twpskWids.h"
#include "twpskWF.h"
#include "twpskScope.h"
#include "server/server.h"
#include "decoderWids.h"
#include <sys/shm.h>
#include <Xm/XmP.h>

extern XtAppContext ac;
extern AppRes appRes;
extern Position winlocs[MAX_DECODERS][2];
extern Wids pskwids;
extern TwpskCB twpskCB;
extern Disp disp;
extern Scope scope;
extern int use_stereo;
extern int init_audio();
extern Widget shell;

Pixel darkBG, lightBG;
enum { MODE_RX, MODE_TX, MODE_TUNE };
int rxtxmode=MODE_RX;


/*
 * appendRXtext - not really a callback, but fit well here.
 * adds characters to the end of the RX text widget
 * and puts cursor to the last position.  Called by workProc()
 * userData is the widget id of the scrollbar.
 *
 * Underline == 1 to underline transmitted text.
 */
void appendRXtext (Widget w, char ch, int zero, XmHighlightMode highlight_mode)
{
   char str[2];
   XmTextPosition lastPos;
   int flag;
   XtPointer *obj = NULL;
   DecoderWid *dec;


   /* build a string as keyed or as caps if doLocalCaps == TRUE */
   if (pskwids.getDoLocalCaps () == TRUE)         /* do local caps? */
   {
      str[0] = toupper(ch);                       /* Build as caps */
   }
   else
   {
      str[0] = ch;                                /* build as keyed */
   }
   str[1] = '\0';

   XmTextDisableRedisplay (w);                    /* stop redisplay */
   lastPos = XmTextGetLastPosition(w);

   if (str[0] == '0' && zero == 1)                /* Print slashed zero? */
   {
      str[0] = '\330';
   }

   if (str[0] == 0x7F || str[0] == '\b')          /* If a DEL or BS char */
   {
      XmTextReplace (w, lastPos-1, lastPos, (char *)"\0");
   }
   else
   {
      XmTextInsert (w, lastPos, str);

#if 0
fprintf (stdout, "%s\n", str);
#endif

   }

   /* get the object and check its rxScrollFlag field */ 
   XtVaGetValues (w,
      XmNuserData, &obj,
      NULL);

   if (obj == (XtPointer)&pskwids)            /* for main window */
   {
      flag = pskwids.getRxScrollFlag();
   }
   else
   {
      dec = (DecoderWid *) obj;               /* for secondary window */
      flag = dec->getRxScrollFlag();
   }

   if (flag != SCROLLING)
   {
      XmTextShowPosition (w, lastPos+1);
      XmTextSetCursorPosition (w, lastPos+1);
   }

   /*
    * highlight text according to highlight_mode 
    */
   if (str[0] != '\n')
   {
      XmTextSetHighlight(w, lastPos, lastPos+1, highlight_mode);
   }
   XmTextEnableRedisplay (w);                      
}


/*
 * appendTXtext - not really a callback, but fit well here.
 * Adds characters to the end of the TX text widget
 * Called by procMacroText and procFileData work proceedure.
 */
void appendTXtext (char ch, int zero)
{
   Widget w = pskwids.getTxText();
   char str[2];
   XmTextPosition lastPos;

   str[0] = ch;                              /* Build a string */
   str[1] = '\0';
 
#if 0
   if (str[0] == '0' && zero == 1)           /* print slashed zero? */
   {
//TJW
//str[0] = '\330';
      str[0] = '0';
   }
#endif
   lastPos = XmTextGetLastPosition(w);
   XmTextSetCursorPosition (w, lastPos);     /* keep cursor at the end */
   XmTextInsert (w, lastPos, str); 
}


/*
 * Callbacks
 */

/*
 * controlsBtnCB - callback for Recv, Xmit, Tune,
 * Wide, Medium, Narrow, Waterfall and Spectrum
 * menu buttons.  Set the toggle button and let the
 * valueChanged callbacks do the rest.
 */
void controlsBtnCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   Widget wid = (Widget)cdata;
      
   XmToggleButtonSetState (wid, True, True);
}


/*
 * tuneCB - value changed callback for tune toggle button
 */
void tuneCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   String str;
   int freq;
   Widget wid = (Widget) cdata;

   if(rxtxmode == MODE_TUNE)
   {
      /* Already in Tune, i.e. nothing to do... */
      return;
   }

   if (wid != NULL)                            /* TB set by menubar*/
   {
      XmToggleButtonSetState (wid, True, True);
      return;
   }

   /* Going to tune mode */
   if(pskwids.getNetmode ())
   {
      XtVaGetValues (pskwids.getRxFreqTF(),      /* get the rx freq */
         XmNvalue, &str,
         NULL);

      XmTextFieldSetString (pskwids.getTxFreqTF(), str);   

      freq = atoi(str);
      commControl(COMM_TXCH, COMM_FREQ, (int)(freq * 100));
   }

   commControl(COMM_TXCH, COMM_MODE, MO_TUNE );
   
   if(rxtxmode == MODE_RX)
      commControl(COMM_TXCH, COMM_PTT, PTTON | PTTFORCE);

   rxtxmode = MODE_TUNE;
   changeBG ();
}


/*
 * rxCB - value changed callback for Recv radio box togglebutton
 */
void rxCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   char cwStr[20];
   int phase;
   Widget wid = (Widget) cdata;

   /* Going into receive mode */
   if (rxtxmode == MODE_RX)
   {
      /* Already receiving, i.e. nothing to do... */
      /* Maybe we should "force" rx if we are still transmitting? */
      return;
   }
   if (wid != NULL)                    /* TB set by menubar */
   {
      XmToggleButtonSetState (wid, True, True);
      return;
   }

   /* if CW ID needed? */ 
   if (rxtxmode != MODE_TUNE && 
      (XmToggleButtonGetState (pskwids.getCwidTB()) == 1))
      {
         strcpy (cwStr, " de ");       /* Build cw id string */
         strncat (cwStr, appRes.call, 15);

         phase = 0;
         scope.drawline (phase, 35, GREEN);
         /* switch to CW; PTT is already on */
         commControl (COMM_TXCH, COMM_MODE, MO_CWSEND);
         commPutData (cwStr, 0);
      }
   /* and send Postamble and schedule PTT off */
   commControl (COMM_TXCH, COMM_PTT, PTTOFF);
   /* turn off CW/Tune mode if necesary */
   commControl (COMM_TXCH, COMM_MODE, MO_NORMAL);
   phase = 0;
   scope.drawline (phase, 35, GREEN);

   /* Now force focus to the xmit Text Widget */
   XmProcessTraversal (pskwids.getTxText(), XmTRAVERSE_CURRENT);
   rxtxmode = MODE_RX;
   changeBG ();
}


/*
 * txCB - value changed callback for Xmit radio box togglebuttons
 */
void txCB (Widget w, XtPointer cdata, XtPointer cbs) 
{
   String str;
   int freq;
   int phase;
   Widget wid = (Widget) cdata;

   /* Going into transmit mode */
   if(rxtxmode==MODE_TX)
   {
      /* Already transmitting, nothing to do 
       * Maybe we should FORCE TX here (if we do a DCD schedule TX
       * otherwise) ?  Then we would have to take care about two
       * TX commands in the queue! (Maybe server/psk31tx does not yet
       * handle this correctly
       */
      return;
   }

   if (wid != NULL)               /* TB set by menubar */
   {
      XmToggleButtonSetState (wid, True, True);
      return;
   }

   /* Going to tx from rx or tune mode */
   if (pskwids.getNetmode ())                /* if NET is set */
   {
      XtVaGetValues (pskwids.getRxFreqTF(),  /* get the rx freq */
         XmNvalue, &str,
         NULL);
                                             /* put rx freq in tx freq */
      XmTextFieldSetString (pskwids.getTxFreqTF(), str);   

      freq = atoi(str);                      /* send tx freq to transmitter */ 
      commControl (COMM_TXCH, COMM_FREQ, (int)(freq * 100));
   }

   phase = 128;
   scope.drawline (phase, 35, GREEN);

   if (rxtxmode == MODE_RX)
   {  /* from RX to TX */
      commControl(COMM_TXCH, COMM_MODE, MO_NORMAL);
      commControl(COMM_TXCH, COMM_PTT, PTTON);
   }
   else
   {  /* from TUNE to TX */
      commControl(COMM_TXCH, COMM_MODE, MO_NORMAL);
   }

   /* Now force focus to the xmit Text Widget */ 
   XmProcessTraversal (pskwids.getTxText(), XmTRAVERSE_CURRENT);
   rxtxmode = MODE_TX;
   changeBG ();
}  


/*
 * changeBG - changes widget w's background to light or dark
 */
void changeBG (void)
{
   int i;
   Widget *w;
   
   w = pskwids.get_rxtTB();

   for (i=0; i<3; i++)
   {
      if (XmToggleButtonGetState(w[i]) == 0)
      {
         XtVaSetValues (w[i],
            XmNbackground, lightBG,
            NULL);
         XmToggleButtonSetState (w[i], False, True);
      }
      else
      {
         XtVaSetValues (w[i],
            XmNbackground, darkBG,
            NULL);
         XmToggleButtonSetState (w[i], True, True);
      }
   }
}   


/*
 * changeDisplayCB value changed callback - changes display
 * from spectrum to waterfall
 */
void changeDisplayCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   disp.changeDisplay ((long)cdata);
}


/*
 * wfDirCB - change the direction of the waterfall
 */
void wfDirCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   disp.setWF_up (XmToggleButtonGetState (w));
}


/*
 * localCapsCB - change local caps mode
 */
void localCapsCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   pskwids.setDoLocalCaps (XmToggleButtonGetState (w));
}


/*
 * txTextCB - modify/verify and cursor motion callback for
 * transmit text widget
 */
void txTextCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int i;
   XmTextVerifyCallbackStruct *pt = (XmTextVerifyCallbackStruct *) cbs;

   switch (pt->reason)
   {
      case XmCR_MODIFYING_TEXT_VALUE:
         if (pt->text->length == 0)                /* Got a backspace */
         { 
            commPutData ((char *)"\b", 1);
         }
         else
         {
//TJW
//printf ("ch = %s\n", pt->text->ptr);
            for (i = 0; i < pt->text->length; i++)
            {
               if (pt->text->ptr[i] == 0)          /* got a DEL */
               {
                  commPutData ((char *)"\x7F", 1);
               }
               else                                /* got ordinary char */
               {
                  commPutData (pt->text->ptr+i, 1);
               }

               /* Should we print zero with a slash? */
               if (pt->text->ptr[i] == '0' && appRes.zero == 1 )
               {
                  pt->text->ptr[i] = '\330';
               }
               /* Should we print it as CAPS? */
               if (pskwids.getDoLocalCaps () == TRUE)
               {
                  pt->text->ptr[i] = toupper (pt->text->ptr[i]);
               }
            }
         }
         break;

      case XmCR_MOVING_INSERT_CURSOR:
         /*
          * If text len == 0, then we probably just cleared the widget
          * so allow the cursor movement.  If the move is back one char
          * then allow it - probably just a backspace key but could be mouse.
          * Any thing else is a mouse click inside the text, so don't do it.
          */
         if (strlen (XmTextGetString (w)) == 0)
         {
            pt->doit = True;    /* This is OK - just cleared the widget*/
            break;
         }

         if (pt->newInsert < pt->currInsert - 1)
         {
            pt->doit = False;  
         }
         else
         {
            pt->doit = True;
         }
         break;

      default:
         fprintf (stderr, "Unexpected TxText callback reason\n");
         break;
   }
}


/*
 * arrowCB - handles the Up/Down arrows for Rx and Tx freq changes.
 * The cdata is the associated textfield widget and userData indicates
 * 'R' or 'T' for recv or xmit freq.
 */
void arrowCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   float freq, delta;
   XtPointer userData;
   String str;
   unsigned char direction;
   Widget textF = (Widget)cdata;
   XmArrowButtonCallbackStruct *cbsPtr = (XmArrowButtonCallbackStruct *)cbs;
   XButtonPressedEvent *event = (XButtonPressedEvent *) cbsPtr->event;

   XtVaGetValues (w,
      XmNuserData, &userData,
      XmNarrowDirection, &direction,
      NULL);

   if (event->state & ShiftMask)            /* if shifted click */
      delta = 8.0;                          /* change by 8 */
   else 
   {
      if (cbsPtr->click_count == 2)          /* if double click, changes by 7 */
      {                                      /* cuz already beem changed by 1 */
         delta = 7.0;
      }
      else
      {
         delta = 1.0;                       /* else single click, change by 1 */
      }
   }

   XtVaGetValues (textF,                    /* Get the freq */
      XmNvalue, &str,
      NULL);
   freq = atof(str);                        /* convert to a float */

   if (direction == XmARROW_UP)
      freq = freq + delta;
   else
      freq = freq - delta;

   if (freq < 0)                            /* no negative freqs */
      freq = 0.0;

   if (freq > MAXFREQ)                      /* nothing over MAXFREQ */
      freq = MAXFREQ;
      
   sprintf (str, "%4.1f", freq);            /* float to a char */
   
   XmTextFieldSetString (textF, str);

   if ((long)userData == 'R')
   {
      //* Set new rx freq or */
      commControl(COMM_RXCH, COMM_FREQ, (int)(freq*100));
   }
   else
   {
      /* Set new tx freq */	   
      commControl(COMM_TXCH, COMM_FREQ, (int)(freq * 100));
   }
}


/*
 * freqTextCB - handles keyboard mod/ver and value changed callbacks
 * for rx and tx freq textfields.  userData is 'R' or 'T' for rx or tx
 * text widget or pointer to DecoderWid object for subwindows
 */
void freqTextCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int i;
   float freq;
   String str;
   XtPointer userData;
   XmTextVerifyCallbackStruct *pt = (XmTextVerifyCallbackStruct *)cbs;

   /* Switch on the type of callback */   
   switch (pt->reason)
   {
      /* Activate Callback */
      /* you typed in a new freq */
      case XmCR_ACTIVATE:
         XtVaGetValues (w,
            XmNuserData, &userData,
            XmNvalue, &str,
            NULL);
            
         freq = atof(str);
         /* first, make sure new (0 >= freq <= MAXFREQ) */
         if (freq < 0 || freq > MAXFREQ)
         {
            if (freq < 0.0)
            {
               freq = 0.0;
               strcpy (str, "0.0");
            }
            else
            {
               freq = MAXFREQ;
               strcpy (str, MAXFREQSTR);
            }
            XtVaSetValues (w,
               XmNvalue, str,
               NULL);
         }
          
         if ((long)userData == 'R')
         {
            /* Set new rx freq */
            commControl(COMM_RXCH, COMM_FREQ, (int)(freq*100));      
	    /* Now force focus to the xmit Text Widget */
	    XmProcessTraversal (pskwids.getTxText(), XmTRAVERSE_CURRENT);
         }
         else if ((long)userData == 'T')
         {
            /* Set new tx freq */
            commControl(COMM_TXCH, COMM_FREQ, (int)(freq * 100));
	    /* Now force focus to the xmit Text Widget */
	    XmProcessTraversal (pskwids.getTxText(), XmTRAVERSE_CURRENT);
         }
	 else {
	    DecoderWid *dec = (DecoderWid *)cdata;
	    if(dec==NULL) {
		    fprintf (stderr,"callback with cdata==NULL and userData="
			    "%p (%c)\n", userData, (char)(long)userData);
		    break;
	    }
	    commControl(dec->commChannel, COMM_FREQ, (int)(freq*100));
	    /* Now force focus to the Swap Button Widget */
	    XmProcessTraversal(dec->getSwapBtn(), XmTRAVERSE_CURRENT);
	 }
         break;

      /* Modify/Verify Callback */
      /* allow only digits or "." in freq text widgets */
      case XmCR_MODIFYING_TEXT_VALUE:
         for (i = 0; i < pt->text->length; i++)
         {   
            if (isdigit (pt->text->ptr[i]) || (pt->text->ptr[i] == '.'))   
               pt->doit = True;
            else
               pt->doit = False;
         }
         break;

      default:
         fprintf (stderr, "freqTextCB - unexpected callback type\n");
         break;      
   }
}


/*
 * afcCB - value changed callback - sets/clears the afc mode 
 */
void afcCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XmToggleButtonCallbackStruct *ptr = (XmToggleButtonCallbackStruct *) cbs;

   /* set AFC to toggle button state */
   commControl(COMM_RXCH, COMM_AFC, ptr->set);
}


/*
 * netCB - sets/clears the net mode 
 * should xfer rx freq to tx freq too???
 */
void netCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XmToggleButtonCallbackStruct *pt = (XmToggleButtonCallbackStruct *) cbs;

   pskwids.setNetmode(pt->set);
}


/*
 * scopeCB - expose callback 
 */
void scopeCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   scope.drawcirc();
}


/*
 * wfCB callback - click to tune
 * uses button release event and pointer x position to change rx freq
 */
void wfCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   float  newfreq;
   char freq_str[8];
   int x;
   XmDrawingAreaCallbackStruct *ptr = (XmDrawingAreaCallbackStruct *) cbs;
   int button_number;
   
   if (ptr->event->xany.type == ButtonPress)
   {
      button_number = ptr->event->xbutton.button;
      x = ptr->event->xbutton.x;

      switch (button_number)
      {
         case Button1:
            newfreq = disp.new_fc(x);
	    sprintf(freq_str, "%4.1f", newfreq);
	    XmTextFieldSetString (pskwids.getRxFreqTF(), freq_str);
	    /* Set new rx freq */
            commControl(COMM_RXCH, COMM_FREQ, (int)(newfreq*100));
            break;

         case Button3:
            open2Rx(x);  	    /* create new RX decoder! */
            break;
      }
   }
}


/*
 * openBtnCB callback -  Opens secondary window at current rx freq.
 */
void openBtnCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   Widget wid;
   char *text;
   int rtn;

   rtn = open2Rx (WF_WIDTH/2);
   if (rtn != -1)  /* If we got a decoder */
   {
      wid = pskwids.getRxText();      
      text = XmTextGetString(wid);
      XmTextSetString(decoderWids[rtn].getTextWid(), text);
      /* Problem? - What if main has been scrolled? */
   }
}


/*
 * Open Secondary Rx decoder - creates a secondary decoder.
 * Used by wfCB with Button 3 pressed and openBtnCB
 * Return == ch, or -1 if none available
 */
int open2Rx (int x)
{
   float freq = disp.new_fc (x);
   int ch;
   Boolean qpsk;

   for(ch=0; ch<MAX_DECODERS; ch++)
   {
      if (decoderWids[ch].visible == -1)
      {
         decoderWids[ch].buildWidgets(DecoderWid::shell,&appRes, ch);
         break;
      } 
      else
         if (decoderWids[ch].visible==0)
            break;
   }
   if (ch >= MAX_DECODERS)
   {
      fprintf (stderr, "error: no free decoder channel!\n");
      return -1;
   }
   else
   {
      int channel = ch+3;
      decoderWids[ch].commChannel = channel; // redundand
      commControl(channel, COMM_MODE, MO_NORMAL); // init channel
      commControl(channel, COMM_DCDLEVEL, 15);

      /* Start new sub windows with LSB=off and AFC=on */
      /* but lets not default the QPSK value */
      qpsk = pskwids.getQPSK();
      decoderWids[ch].setQPSK(qpsk);
      commControl(channel, COMM_QPSK, qpsk);
      commControl(channel, COMM_LSB, 0);
      commControl(channel, COMM_AFC, 1);
      commControl(channel, COMM_FREQ, (int)(freq*100));
      decoderWids[ch].updateDisplay(freq,0,0);

      /* clear text content */
      XmTextSetString(decoderWids[ch].getTextWid(), (char *)"");
      decoderWids[ch].visible = 1;

      XtPopup (decoderWids[ch].getMainWid(), XtGrabNone);

      XtVaSetValues (decoderWids[ch].getMainWid(),
// TJW         XmNmwmDecorations, MWM_DECOR_ALL | MWM_DECOR_RESIZEH,
         NULL); 

      return ch;
   }
}


/*
 * dcdCB - Squelch value changed callback 
 * sets dcd to opposite of button state(set)
 */
void dcdCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XmToggleButtonCallbackStruct *ptr = (XmToggleButtonCallbackStruct *) cbs;
   int set = ptr->set;

   commControl(COMM_RXCH, COMM_DCD, !set);
}


/*
 * fftScaleCB - handles FFT speed
 */
void fftScaleCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XmScaleCallbackStruct *ptr = (XmScaleCallbackStruct *)cbs;

   int val = ptr->value;
   disp.fft_setup(-1, val);
}


/*
 * qpskCB - enable/disable qpsk mode
 */
void qpskCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XmToggleButtonCallbackStruct *ptr = (XmToggleButtonCallbackStruct *) cbs;

   commControl(COMM_RXCH, COMM_QPSK, ptr->set);
   commControl(COMM_TXCH, COMM_QPSK, ptr->set);
}


/*
 * lsbCB - enable/disable lsb
 */
void lsbCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XmToggleButtonCallbackStruct *ptr = (XmToggleButtonCallbackStruct *) cbs;

   commControl(COMM_RXCH, COMM_LSB, ptr->set);
   commControl(COMM_TXCH, COMM_LSB, ptr->set);
}


/*
 * bwCB - changed bandwidth displayed
 */
void bwCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int samp = (long) cdata;
   XmToggleButtonCallbackStruct *pt = (XmToggleButtonCallbackStruct *) cbs;

   if (pt->set == True)
   {
      disp.set_samples(samp);
   }
}


/*
 * fileCB - Main menu File buttons callback.
 * Builds the pathname for the text file and
 * starts a work process to send the file text.
 */

void fileCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   char *filepath, *home, *filename;
   char *filetext;
   XtAppContext ac = XtWidgetToApplicationContext(w);

   XtVaGetValues (w,           /* Get the file name */
      XmNuserData, &filename,
      NULL);

   /* XtMalloc space for path to .twpskDir + fileName */
   /* and build the pathname for the text file     */

   home = getenv ("HOME");
   filepath = (char *) XtMalloc (strlen (home) + strlen ("/.twpskDir/") +
                               strlen (filename) +2);
   strcpy (filepath, home);
   strcat (filepath, "/.twpskDir/");
   strcat (filepath, filename);

   /* read the file */
   filetext = getFile(filepath);     /* procFileText must XtFree this! */

   /* start a work proc */
   XtAppAddWorkProc (ac, procFileText, (XtPointer) filetext);
   XtFree (filepath);                /* Clean house */
}


/*
 * quitCB - time to QRT!
 * First, check the secondary decoder windows.
 * If they were used and closed, save their XY location in the winlocs array.
 * If they were used, but not closed, leave its position in winlocs alone.
 * Write the array to $HOME/.twpskDir/.twpsk.dat
 */
void quitCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int i, rtn;
   Position x, y;

   for (i=0; i<MAX_DECODERS; i++)
   {
      /* If this decoder was used and has been closed,
       * then get its location and save it in the winlocs array.
       */ 
//TJW check it out. double check?
      if (decoderWids[i].visible >= 0 && decoderWids[i].visible == 0)
      { 
         XtVaGetValues (decoderWids[i].getMainWid(),
            XmNx, &x,
            XmNy, &y,
            NULL);
         winlocs[i][0] = x;
         winlocs[i][1] = y;
      }
   }
   /* write the array to ini file */
   if ((rtn = iniProc ('w')) == FALSE)
   {
      fprintf (stderr, "iniProc failed\n");
   }
   exit (0);
}


/*
 * rxFreqFocusCB - rxFreqTF focus callback
 * used by main window and subwindows
 */
void rxFreqFocusCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   static Boolean savedAFC;
   XmAnyCallbackStruct *ptr = (XmAnyCallbackStruct *) cbs;

   switch (ptr->reason)
   {
      case XmCR_FOCUS:
         /* getting focus */
         XtVaSetValues (w,
            XmNbackground, darkBG,
            NULL);

         /* clear focusFlag in main or 2nd rx win object */
         if(cdata==(XtPointer)&pskwids)
         {
            savedAFC = pskwids.getAFC();
            pskwids.setAFC(False);
            pskwids.setRxFreqFocus(True);
         }
         else
         {
            DecoderWid *dec = (DecoderWid *)cdata;
            savedAFC = dec->getAFC();
            dec->setAFC(False);
            dec->setFreqFocus(True);
         }
         break;


      case XmCR_LOSING_FOCUS:
         /* lost focus */
         XtVaSetValues (w,
            XmNbackground, lightBG,
            NULL);

         /* set focusFlag in main or 2nd rx win object */
         if(cdata==(XtPointer) &pskwids)
         {
            pskwids.setAFC(savedAFC);
            pskwids.setRxFreqFocus(False);
            XmProcessTraversal (pskwids.getTxText(), XmTRAVERSE_CURRENT);
         }
         else
         {
            DecoderWid *dec = (DecoderWid *)cdata;
            dec->setAFC (savedAFC);
            dec->setFreqFocus(False);
            XmProcessTraversal (dec->getRxText(), XmTRAVERSE_CURRENT);
         }
         break;

      default:
         fprintf (stderr, "rxFreqFocusCB: gain/losing focus error\n");
         break;
   }
}


/*
 * rxTextCB - gainPrimary and motionVerify callbacks for rxText
 */
void rxTextCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   static char save[MAXCALL+1] = "";   /* room for the null */
   size_t i;
   char *str;
   int shmid;
   callspt *calls = NULL;
   XmTextVerifyCallbackStruct *pt = (XmTextVerifyCallbackStruct *) cbs;

   switch (pt->reason)
   {
      case XmCR_GAIN_PRIMARY:
      case XmCR_MOVING_INSERT_CURSOR:
         /* get the selected text */
         str = XmTextGetSelection (w);

         /* check to see if nothing or too much is selected*/
         if ((str == NULL) || (strlen (str) > MAXCALL))
         {
            /* nothing selected or selection is to long, so go away */
            XtFree (str);
            return;
         }

         /* see if string has not changed from last call */
         if ((strncmp (str, save, MAXCALL) == 0) || (calls->outCall == '\0'))
         {
            /* save == str so nothing changed */
            XtFree (str);
            return;
         }

         /* save original form in case of extra callbacks on same thing */
         strncpy (save, str, MAXCALL);

         /* change any slashed 0's to an ordinary 0 */
         for (i=0; i<strlen(str); i++)
         {
            if (str[i] == '\330')
            {
               str[i] = '0';
            }
         }

         /* put selection text in shared mem */
         if ((shmid = shmget (KEY, SHMSIZE, 0600)) < 0)
         {
            fprintf (stderr, "No twlog!\n");
            XtFree (str);
            return;
         }

         if ((calls = (callspt *)shmat (shmid, NULL, 0)) ==  (callspt *)-1)
         {
            perror ("twpsk - shmat failed");
            XtFree (str);
            return;
         }

         strncpy (calls->outCall, str, MAXCALL);
         shmdt ((void *)calls);                       /* detach shm   */
         XtFree (str);
         break;

      default:
         fprintf (stdout, "invalid callback or room to grow\n");
         break;
   }
   return;
}


/*
 * rxScrollBarCB - clears/sets rxScrollFlag to
 * enable/disable rxText scrolling
 */
void rxScrollBarCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int max, position, slider;
   int scrollMode;
   Widget rxText;
   DecoderWid *dec = (DecoderWid *)(cdata);

   /* Get the rxText widget of the Object that called us */
   if (cdata == (XtPointer)&pskwids)  /* for the main window */
      rxText = pskwids.getRxText();      
   else                               /* for the secondary windows */
      rxText = dec->getRxText();

   /* get some values from the scrollbar */
   XtVaGetValues (w,
      XmNmaximum, &max,
      XmNvalue, &position,
      XmNsliderSize, &slider,
      NULL);

   /* Set scroll mode off */
   if (max - position - slider == 0) 
   {
      XtVaSetValues (rxText,   /* Go to normal mode if at the botton */
         XmNbackground, lightBG,
         NULL);
      scrollMode = 0;
   }
   else
   /* Set scroll mode on */
   {
      XtVaSetValues (rxText,
         XmNbackground, darkBG,
         NULL);
      scrollMode = 1;
   }

   /* Set scroll mode to object */
   if (cdata == (XtPointer)&pskwids)
   {
      pskwids.setRxScrollFlag (scrollMode);    /* for the main window */
   }
   else
   {
      dec->setRxScrollFlag (scrollMode);       /* for the secondary windows */
   }
}


/*
 * brightnessCB - sets disp.brightness equal to scale value
 */
void brightCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XmScaleCallbackStruct *ptr = (XmScaleCallbackStruct *)cbs;
   
   disp.brightness = ptr->value;
}


/*
 * popupDiagCB - pops up the dialog box passed as cdata
 */

void popupDiagCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   Widget diag = (Widget) cdata;

   XtManageChild (diag);
}


/*
 * closeVideoDiagCB
 * doing nothing now
 */
void closeVideoDiagCB (Widget w, XtPointer cdata, XtPointer cbs)
{

}


/*
 * popupHandler - Event handler to popup clear text menu
 */
void popupHandler (Widget w, XtPointer cdata, XEvent *event,  Boolean *cont)
{
   Widget menuWid = (Widget)cdata;

   if (event->type == ButtonPress && event->xbutton.button == Button3)
   {
      XmMenuPosition (menuWid, &(event->xbutton));
      XtManageChild (menuWid);
   }
}


/*
 * clrTextCB - clears the text in the Rx and Tx window
 */
void clrTextCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   Widget wid;
   DecoderWid *dec;
   Pixel bg;

// TJW
// commControl(COMM_TXCH, COMM_PTT, PTTOFF);
   XtVaGetValues (XtParent(w),       /* Get rowColumn (menu) wid */
      XmNuserData, &wid,             /* userData is ID of text widget */
      XmNforeground, &bg,            /* HACK - only works if popup */
      NULL);                         /* are correct color */

   XtVaSetValues (wid,
      XmNvalue, "",                  /* clear the text widget */
      XmNcursorPosition,(XmTextPosition) 0,
      XmNbackground, bg,             /* and make sure bg is correct */
      NULL);

   /* make sure the scrollFlag gets cleared too - this way for main window */
   if (wid == pskwids.getRxText() || wid == pskwids.getTxText())
   {
      pskwids.setRxScrollFlag(0);
   }
   else    /* or this way for the secondary windows */
   {
      XtVaGetValues (wid,
         XmNuserData, &dec,          /* Get userData (this ptr) */
         NULL);
      dec->setRxScrollFlag(0);
   }
}


/*
 * sendOver Action - gets the other call from shared memory,
 * gets your call, builds the "over" string, and sends it. 
 * If there is no call - just send de mycall
 */
void sendOver (Widget w, XEvent *e, String args[], Cardinal *nargs)
{

   char *pt;

   if ((pt = getHisCall()) != (char *)0)        /* get his call */
   {                                            /* got it */
      procMacroText(pt);                        /* send his call, */ 
      shmdt ((void *)pt);                       /* detach shm   */
   }
   procMacroText ((char*) "de");                /* send de */
   procMacroText (appRes.call);                 /* and send my call */
}


/*
 * sendHisCall Action - send the call from twlog shared memory
 */
void sendHisCall (Widget w, XEvent *e, String args[], Cardinal *nargs)
{
   char *ch;

   if ((ch = getHisCall()) != (char *)0)        /* get his call */
   {                                            /* got it */
      procMacroText(ch);                        /* send his call, */ 
      shmdt ((void *)ch);                       /* detach shm   */
   }
}


/*
 * sendMacro Action - send all the action arguments
 */
void sendMacro (Widget w, XEvent *e, String args[], Cardinal *nargs)
{
   int i;

   for (i=0; i<(int)*nargs; i++)
   {
      procMacroText (args[i]);
   }
}


/*
 * getHisCall - attach the shared menory if running twlog and returns
 * a char * to his call
 * Used by sendOver and sendCall
 */

char *getHisCall ()
{
   int shmid;
   callspt *calls;

   if ((shmid = shmget (KEY, SHMSIZE, 0666)) < 0)
   {
      fprintf (stderr, "No twlog!\n");
      return ((char*) 0);
   }

   if ((calls = (callspt *)shmat (shmid, NULL, 0)) ==  (callspt *)-1)
   {
      perror ("twpsk - shmat failed");
      return ((char *) 0);
   }
   return (calls->inCall);
}


/*
 * procMacroText - sends a string to appendTXtext
 */
void procMacroText (char *ch)
{
   int i;

   for (i=0; ch[i] != '\0'; i++)
   {
      appendTXtext (ch[i], appRes.zero);            /* append the text */
   }
   appendTXtext (' ', appRes.zero);                 /* done, append a space */
}


/*
 * seekCB - adds a timeout routine for seek
 */
#define TIMER 5
XtIntervalId id=0;
int first = 1;
int haltFlag = 0;
int seeking = 0;

void seekCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   if (seeking == 1)
   {
      haltFlag = 1;
      return;
   }
   first = 1;
   seeking = 1;
   XtVaSetValues (pskwids.getRxFreqTF(),
      XmNforeground, WhitePixelOfScreen(XtScreen(pskwids.getRxFreqTF())),
      NULL);
   id = XtAppAddTimeOut (ac, TIMER, (XtTimerCallbackProc) seekTimeOut, cdata);
}


/*
 * haltCB - removes the timeout
 */
void haltCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   if (seeking)
   {
      haltFlag = 1;
   }
}


/*
 * seekTimeOut - 
 */
#define STEP1 50.0
#define STEP2 5.0

void seekTimeOut (XtPointer cdata, XtIntervalId id)
{
   String str=(char *)"";
   float freq;
   int rtn;
   float step = STEP1;

   if (!first)                    /* Not the first time, so check data */
   {
      step = STEP2;
      rtn = disp.find_center(WF_WIDTH/2);
      if (rtn != -1)              /* Found one, so */
      {
         freq = disp.new_fc(rtn);
         commControl(COMM_RXCH, COMM_FREQ, (int)(freq*100));
         first = 1;
         seeking = 0;
         XtVaSetValues (pskwids.getRxFreqTF(),
            XmNforeground, BlackPixelOfScreen(XtScreen(pskwids.getRxFreqTF())),
            NULL);
         return;                  /* and don't sked another timeout */
      }
   }

   /* on pass .... */
   first = 0;
   str = XmTextGetString (pskwids.getRxFreqTF());
   freq = atof (str);

   if ((long)cdata == 'U')
   {
      freq = freq + step;
   }
   if ((long)cdata == 'D')
   {
      freq = freq - step;
   }

   sprintf (str, "%4.1f", freq);

   /* See if we should halt or about to go over the edge of the world */
   if ( (haltFlag == 1) || (freq < 0.0) || (freq > MAXFREQ) )
   {
      first = 1;
      haltFlag = 0;
      seeking = 0;
      XtVaSetValues (pskwids.getRxFreqTF(),
         XmNforeground, BlackPixelOfScreen(XtScreen(pskwids.getRxFreqTF())),
         NULL);
   }
   else
   {
      XmTextFieldSetString (pskwids.getRxFreqTF(), str);
      commControl(COMM_RXCH, COMM_FREQ, (int)(freq*100));
      id = XtAppAddTimeOut(ac, TIMER, (XtTimerCallbackProc) seekTimeOut, cdata);
   }
}


/*
 * chansCB - specify soundcard channels
 */
void chansCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   use_stereo = (long)cdata;
}
