#ifndef SIGNAL_CC
#define SIGNAL_CC
//---------------------------------------------------------------------------
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.


//
// $Id: Signal.cc,v 1.2 2003/12/08 19:24:39 dmartin Exp $
//
//---------------------------------------------------------------------------

#include "SourceData.hh"

template <class Type>
inline
Signal<Type>::Signal() :
  SignalBase( EXPLICIT, true, drvVal.getUniversalKind() ){
  source = NULL;
  compositeResolvedSignal = false;
  sourceBaseDeleteFlag = false;
}


template <class Type>
inline
Signal<Type>::Signal(const int i) :
  SignalBase( EXPLICIT, true, drvVal.getUnivseralKind() ){
  source = NULL;
  drvVal = Type(i);
  effVal = Type(i);
  compositeResolvedSignal = false;  
  sourceBaseDeleteFlag = false;
}

template <class Type>
inline
Signal<Type>::Signal(const VHDLData& val) :
  SignalBase( EXPLICIT, true, val.getUniversalKind() ){
  source = NULL;
  this->operator=(val);
  compositeResolvedSignal = false;
  sourceBaseDeleteFlag = false;
}


template <class Type>
inline
Signal<Type>::Signal(const Signal<Type>& s)
 : SignalBase(s.name) {
  *source = *s.source;
  drvVal = s.drvVal;
  effVal = s.effVal;
  // Yes, these make multiple pointers to the same memory.  Since the
  // contents of this memory never change, it is allocated once at the
  // beginning of the simulation, and deleted at the end.  SO, everyone
  // looks at the same memory, and it never needs to be
  // newd/deleted/copied.
  type = s.type;
  sensitive = s.sensitive;
  numAttributes = s.numAttributes;
  fanDest = s.fanDest;
  attributeList = s.attributeList;
  compositeResolvedSignal = false;
  allAttributes = s.allAttributes;
  sourceBaseDeleteFlag = false;
}

template <class Type>
inline
Signal<Type>::~Signal() {
//   delete additionalDriverList;
  if ( sourceBaseDeleteFlag == true ){
    delete source;
  }
}

template <class Type>
inline ObjectBase &
Signal<Type>::operator=( const ObjectBase& s){
//   Signal<Type> sig = (Signal<Type>&) s;
  switch(s.getKind()) {
  case SIGNAL:
  case IMPLICIT_SIGNAL:
    SignalBase::operator=( dynamic_cast<const SignalBase &>(s));
    source = dynamic_cast<const Signal<Type> &>(s).source;
    drvVal = dynamic_cast<const Signal<Type> &>(s).drvVal;
    effVal = dynamic_cast<const Signal<Type> &>(s).effVal;
    sourceBaseDeleteFlag = dynamic_cast<const Signal<Type> &>(s).sourceBaseDeleteFlag;
    // Yes, these make multiple pointers to the same memory.  Since the
    // contents of this memory never change, it is allocated once at the
    // beginning of the simulation, and deleted at the end.  SO, everyone
    // looks at the same memory, and it never needs to be
    // newd/deleted/copied.
    compositeResolvedSignal =  dynamic_cast<const SignalBase &>(s).isCompositeResolvedSignal();
    break;
  case VARIABLE:
    drvVal = s.readVal();
    effVal = s.readVal();
    break;
  case SIGNAL_NETINFO:
    break;
  default:
    cerr << "Unknown ObjectBase::ObjectType" << endl;
    abort();
  }
  return *this;
}

template <class Type>
inline ObjectBase&
Signal<Type>:: operator=(const VHDLData& s) {
  //This  functions is used only during initialization
  //Will not be called later
  drvVal = s;
  effVal = s;
  return ((ObjectBase &) (*this));
}


template <class Type>
inline void
Signal<Type>::updateDriver( const int sigId, 
			    int srcProcessId,
			    const VHDLData *data, 
			    const ArrayInfo *sInfo, 
			    const ArrayInfo *dInfo ){
  SignalBase *driverInBlock = 0;

  if( ( sInfo == NULL && dInfo == NULL ) ||
      ( sInfo->operator==( defaultInfo ) && dInfo->operator==( defaultInfo ) )) {
    // Update the correct driver.
    if( source == NULL ){
      cerr << "Error: signal with name |" << getName() << "| and id |"
	   << getSigId() << "| has no driver but was asked to update.  This "
	   << "is an unrecoverable condition." << endl;
	abort();
    }
    
    if ( source->getDriver( srcProcessId ) != NULL ){
      const Type *asType = dynamic_cast<const Type *>(data);
      ASSERT( asType != 0 );
      source->getDriver( srcProcessId )->updateData( *asType );
    } 
    else {
      ASSERT(additionalDriverList != NULL);
      for ( int i = 0 ; i < getAdditionalDriverList()->getNumberOfElements(); i++){
	VHDLType *currentDriver = (VHDLType *)getAdditionalDriverList()->getElement(i);
	driverInBlock = currentDriver->findSigInBlock( sigId, srcProcessId );
	if ( driverInBlock != NULL){
	  driverInBlock->updateDriver( sigId, srcProcessId, data, sInfo, dInfo );
	  return; // work done quit.
	}
      }
      cerr << "Signal<Type>::updateDriver(..) could not update the driver" << endl;
      abort();
    }
  } 
  else {
    cerr << "Attempt to update driver of non-scalar signal" << endl;
    abort();
  }
}

template <class Type>
inline void
Signal<Type>::disconnectDriver(VHDLKernel *srcId) {
  // disconnect the correct driver.
  SourceData* driver = source->getDriver(srcId);
  ASSERT(driver != NULL);
  driver->disConnect();
}

 
template <class Type>
inline VHDLType*
Signal<Type>::readDriver(VHDLKernel * sigsrc) {
  return source->getDriver(sigsrc)->getData();
}


template <class Type>
inline void
Signal<Type>::updateDrvVal(const VHDLData* data, const ArrayInfo *sInfo, 
			      const ArrayInfo *dInfo) {
  if(sInfo == NULL && dInfo == NULL) {
    drvVal = *(Type*)data;
  } else if (data->length() != drvVal.length()) {
   // if we're not assigning to the entire bit_vector
#ifdef VHDLDBG
    cout << name << "Data length is " << sInfo->length() << ", my length is " 
	 << drvVal.length();
    cout << ".  Assigning data |";
    data->print(cout);
    cout << "| to sig " << name << " slice " << *sInfo << endl;
#endif
    if(dInfo != NULL) {
      drvVal.assignSlice(data, sInfo, dInfo);
    } else {
      drvVal.assignSlice(data, sInfo);
    }      
  }
  else { // assignment is for.operator== arrays
    drvVal = *(Type*)data;
  }
}


template <class Type>
inline VHDLData*
Signal<Type>::readDrvVal() {
  return &drvVal;
}

template <class Type>
inline void
Signal<Type>::updateEffVal(const VHDLData& data, const ArrayInfo *sInfo,
				    const ArrayInfo *dInfo) {
  updateEffVal(&data, sInfo, dInfo);
}

template <class Type>
inline void
Signal<Type>::updateEffVal( const VHDLData* data, 
			    const ArrayInfo *sInfo,
			    const ArrayInfo *dInfo) {
  if(sInfo == NULL && dInfo == NULL) {
    effVal = dynamic_cast<const Type &>(*data);
  } 
  else if (data->length() != drvVal.length()) {
   // if we're not assigning to the entire bit_vector
#ifdef VHDLDBG
    cout << name << "Data length is " << sInfo->length() << ", my length is " 
	 << effVal.length();
    cout << ".  Assigning data |";
    data->print(cout);
    cout << "| to sig " << name << " slice " << *sInfo << endl;
#endif
    if(dInfo != NULL) {
      effVal.assignSlice(data, sInfo, dInfo);
    } else {
      effVal.assignSlice(data, sInfo);
    }      
  }
  else { // assignment is for.operator== arrays
    effVal = dynamic_cast<const Type &>(*data);
  }
}

template <class Type>
inline VHDLData*
Signal<Type>::readEffVal() {
  return &effVal;
}


template <class Type>
inline vector<FanDest *> &
Signal<Type>::getFanList() const {
  return fanDest;
}


template <class Type>
inline VHDLType*
Signal<Type>::resolve( VHDLKernel *processPtr ){
  return source->resolve( processPtr, getType() );
}

// if you can figure out how to make operator<< link for this, do it!
template <class Type>
inline void 
Signal<Type>::print(ostream& os) const {
  register int i;

  os << "Signal(" << getSigId() << "): " << getType() << ":";
  os << " drv(" << drvVal << "), eff(" << effVal << ")";
  if( getBusResFn() != -1 ){
    os << "busResFn(#" << getBusResFn() << ")";
  }
//   os << endl << "Source: " << endl;
//   if(source != NULL) {
//     source->print(os);
//   } else {
//     os << "(null)";
//   }
  if ( getFanOut() > 0) {
    os << "    fanouts: ";
    for (i = 0; i < getFanOut(); i++) {
      os << getFanDest()[i] << ": ";
    }
    os << "\n";
  }
  if( getNumAttributes() > 0) {
    os << "    Attributes: ";
    for (i = 0; i < getNumAttributes(); i++) {
      os << const_cast<Signal<Type> *>(this)->getAttributeList()[i] << " ";
    }
    os << endl;
  }
  os << "Is a composite resolved Signal? " << compositeResolvedSignal << endl;
}

template <class Type>
inline SignalBase *
Signal<Type>::findSigInBlock( int sigId, VHDLKernel *srcId ){
  if( getSigId() == sigId ){
    if (source->getDriver(srcId) != NULL ){
      return this;
    }
  }
  
  return NULL;
}

#endif
