#include "SourceInfo.hh"
#include "SourceData.hh"
#include "VHDLType.hh"
#include "VHDLProcess.hh"

const int CHILD_NOT_ADDED = -99;

SourceInfo::SourceInfo(void) : SourceBase() {
  upConvertFnId   = DEFAULT_TCF_ID;
  downConvertFnId = DEFAULT_TCF_ID;
  resolveFnId     = DEFAULT_RF_ID;
}

SourceInfo::~SourceInfo() {
  for ( int childNo = 0 ; childNo < getNumChildren() ; childNo++ ){
    delete children[childNo];
  }
}

SourceInfo&
SourceInfo::operator = (const SourceInfo& source) {
  for(int i = 0; i < source.getNumChildren(); i++ ){
    *(children[i]) = *(source.getChild(i)); // This recursively copies the
  }				               // grandchildren as well!!
  return (*this);
}

void 
SourceInfo::setUpTypeConversionFnId(TypeConversionFnId_t fnId) {
  upConvertFnId = fnId;
}

void 
SourceInfo::setDownTypeConversionFnId(TypeConversionFnId_t fnId) {
  downConvertFnId = fnId;
}

void 
SourceInfo::setResolutionFnId(ResolutionFnId_t fnId) {
  resolveFnId = fnId;
}

int 
SourceInfo::getNumChildren() const {
  return children.size();
}

int
SourceInfo::getNumActiveChildren() const {
  int retval = 0;

  for( int i = 0; i < getNumChildren(); i++ ){
    if( children[i]->_is_driver_connected() == true ){
      retval++;
    }
  }
  return retval;
}

bool
SourceInfo::_is_child_present(VHDLKernel *childId) {
  if( parent != NULL ){
    return parent->_is_child_present(childId);
  } 
  else {
    for( unsigned int i = 0; i < rootDriverTable.size(); i++ ){
      if( rootDriverTable[i] == childId ){
	return true;
      }
    }
    return false;
  }
}

bool
SourceInfo::_is_driver_connected() {
  // Return true only if atleast one child can contribute a driver!!
  
  for( int i=0; i < getNumChildren(); i++ ){
    if( children[i]->_is_driver_connected() == true ) {
      return true;
    }
  }
  return false;
}

int 
SourceInfo::addChild(SourceBase *newChild) {
  if( newChild->getNumChildren() == 0 ){
    if( newChild->get_kind() == SourceBase::SOURCE_INFO ){
      return CHILD_NOT_ADDED;
    }
  }
  int childPresent = 0;
  if( newChild->get_kind() == SourceBase::SOURCE_DATA ){
    if( _is_child_present(newChild->getSourceId()) ){
      cerr << "Try to add another driver for the same process"
	   << " with id" << newChild->getSourceId() << endl;
      childPresent = 1;
    }
    else {
      childPresent = 0;
    }
  }
  else if( newChild->get_kind() == SourceBase::SOURCE_INFO ){
    SourceInfo *siPtr = (SourceInfo*)newChild;
    for( unsigned int i = 0; i < rootDriverTable.size(); i++ ){
      if(  siPtr->_is_child_present(rootDriverTable[i]) ){
	childPresent = 1;
	cerr << "Try to add another driver for the same process"
	     << " with SourceInfo containing id" << rootDriverTable[i] << endl;
	break;
      }
    }
  } else if(  newChild->get_kind() == SourceBase::SOURCE_TYPE_CONVERT){
    // nothing to be checked for a node of SOURCE_TYPE_CONVERT
    // childPresent = 0; but that's done in initialization itself, so why worry
  }

  
  if( childPresent == 1 ){
    return CHILD_NOT_ADDED;
  }
  else {
    children.push_back( newChild );
    newChild->setParent(this);

    // only if the newChild is a SOURCE_DATA or SOURCE_INFO then the
    // the drivers are collected at the root of the resolution tree.
    if( newChild->get_kind() != SourceBase::SOURCE_TYPE_CONVERT ){
      addDriversToRootDriverTable(newChild);
    }
    
    return getNumChildren();
  }
}

int 
SourceInfo::addChild(VHDLType *newDriver, VHDLKernel* childId) {
  SourceData *newChild = NULL;

  if( _is_child_present(childId) ){
    return CHILD_NOT_ADDED;
  }

  newChild = new SourceData();
  newChild->setParent(this);
  newChild->addChild(newDriver, childId);

  children.push_back( newChild );

  if( childId != ANONYMOUS_PROCESS_ID ){
    addToRootDriverTable( newChild );
  }
  
  return getNumChildren();
}

SourceBase *
SourceInfo::getChild(const int childId) const {
  ASSERT(childId < getNumChildren());
  return children[childId];
}

SourceBase&
SourceInfo::operator[] (VHDLKernel *childId) {
  ASSERT(childId != NULL);
  return *getDriver(childId);
}

void
SourceInfo::addToRootDriverTable(SourceData *leafNode) {
  if( parent != NULL ){		// This is not the root node.
    parent->addToRootDriverTable(leafNode);
  }
  else {			// This is the root node.
    rootDriverTable.push_back( leafNode->getSourceId() );
    if( leafNode->getSourceId() != SourceBase::ANONYMOUS_PROCESS_ID ){
      childIndex.insert( leafNode->getSourceId()->getName(), leafNode );
    }
  }
}

void 
SourceInfo::addDriversToRootDriverTable(SourceBase *newNode) {
  if( parent != NULL ){
    parent->addDriversToRootDriverTable(newNode);
  } 
  else {
    for(int i = 0; i < newNode->getTableSize(); i++ ){
      addToRootDriverTable(newNode->getDriveratIndex(i));
    } // for
  } // else
}

VHDLType *
SourceInfo::resolve( VHDLKernel *processPtr, SigType typ ){
  VHDLType *retval;

  int activeChildren = getNumActiveChildren();
  VHDLType **valueArray = NULL;
  int i = 0;
  int j = 0;
  if(activeChildren == 0 ){	// Sanity check.
    // This could happen for signals of kind bus or register
    // When the kind is a bus then pass a NULL array to the resolution
    // function to find the effcetive value whereas in case of a register
    // the value does not change 
    if( typ == G_BUS ){
      // Pass a NULL array to the resolution function
      ASSERT(resolveFnId != SourceBase::DEFAULT_RF_ID);
      retval = ((savantResolutionFn[resolveFnId])( static_cast<VHDLProcess *>(processPtr), 0, NULL));
      return retval;
    } else {
      return NULL;
    }
  }

  valueArray = (VHDLType **) new VHDLType*[activeChildren];
  TypeConversionFnId_t fnId = SourceBase::DEFAULT_TCF_ID;
  VHDLType *temp_vhdl_type_ptr = NULL;  
  // Two counters are used to point to the drivers.
  // "i" refers to all the drivers in the simulation
  // "j" refers to the set of all connected Drivers during the
  // current simulation period.
  for(j=0, i = 0; i < getNumChildren(); i++ ){
    if( children[i]->_is_driver_connected() == true ){
      //Call the resolution function if the driver is connected
      valueArray[j] = children[i]->resolve(processPtr);
      //Call the type conversion if the driver is connected
      // First we have to do the up-type-conversion. Then we will pass on the
      // resultant vector to the resolution function. The down-type
      // conversion will be taken care of by the fanout, since this is a
      // sequence of functions operating on a data.
      temp_vhdl_type_ptr = NULL;
      fnId = children[i]->getUpTypeConversionFnId();
      if( fnId != SourceBase::DEFAULT_TCF_ID ){
	temp_vhdl_type_ptr = valueArray[j];
	valueArray[j] = (savantTypeConversionFn[fnId])( static_cast<VHDLProcess *>(processPtr),
							valueArray[j] );
	delete temp_vhdl_type_ptr;
      }
      j++;
    }
  }
  
  if( j != activeChildren ){
    // Insanity check!!!
    cerr << "Internal Error : Number of active children != number of"
	 << " drivers.\n";
    activeChildren = j;
  }
  
  //Call the Resolution function after calling the resolve of the 
  //Drivers
  if( activeChildren > 1 ){
    if( resolveFnId != SourceBase::DEFAULT_RF_ID ){
      retval = (savantResolutionFn[resolveFnId])( static_cast<VHDLProcess *>(processPtr),
						  activeChildren, valueArray);
    }
    else {
      cerr << "Number of active children for a signal is > 1. But it "
	   << " is not a resolved subtype!\n";
      retval = valueArray[0];
    }
  } else {
    retval = valueArray[0];
  }

  delete []valueArray;
  return retval;
}

void
SourceInfo::print(ostream& os) const {
  static int indent = 0;
  int i;

  for(i = 0; i < indent; i++ ){
    os << " ";
  }

  indent += 4;			// Indent each child by 4 spaces.
  
  os << "getNumChildren()(" << getNumChildren() << ") "
     << "upFn(" << upConvertFnId << ") "
     << "downFn(" << downConvertFnId << ") "
     << "resolveFn(" << resolveFnId << ")" << endl;

  for(i = 0; i < getNumChildren(); i++ ){
    children[i]->print(os);
  }

  indent -= 4;
}

SourceData*
SourceInfo::getDriver( VHDLKernel* id ){
  unsigned int i;
  if( parent != NULL ){
    return parent->getDriver(id);
  } 
  else {
    for(i = 0; i < rootDriverTable.size(); i++ ){
      if( rootDriverTable[i] == id ){
	break;
      }	// if
    } // for
    if( i == rootDriverTable.size() ){
      //For the following VHDL code
      //  type two_array is array (1 to 2) of integer;
      //  signal v : two_array;
      //       proc1 : process
      // 	begin
      
      // 	v(1) <= 1 after 10 ns;
      //       wait;
      
      //       end process;
      //There should be only one driver added to the signal since it is a
      //unresolved signal. But the source of the signal for each sub-element
      //can be different. Currently the elaborator ensures that only one driver
      // is added to the any unresolved signal. But the source of the signal's
      //sub-element is not determined at elaboration time, and hence the
      //following fix
      if( i == 1 ){
	rootDriverTable[0]  = id;
	childIndex.find( id->getName() )->setSourceId(id);
	return static_cast<SourceData *>(childIndex.find(id->getName()));
      }
      else {
	// cerr << "Driver " << id << " not present in the source tree.  "
	//       << "Aborting." << endl;
	// abort();
	return NULL;
      }
    } 
    else {
      return static_cast<SourceData *>(childIndex.find(id->getName()));
    }
  }
}


SourceData *
SourceInfo::getDriver( int processId ) {
  unsigned int i = 0;
  if( parent != NULL ){
    return parent->getDriver( processId );
  } 
  else {
    for(i = 0; i < rootDriverTable.size(); i++ ){
      if( rootDriverTable[i]->getVHDLProcessId() == processId ){
	break;
      }	// if
    } // for
    if( i <= rootDriverTable.size() ){
      return getDriver( rootDriverTable[i] );
    }
    else{
      return 0;
    }
  }
}

SourceData *
SourceInfo::getDriveratIndex(int position) {
  vector<SourceBase *> *elements = childIndex.getElementVector();
  SourceData *retval = static_cast<SourceData *>( (*elements)[position] );
  delete elements;
  return retval;
}
