//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.0.1, Copyright (C) Peter A. Buhr 1994
// 
// uSocket.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 17:06:26 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Aug 12 14:07:03 2004
// Update Count     : 837
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#include <uC++.h>
#include <uSocket.h>

//#include <uDebug.h>

#include <cstring>					// strerror, memset
#include <unistd.h>					// read, write, close, etc.
#include <sys/socket.h>
#include <sys/un.h>


#ifndef SUN_LEN
#define SUN_LEN(su) (sizeof(*(su)) - sizeof((su)->sun_path) + strlen((su)->sun_path))
#endif


//######################### uSocket #########################


void uSocket::uCreateSocket( int domain, int type, int protocol ) {
    for ( ;; ) {
	access.fd = ::socket( domain, type, protocol );
      if ( access.fd != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( access.fd == -1 ) {
        // The exception should be caught by constructor in
	// uSocketServer or uSocketClient and throw a different exception
	// indicating the failure to create a server or a client.  However, it
	// cannot be implemented because the initialization list in a
	// constructor cannot be encapsulated in a try block.  Actually, there
	// is a feature in c++ allowing catch clause to handle exception
	// propoagated out of the initialization list but g++ does not have it
	// yet.

        uThrow uOpenFailure( *this, domain, type, protocol, "unable to open socket" );
    } // if

    // Sockets always treated as non-blocking.
    access.poll.uSetStatus( uPoll::uAlwaysPoll );
    access.poll.uSetPollFlag( access.fd );

    // Sockets always have local-address reuse enable.
    const int enable = 1;				// 1 => enable option
    if ( setsockopt( access.fd, SOL_SOCKET, SO_REUSEADDR,  &enable, sizeof(enable) ) == -1 ) {
        uThrow uOpenFailure( *this, domain, type, protocol, "unable to set socket-option" );
    } // if
} // uSocket::uCreateSocket


uSocket::uSocket( int domain, int type, int protocol ) : domain( domain ), type( type ), protocol( protocol ) {
    uCreateSocket( domain, type, protocol );
} // uSocket::uSocket


uSocket::~uSocket() {
    int code;

    uThisCluster().uCloseFD( access.fd );
    for ( ;; ) {
	code = ::close( access.fd );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
        if ( ! std::uncaught_exception() ) uThrow uCloseFailure( *this, "unable to close socket" );
    } // if
} // uSocket::~uSocket


//######################### uSocketIO #########################


void uSocketIO::inetAddr::getinetAddr( unsigned short port, const char *name ) {
    if ( name[0] != '\0' ) {				// no name ?
	hostent hp;
	int herrno = 0;
	char buffer[8 * 1024];				// best guess at size
	hostent *rc;

	for ( int i = 0; i < 5; i += 1 ) {		// N tries
#if defined( __solaris__ ) || defined( __irix__ )
	    rc = gethostbyname_r( name, &hp, buffer, sizeof(buffer), &herrno );
	    if ( rc != 0 ) break;			// success ?
#elif defined( __linux__ )
	    gethostbyname_r( name, &hp, buffer, sizeof(buffer), &rc, &herrno );
	    if ( rc != 0 ) break;			// success ?
#endif
	    if ( herrno != TRY_AGAIN ) break;		// failure ?
	    sleep( 1 );					// TRY_AGAIN => delay and try again
	} // for
	if ( rc == 0 ) {
	    code = 1;
	    return;
	} // if
	memcpy( &sin_addr.s_addr, hp.h_addr, hp.h_length );
    } else {
	sin_addr.s_addr = htonl( INADDR_ANY );
    } // if

    code = 0;
    sin_family = AF_INET;
    sin_port = htons( port );
    memset( &sin_zero, '\0', sizeof(sin_zero) );
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketIO &)0x%p::inetAddr.getinetAddr name:%s, sin_family:%d sin_port:%d (%x), sin_addr:%d.%d.%d.%d\n",
	       this, name, sin_family, sin_port, sin_port,
	       sin_addr._S_un._S_un_b.s_b1, sin_addr._S_un._S_un_b.s_b2, sin_addr._S_un._S_un_b.s_b3, sin_addr._S_un._S_un_b.s_b4 );
#endif // __U_DEBUG_H__
} // uSocketIO::inetAddr::getinetAddr


uSocketIO::inetAddr::inetAddr( unsigned short port, const char *name ) {
    strcpy( _name, name );				// copy name
    getinetAddr( port, name );
} // uSocketIO::inetAddr::inetAddr


int uSocketIO::_send( char *buf, int len, int flags ) {
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::send( access.fd, buf, len, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::_send


uSocketIO::uSocketIO( uIOaccess &acc, sockaddr *saddr ) : uFileIO( acc ), saddr( saddr ) {
} // uSocketIO::uSocketIO

const struct sockaddr *uSocketIO:: getsockaddr() {
    return saddr;
} // uSocketIO::getsockname


int uSocketIO::getsockname( sockaddr *name, socklen_t *len ) {
    return ::getsockname( access.fd, name, len );
} // uSocketIO::getsockname


int uSocketIO::getpeername( sockaddr *name, socklen_t *len ) {
    return ::getpeername( access.fd, name, len );
} // uSocketIO::getpeername


int uSocketIO::send( char *buf, int len, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _send( buf, len, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::uWriteSelect;
	    if ( timeout == NULL ) {
		uThisCluster().uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uWriteTimeout( *sa, buf, len, flags, NULL, 0, timeout, "timeout during socket send" );
		    } else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uWriteTimeout( *ss, buf, len, flags, NULL, 0, timeout, "timeout during socket send" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uWriteTimeout( *sc, buf, len, flags, NULL, 0, timeout, "timeout during socket send" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.send : internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uWriteFailure( *sa, buf, len, flags, NULL, 0, timeout, "socket send fails" );
	} else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uWriteFailure( *ss, buf, len, flags, NULL, 0, timeout, "socket send fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uWriteFailure( *sc, buf, len, flags, NULL, 0, timeout, "socket send fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.send : internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::send


int uSocketIO::_sendto( char *buf, int len, int flags, sockaddr *to, socklen_t tolen ) {
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::sendto( access.fd, buf, len, flags, (sockaddr *)to, tolen );
    int terrno = errno;					// preserve errno across calls
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketIO &)0x%p._sendto attempting send to fd:%d, buf:0x%p, len:%d, flags:0x%x, to:0x%p, tolen:%d code:%d errno:%d\n",
	       this, access.fd, buf, len, flags, to, tolen, code, errno );
#endif // __U_DEBUG_H__
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::_sendto


int uSocketIO::sendto( char *buf, int len, int flags, uDuration *timeout ) {
    return sendto( buf, len, saddr, saddrlen, flags, timeout );
} // uSocketIO::sendto


int uSocketIO::sendto( char *buf, int len, struct sockaddr *to, socklen_t tolen, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _sendto( buf, len, flags, to, tolen );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::uWriteSelect;
	    if ( timeout == NULL ) {
		uThisCluster().uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uWriteTimeout( *sa, buf, len, flags, to, tolen, timeout, "timeout during socket sendto" );
		    } else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uWriteTimeout( *ss, buf, len, flags, to, tolen, timeout, "timeout during socket sendto" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uWriteTimeout( *sc, buf, len, flags, to, tolen, timeout, "timeout during socket sendto" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.sendto : internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uWriteFailure( *sa, buf, len, flags, to, tolen, timeout, "socket sendto fails" );
	} else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uWriteFailure( *ss, buf, len, flags, to, tolen, timeout, "socket sendto fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uWriteFailure( *sc, buf, len, flags, to, tolen, timeout, "socket sendto fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.sendto : internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::sendto


int uSocketIO::_recv( char *buf, int len, int flags ) {
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::recv( access.fd, buf, len, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::_recv


int uSocketIO::recv( char *buf, int len, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _recv( buf, len, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::uReadSelect;
	    if ( timeout == NULL ) {
		uThisCluster().uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uReadTimeout( *sa, buf, len, flags, NULL, NULL, timeout, "timeout during socket recv" );
		    } else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uReadTimeout( *ss, buf, len, flags, NULL, NULL, timeout, "timeout during socket recv" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uReadTimeout( *sc, buf, len, flags, NULL, NULL, timeout, "timeout during socket recv" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.recv : internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uReadFailure( *sa, buf, len, flags, NULL, NULL, timeout, "socket recv fails" );
	} else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uReadFailure( *ss, buf, len, flags, NULL, NULL, timeout, "socket recv fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uReadFailure( *sc, buf, len, flags, NULL, NULL, timeout, "socket recv fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.recv : internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::recv


int uSocketIO::_recvfrom( char *buf, int len, int flags, sockaddr *from, socklen_t *fromlen ) {
#if defined( __irix__ )
    // TEMPORARY: sets fromlen to 0 when EWOULDBLOCK
    int tmp;
    if ( fromlen != 0 ) tmp = *fromlen;			// save *fromlen, as it may be set to 0 after each attempt
#endif // __irix__
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::recvfrom( access.fd, buf, len, flags, (sockaddr *)from, (socklen_t *) fromlen );
    int terrno = errno;					// preserve errno across calls
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketIO &)0x%p._recvfrom attempting read from fd:%d, buf:0x%p, len:%d, flags:0x%x, from:0x%p, fromlen:%d code:%d errno:%d\n",
	       this, access.fd, buf, len, flags, from, fromlen, code, errno );
#endif // __U_DEBUG_H__
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
#if defined( __irix__ )
    // TEMPORARY: sets fromlen to 0 when EWOULDBLOCK
    if ( fromlen != 0 && *fromlen == 0 ) *fromlen = tmp; // reset *fromlen after each attempt
#endif // __irix__
    return code;
} // uSocketIO::_recvfrom


int uSocketIO::recvfrom( char *buf, int len, int flags, uDuration *timeout ) {
    saddrlen = baddrlen;				// set to receive buffer size
    return recvfrom( buf, len, saddr, &saddrlen, flags, timeout );
} // uSocketIO::recvfrom


int uSocketIO::recvfrom( char *buf, int len, struct sockaddr *from, socklen_t *fromlen, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _recvfrom( buf, len, flags, from, fromlen );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::uReadSelect;
	    if ( timeout == NULL ) {
		uThisCluster().uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uReadTimeout( *sa, buf, len, flags, from, fromlen, timeout, "timeout during socket recvfrom" );
		    } else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uReadTimeout( *ss, buf, len, flags, from, fromlen, timeout, "timeout during socket recvfrom" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uReadTimeout( *sc, buf, len, flags, from, fromlen, timeout, "timeout during socket recvfrom" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.recvfrom : internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uReadFailure( *sa, buf, len, flags, from, fromlen, timeout, "socket recvfrom fails" );
	} else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uReadFailure( *ss, buf, len, flags, from, fromlen, timeout, "socket recvfrom fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uReadFailure( *sc, buf, len, flags, from, fromlen, timeout, "socket recvfrom fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.recvfrom : internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::recvfrom


int uSocketIO::_recvmsg( struct msghdr *msg, int flags ) {
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::recvmsg( access.fd, msg, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.uGetStatus() == uPoll::uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::_recvmsg


int uSocketIO::recvmsg( struct msghdr *msg, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _recvmsg( msg, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::uReadSelect;
	    if ( timeout == NULL ) {
		uThisCluster().uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uReadTimeout( *sa, (const char *)msg, 0, flags, NULL, NULL, timeout, "timeout during socket recvmsg" );
		    } else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uReadTimeout( *ss, (const char *)msg, 0, flags, NULL, NULL, timeout, "timeout during socket recvmsg" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uReadTimeout( *sc, (const char *)msg, 0, flags, NULL, NULL, timeout, "timeout during socket recvmsg" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.recvmsg : internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uReadFailure( *sa, (const char *)msg, 0, flags, NULL, NULL, timeout, "socket recvmsg fails" );
	} else if ( uSocketServer *ss = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uReadFailure( *ss, (const char *)msg, 0, flags, NULL, NULL, timeout, "socket recvmsg fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uReadFailure( *sc, (const char *)msg, 0, flags, NULL, NULL, timeout, "socket recvmsg fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.recvmsg : internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::recvmsg


//######################### uSocketServer #########################


void uSocketServer::acceptor() {
    acceptorCnt += 1;
} // uSocketServer::acceptor


void uSocketServer::unacceptor() {
    acceptorCnt -= 1;
} // uSocketServer::unacceptor


int uSocketServer::_accept( struct sockaddr *adr, socklen_t *len ) {
    int tmp = 0;
    if ( len != NULL ) tmp = *len;			// save *len, as it may be set to 0 after each attempt
    if ( socket.access.poll.uGetStatus() == uPoll::uPollOnDemand ) socket.access.poll.uSetPollFlag( socket.access.fd );
    int fd = ::accept( socket.access.fd, (sockaddr *)adr, len );
    int terrno = errno;					// preserve errno across calls
    if ( socket.access.poll.uGetStatus() == uPoll::uPollOnDemand ) socket.access.poll.uClearPollFlag( socket.access.fd );
    errno = terrno;
    if ( len != NULL && *len == 0 ) *len = tmp;		// reset *len after each attempt
    return fd;
} // uSocketServer::_accept


void uSocketServer::uCreateSocketServer1( const char *name, int type, int protocol, int backlog ) {
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer1 attempting binding to name:%s\n", this, name );
#endif // __U_DEBUG_H__

    ((sockaddr_un *)saddr)->sun_family = AF_UNIX;
    strncpy( ((sockaddr_un *)saddr)->sun_path, name, sizeof(((sockaddr_un *)saddr)->sun_path) );
    saddrlen = SUN_LEN( (sockaddr_un *)saddr );
    baddrlen = sizeof(sockaddr_un);

    for ( ;; ) {
	code = ::bind( socket.access.fd, saddr, saddrlen );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
        uThrow uOpenFailure( *this, 0, name, AF_UNIX, type, protocol, backlog, "unable to bind name to socket" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    code = ::listen( socket.access.fd, backlog );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, 0, name, AF_UNIX, type, protocol, backlog, "unable to listen on socket" );
	} // if
    } // if

    acceptorCnt = 0;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer1 binding to name:%s\n", this, name );
#endif // __U_DEBUG_H__
} // uSockeServer::uCreateSocketServer1


uSocketServer::uSocketServer( const char *name, int type, int protocol, int backlog ) :
	uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketServer1( name, type, protocol, backlog );
} // uSocketServer::uSocketServer


void uSocketServer::uCreateSocketServer2( unsigned short port, int type, int protocol, int backlog ) {
    int code;

    if ( ((inetAddr *)saddr)->code > 0 ) {		// problem generating internet address ?
        uThrow uOpenFailure( *this, port, ((inetAddr *)saddr)->_name, AF_INET, type, protocol, backlog, "internet name lookup failure" );
    } // if

    baddrlen = saddrlen = sizeof(sockaddr_in);

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer2 attempting binding to port:%d, name:%s\n", this, port, ((inetAddr *)saddr)->_name );
#endif // __U_DEBUG_H__

    for ( ;; ) {
	code = ::bind( socket.access.fd, saddr, saddrlen );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
	uThrow uOpenFailure( *this, port, ((inetAddr *)saddr)->_name, AF_INET, type, protocol, backlog, "unable to bind name to socket" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    code = ::listen( socket.access.fd, backlog );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, port, ((inetAddr *)saddr)->_name, AF_INET, type, protocol, backlog, "unable to listen on socket" );
	} // if
    } // if

    acceptorCnt = 0;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer2 binding to port:%d, name:%s\n", this, port, ((inetAddr *)saddr)->_name );
#endif // __U_DEBUG_H__
} // uSocketServer::uCreateSocketServer2


uSocketServer::uSocketServer( unsigned short port, int type, int protocol, int backlog ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketServer2( port, type, protocol, backlog );
} // uSocketServer::uSocketServer


void uSocketServer::uCreateSocketServer3( unsigned short *port, int type, int protocol, int backlog ) {
    uCreateSocketServer2( 0, type, protocol, backlog );	// 0 port number => select an used port

    getsockname( saddr, &saddrlen );			// insert unsed port number into address ("bind" does not do it)
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer3 binding to port:%d\n", this, ntohs( ((sockaddr_in *)saddr)->sin_port ) );
#endif // __U_DEBUG_H__
    *port = ntohs( ((sockaddr_in *)saddr)->sin_port );
} // uSocketServer::uCreateSocketServer3


uSocketServer::uSocketServer( unsigned short *port, int type, int protocol, int backlog ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( 0 ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketServer3( port, type, protocol, backlog );
} // uSocketServer::uSocketServer


uSocketServer::~uSocketServer() {
    delete saddr;
    if ( acceptorCnt != 0 ) {
	if ( ! std::uncaught_exception() ) uThrow uCloseFailure( *this, acceptorCnt, "closing socket server with outstanding acceptor(s)" );
    } // if
} // uSocketServer::~uSocketServer


void uSocketServer::uSetClient( struct sockaddr *addr, socklen_t len ) {
#ifdef __U_DEBUG__
    if ( len > baddrlen ) {
	uAbort( "(uSocketServer &)0x%p.uSetClient( addr:0x%p, len:%d ), new server address to large.", this, addr, len );
    } // if
#endif // __U_DEBUG__
    memcpy( saddr, addr, len );
} // uSocketServer::uSetClient


void uSocketServer::uGetClient( struct sockaddr *addr, socklen_t *len ) {
#ifdef __U_DEBUG__
    if ( *len < baddrlen ) {
	uAbort( "(uSocketServer &)0x%p.uGetClient( addr:0x%p, len:%d ), area for server address to small.", this, addr, *len );
    } // if
#endif // __U_DEBUG__
    *len = baddrlen;
    memcpy( addr, saddr, baddrlen );
} // uSocketServer::uGetServer


//######################### uSocketAccept #########################


const uSocket &uSocketAccept::sock() const {
    return socketserver.socket;
} // uSocketAccept::sock


void uSocketAccept::uCreateSocketAcceptor( uDuration *timeout, struct sockaddr *adr, socklen_t *len ) {
    baddrlen = saddrlen = socketserver.saddrlen;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept before accept\n", this );
#endif // __U_DEBUG_H__

    for ( ;; ) {
	access.fd = socketserver._accept( adr, len );
      if ( access.fd != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::uReadSelect;
	    if ( timeout == NULL ) {
		uThisCluster().uSelect( socketserver.socket.access.fd, mask );
	    } else {
		timeval t = *timeout;		// convert to timeval for select
		if ( uThisCluster().uSelect( socketserver.socket.access.fd, mask, &t ) == 0 ) { // timeout ?
#ifdef __U_DEBUG_H__
		    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept timeout fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__
		    uThrow uOpenTimeout( *this, timeout, adr, len, "timeout during accept" );
		} // if
	    } // if
	} // if
    } // for

    if ( access.fd == -1 ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept accept failure fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__
	uThrow uOpenFailure( *this, timeout, adr, len, "unable to accept connection on socket server" );
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept after accept fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__

    // On some UNIX systems the file descriptor created by accept inherits the
    // non-blocking characteristic from the base socket; on other system this
    // does not seem to occur, so explicitly set the file descriptor to
    // non-blocking.

    access.poll.uSetStatus( uPoll::uAlwaysPoll );
    access.poll.uSetPollFlag( access.fd );
    openAccept = true;
} // uSocketAccept::uCreateSocketAcceptor


int uSocketAccept::_close() {
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.close, enter fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__

    uThisCluster().uCloseFD( access.fd );
    for ( ;; ) {
	code = ::close( access.fd );
	if ( code != -1 ) break;
	if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.close, exit fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__
    openAccept = false;

    return code;
} // uSocketAccept::_close


uSocketAccept::uSocketAccept( uSocketServer &s, struct sockaddr *adr, socklen_t *len ) :
	uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( NULL ), adr( adr ), len( len ) {
    uCreateSocketAcceptor( NULL, adr, len );
    socketserver.acceptor();
} // uSocketAccept::uSocketAccept


uSocketAccept::uSocketAccept( uSocketServer &s, uDuration *timeout, struct sockaddr *adr, socklen_t *len ) :
	uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( timeout ), adr( adr ), len( len ) {
    uCreateSocketAcceptor( timeout, adr, len );
    socketserver.acceptor();
} // uSocketAccept::uSocketAccept


uSocketAccept::uSocketAccept( uSocketServer &s, bool doAccept, struct sockaddr *adr, socklen_t *len ) :
	uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( NULL ), adr( adr ), len( len ) {
    if ( doAccept ) {
	uCreateSocketAcceptor( NULL, adr, len );
    } // if
    socketserver.acceptor();
} // uSocketAccept::uSocketAccept


uSocketAccept::uSocketAccept( uSocketServer &s, uDuration *timeout, bool doAccept, struct sockaddr *adr, socklen_t *len ) :
	uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( timeout ), adr( adr ), len( len ) {
    if ( doAccept ) {
	uCreateSocketAcceptor( timeout, adr, len );
    } // if
    socketserver.acceptor();
} // uSocketAccept::uSocketAccept


uSocketAccept::~uSocketAccept() {
    socketserver.unacceptor();
    if ( openAccept && _close() == -1 ) {
	if ( ! std::uncaught_exception() ) uThrow uCloseFailure( *this, "unable to close socket acceptor" );
    } // if
} // uSocketAccept::~uSocketAccept


void uSocketAccept::accept() {
    accept( timeout );					// use global timeout
} // uSocketAccept::accept


void uSocketAccept::accept( uDuration *timeout ) {
    close();
    uSocketAccept::timeout = timeout;
    uCreateSocketAcceptor( timeout, adr, len );
} // uSocketAccept::accept


void uSocketAccept::close() {
    if ( openAccept && _close() == -1 ) {
	uThrow uCloseFailure( *this, "unable to close socket acceptor" );
    } // if
} // uSocketAccept::close


//######################### uSocketClient #########################


void uSocketClient::uCreateSocketClient1( const char *name, uDuration *timeout, int type, int protocol ) {
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient1 attempting connection to name:%s\n", this, name );
#endif // __U_DEBUG_H__

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	((sockaddr_un *)saddr)->sun_family = AF_UNIX;
	strncpy( ((sockaddr_un *)saddr)->sun_path, name, sizeof(((sockaddr_un *)saddr)->sun_path) );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );
	baddrlen = sizeof(sockaddr_un);

	for ( ;; ) {
	    if ( socket.access.poll.uGetStatus() == uPoll::uPollOnDemand ) socket.access.poll.uSetPollFlag( socket.access.fd );
	    code = ::connect( socket.access.fd, saddr, saddrlen );
	    int terrno = errno;				// preserve errno across calls
	    if ( socket.access.poll.uGetStatus() == uPoll::uPollOnDemand ) socket.access.poll.uClearPollFlag( socket.access.fd );
	    errno = terrno;
	  if ( code != -1 ) break;
	  if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	    if ( errno == U_EWOULDBLOCK ) {
		int mask = uCluster::uReadSelect;
		if ( timeout == NULL ) {
		    uThisCluster().uSelect( socket.access.fd, mask );
		} else {
		    timeval t = *timeout;		// convert to timeval for select
		    if ( uThisCluster().uSelect( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
			uThrow uOpenTimeout( *this, 0, name, timeout, AF_UNIX, type, protocol, "timeout during connect" );
		    } // if
		} // if
	    } // if
	} // for
	// Alternate approach for waiting for a non-blocking connect.
	if ( code == -1 && errno == EINPROGRESS ) {
	    int mask = uCluster::uWriteSelect;
	    if ( timeout == NULL ) {
		uThisCluster().uSelect( socket.access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().uSelect( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
		    uThrow uOpenTimeout( *this, 0, name, timeout, AF_UNIX, type, protocol, "timeout during connect" );
		} // if
	    } // if
	    code = 0;
	} // if

	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, 0, name, timeout, AF_UNIX, type, protocol, "unable to connect to socket" );
	} // if
    } else {
	// Create temporary file for the client to communicate through.
	((sockaddr_un *)saddr)->sun_family = AF_UNIX;
	tmpnm = tempnam( NULL, "uC++" );
	strncpy( ((sockaddr_un *)saddr)->sun_path, tmpnm, sizeof(((sockaddr_un *)saddr)->sun_path) );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );

	for ( ;; ) {
	    code = ::bind( socket.access.fd, saddr, saddrlen );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, 0, name, timeout, AF_UNIX, type, protocol, "unable to bind name to socket" );
	} // if

	// Now set default address specified by user.
	strncpy( ((sockaddr_un *)saddr)->sun_path, name, sizeof(((sockaddr_un *)saddr)->sun_path) );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );
	baddrlen = sizeof(sockaddr_un);
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient1 connection to name:%s\n", this, name );
#endif // __U_DEBUG_H__
} // uSocketClient::uCreateSocketClient1


uSocketClient::uSocketClient( const char *name, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketClient1( name, NULL, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( const char *name, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketClient1( name, timeout, type, protocol );
} // uSocketClient::uSocketClient


void uSocketClient::uCreateSocketClient2( unsigned short port, const char *name, uDuration *timeout, int type, int protocol ) {
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient2 attempting connection to port:%d, name:%s\n", this, port, name );
#endif // __U_DEBUG_H__

    if ( ((inetAddr *)saddr)->code > 0 ) {		// problem generating internet address ?
        uThrow uOpenFailure( *this, port, ((inetAddr *)saddr)->_name, timeout, AF_INET, type, protocol, "internet name lookup failure" );
    } // if

    baddrlen = saddrlen = sizeof(sockaddr_in);

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    if ( socket.access.poll.uGetStatus() == uPoll::uPollOnDemand ) socket.access.poll.uSetPollFlag( socket.access.fd );
	    code = ::connect( socket.access.fd, saddr, saddrlen );
	    int terrno = errno;			// preserve errno across calls
	    if ( socket.access.poll.uGetStatus() == uPoll::uPollOnDemand ) socket.access.poll.uClearPollFlag( socket.access.fd );
	    errno = terrno;
	  if ( code != -1 ) break;
	  if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	    if ( errno == U_EWOULDBLOCK ) {
		int mask = uCluster::uReadSelect;
		if ( timeout == NULL ) {
		    uThisCluster().uSelect( socket.access.fd, mask );
		} else {
		    timeval t = *timeout;		// convert to timeval for select
		    if ( uThisCluster().uSelect( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
			uThrow uOpenTimeout( *this, port, name, timeout, AF_INET, type, protocol, "timeout during connect" );
		    } // if
		} // if
	    } // if
	} // for
	// Alternate approach for waiting for a non-blocking connect.
	if ( code == -1 && errno == EINPROGRESS ) {
	    int mask = uCluster::uWriteSelect;
	    if ( timeout == NULL ) {
		uThisCluster().uSelect( socket.access.fd, mask );
	    } else {
		timeval t = *timeout;		// convert to timeval for select
		if ( uThisCluster().uSelect( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
		    uThrow uOpenTimeout( *this, port, name, timeout, AF_INET, type, protocol, "timeout during connect" );
		} // if
	    } // if
	    code = 0;
	} // if
	
	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, port, name, timeout, AF_INET, type, protocol, "unable to connect to socket" );
	} // if
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient2 connection to port:%d, name:%s\n", this, port, name );
#endif // __U_DEBUG_H__
} // uSocketClient::uCreateSocketClient2


uSocketClient::uSocketClient( unsigned short port, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, ((inetAddr *)saddr)->_name, NULL, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, ((inetAddr *)saddr)->_name, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, const char *name, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port, name ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, name, NULL, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, const char *name, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port, name ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, name, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::~uSocketClient() {
    if ( saddr->sa_family == AF_UNIX && socket.type == SOCK_DGRAM ) {
	// Remove temporary file created for communication.
	int code = unlink( tmpnm );
	delete tmpnm;
	if ( code == -1 ) {
	    delete saddr;
	    if ( ! std::uncaught_exception() ) uThrow uCloseFailure( *this, "unlink failed for temporary pipe" );
	} // if
    } // if
    delete saddr;
} // uSocketClient::~uSocketClient


void uSocketClient::uSetServer( struct sockaddr *addr, socklen_t len ) {
#ifdef __U_DEBUG__
    if ( len > baddrlen ) {
	uAbort( "(uSocketClient &)0x%p.uSetServer( addr:0x%p, len:%d ), new server address to large.", this, addr, len );
    } // if
#endif // __U_DEBUG__
    memcpy( saddr, addr, len );
} // uSocketClient::uSetServer


void uSocketClient::uGetServer( struct sockaddr *addr, socklen_t *len ) {
#ifdef __U_DEBUG__
    if ( *len < baddrlen ) {
	uAbort( "(uSocketClient &)0x%p.uGetServer( addr:0x%p, len:%d ), area for server address to small.", this, addr, *len );
    } // if
#endif // __U_DEBUG__
    *len = baddrlen;
    memcpy( addr, saddr, baddrlen );
} // uSocketClient::uGetServer


//######################### uSocket (cont) #########################


uSocket::uFailure::uFailure( const uSocket &socket, const char *const msg ) : uIOFailure( msg ), _socket( socket ) {}

const uSocket &uSocket::uFailure::socket() const { return _socket; }

void uSocket::uFailure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p, %.256s.", &socket(), message() );
} // uSocket::uFailure::defaultTerminate


uSocket::uOpenFailure::uOpenFailure( const uSocket &socket, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocket::uFailure( socket, msg ), domain( domain ), type( type ), protocol( protocol ) {}

void uSocket::uOpenFailure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p.uSocket( domain:%d, type:%d, protocol:%d ), %.256s.\nError(%d) : %s.",
	    &socket(), domain, type, protocol, message(), errNo(), strerror( errNo() ) );
} // uSocket::uOpenFailure::defaultTerminate


uSocket::uCloseFailure::uCloseFailure( const uSocket &socket, const char *const msg ) :	uSocket::uFailure( socket, msg ) {}

void uSocket::uCloseFailure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p.~uSocket(), %.256s %d.\nError(%d) : %s.",
	    &socket(), message(), socket().access.fd, errNo(), strerror( errNo() ) );
} // uSocket::uCloseFailure::defaultTerminate


uInitEvent(uSocket::uFailure);
uInitEvent(uSocket::uOpenFailure);
uInitEvent(uSocket::uCloseFailure);


//######################### uSocketServer (cont) #########################


uSocketServer::uFailure::uFailure( const uSocketServer &server, const char *const msg ) : uSocket::uFailure( server.socket, msg ), _server( server ) {
} // uSocketServer::uFailure::uFailure

const uSocketServer &uSocketServer::uFailure::server() const {
    return _server;
} // uSocketServer::uFailure::server

int uSocketServer::uFailure::fileDescriptor() const {
    return server().access.fd;
} // uSocketServer::uFailure::fileDescriptor

void uSocketServer::uFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p, %.256s.\nError(%d) : %s.",
	    &server(), message(), errNo(), strerror( errNo() ) );
} // uSocketServer::uFailure::defaultTerminate


uSocketServer::uOpenFailure::uOpenFailure( const uSocketServer &server, unsigned short port, const char *name, int domain, int type, int protocol, int backlog, const char *const msg ) :
	uSocketServer::uFailure( server, msg ), port( port ), domain( domain ), type( type ), protocol( protocol ), backlog( backlog ) {
    strcpy( _name, name );				// copy file name
} // uSocketServer::uOpenFailure::uOpenFailure

const char *const uSocketServer::uOpenFailure::name() const { return _name; }

void uSocketServer::uOpenFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.uSocketServer( port:%d, name:\"%s\", cluster:0x%p, domain:%d, type:%d, protocol:%d, backlog:%d ), %.256s.\nError(%d) : %s.",
	    &server(), port, name(), &uThisCluster(), domain, type, protocol, backlog, message(), errNo(), strerror( errNo() ) );
} // uSocketServer::uOpenFailure::defaultTerminate


uSocketServer::uCloseFailure::uCloseFailure( const uSocketServer &server, int acceptorCnt, const char *const msg ) :
	uSocketServer::uFailure( server, msg ), acceptorCnt( acceptorCnt ) {
} // uSocketServer::uCloseFailure::uCloseFailure

void uSocketServer::uCloseFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.~uSocketServer(), %.256s, %d acceptor(s) outstanding.",
	    &server(), message(), acceptorCnt );
} // uSocketServer::uCloseFailure::defaultTerminate


uSocketServer::uReadFailure::uReadFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketServer::uReadFailure::uReadFailure

void uSocketServer::uReadFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketServer::uReadFailure::defaultTerminate


uSocketServer::uReadTimeout::uReadTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::uReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketServer::uReadTimeout::uReadTimeout

void uSocketServer::uReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketServer::uReadTimeout::defaultTerminate


uSocketServer::uWriteFailure::uWriteFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketServer::uWriteFailure::uWriteFailure

void uSocketServer::uWriteFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketServer::uWriteFailure::defaultTerminate


uSocketServer::uWriteTimeout::uWriteTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::uWriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketServer::uWriteTimeout::uWriteTimeout

void uSocketServer::uWriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketServer::uWriteTimeout::defaultTerminate


uInitEvent(uSocketServer::uFailure);
uInitEvent(uSocketServer::uOpenFailure);
uInitEvent(uSocketServer::uCloseFailure);
uInitEvent(uSocketServer::uReadFailure);
uInitEvent(uSocketServer::uReadTimeout);
uInitEvent(uSocketServer::uWriteFailure);
uInitEvent(uSocketServer::uWriteTimeout);


//######################### uSocketAccept (cont) #########################


uSocketAccept::uFailure::uFailure( const uSocketAccept &acceptor, const char *const msg ) :
	uSocket::uFailure( acceptor.sock(), msg ), _acceptor( acceptor ) {}

const uSocketAccept &uSocketAccept::uFailure::acceptor() const {
    return _acceptor;
} // uSocketAccept::uFailure::acceptor

const uSocketServer &uSocketAccept::uFailure::server() const {
    return _acceptor.socketserver;
} // uSocketAccept::uFailure::server

int uSocketAccept::uFailure::fileDescriptor() const {
    return acceptor().access.fd;
} // uSocketAccept::uFailure::fileDescriptor

void uSocketAccept::uFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p( server:0x%p ), %.256s.", &acceptor(), &server(), message() );
} // uSocketAccept::uFailure::defaultTerminate


uSocketAccept::uOpenFailure::uOpenFailure( const uSocketAccept &acceptor, const class uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg ) :
	uSocketAccept::uFailure( acceptor, msg ), timeout( timeout ), adr( adr ), len( len ) {}

void uSocketAccept::uOpenFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.uSocketAccept( server:0x%p, timeout:0x%p, adr:0x%p, len:0x%p ), %.256s, error(%d) : %s",
	    &acceptor(), &server(), timeout, adr, len, message(), errNo(), strerror( errNo() ) );
} // uSocketAccept::uOpenFailure::defaultTerminate


uSocketAccept::uOpenTimeout::uOpenTimeout( const uSocketAccept &acceptor, const class uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg ) :
	uSocketAccept::uOpenFailure( acceptor, timeout, adr, len, msg ) {}

void uSocketAccept::uOpenTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.uSocketAccept( server:0x%p, timeout:0x%p, adr:0x%p, len:0x%p ), %.256s",
	    &acceptor(), &server(), timeout, adr, len, message() );
} // uSocketAccept::uOpenTimeout::defaultTerminate


uSocketAccept::uCloseFailure::uCloseFailure( const uSocketAccept &acceptor, const char *const msg ) :
	uSocketAccept::uFailure( acceptor, msg ) {}

void uSocketAccept::uCloseFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.~uSocketAccept(), %.256s.\nError(%d) : %s.",
	    &acceptor(), message(), errNo(), strerror( errNo() ) );
} // uSocketAccept::uCloseFailure::defaultTerminate


uSocketAccept::uReadFailure::uReadFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketAccept::uReadFailure::uReadFailure

void uSocketAccept::uReadFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketAccept::uReadFailure::defaultTerminate


uSocketAccept::uReadTimeout::uReadTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::uReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketAccept::uReadTimeout::uReadTimeout

void uSocketAccept::uReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketAccept::uReadTimeout::defaultTerminate


uSocketAccept::uWriteFailure::uWriteFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketAccept::uWriteFailure::uWriteFailure

void uSocketAccept::uWriteFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketAccept::uWriteFailure::defaultTerminate


uSocketAccept::uWriteTimeout::uWriteTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::uWriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketAccept::uWriteTimeout::uWriteTimeout

void uSocketAccept::uWriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketAccept::uWriteTimeout::defaultTerminate


uInitEvent(uSocketAccept::uFailure);
uInitEvent(uSocketAccept::uOpenFailure);
uInitEvent(uSocketAccept::uOpenTimeout);
uInitEvent(uSocketAccept::uCloseFailure);
uInitEvent(uSocketAccept::uReadFailure);
uInitEvent(uSocketAccept::uReadTimeout);
uInitEvent(uSocketAccept::uWriteFailure);
uInitEvent(uSocketAccept::uWriteTimeout);


//######################### uSocketClient (cont) #########################


uSocketClient::uFailure::uFailure( const uSocketClient &client, const char *const msg ) : uSocket::uFailure( client.socket, msg ), _client( client ) {
} // uSocketClient::uFailure::uFailure

const uSocketClient &uSocketClient::uFailure::client() const {
    return _client;
} // uSocketClient::uFailure::client

int uSocketClient::uFailure::fileDescriptor() const {
    return client().access.fd;
} // uSocketClient::uFailure::fileDescriptor

void uSocketClient::uFailure::defaultTerminate() const {
	uAbort( "(uSocketClient &)0x%p, %.256s.",
		&client(), message() );
} // uSocketClient::uFailure::defaultTerminate


uSocketClient::uOpenFailure::uOpenFailure( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocketClient::uFailure( client, msg ), port( port ), timeout( timeout ), domain( domain ), type( type ), protocol( protocol ) {
    strcpy( _name, name );				// copy file name
} // uSocketClient::uOpenFailure::uOpenFailure

const char *const uSocketClient::uOpenFailure::name() const { return _name; }

void uSocketClient::uOpenFailure::defaultTerminate() const {
    if ( domain == AF_UNIX ) {
	uAbort( "(uSocketClient &)0x%p.uSocketClient( name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.\nError(%d) : %s.",
	       &client(), name(), &uThisCluster(), timeout, domain, type, protocol, message(), errNo(), strerror( errNo() ) );
    } else { // AF_INET
	uAbort( "(uSocketClient &)0x%p.uSocketClient( port:%d, name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.\nError(%d) : %s.",
	       &client(), port, name(), &uThisCluster(), timeout, domain, type, protocol, message(), errNo(), strerror( errNo() ) );
    } // if
} // uSocketClient::uOpenFailure::defaultTerminate


uSocketClient::uOpenTimeout::uOpenTimeout( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocketClient::uOpenFailure( client, port, name, timeout, domain, type, protocol, msg ) {
} // uSocketClient::uOpenTimeout::uOpenTimeout

void uSocketClient::uOpenTimeout::defaultTerminate() const {
    if ( domain == AF_UNIX ) {
	uAbort( "(uSocketClient &)0x%p.uSocketClient( name:\"%s\", cluster:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.",
		&client(), name(), &uThisCluster(), domain, type, protocol, message() );
    } else { // AF_INET
	uAbort( "(uSocketClient &)0x%p.uSocketClient( port:%d, name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.",
		&client(), port, name(), &uThisCluster(), timeout, domain, type, protocol, message() );
    } // if
} // uSocketClient::uOpenTimeout::defaultTerminate


uSocketClient::uCloseFailure::uCloseFailure( const uSocketClient &client, const char *const msg ) :
	uSocketClient::uFailure( client, msg ) {}

void uSocketClient::uCloseFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.~uSocketClient(), %.256s.\nError(%d) : %s.",
	    &client(), message(), errNo(), strerror( errNo() ) );
} // uSocketClient::uCloseFailure::defaultTerminate


uSocketClient::uReadFailure::uReadFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketClient::uReadFailure::uReadFailure

void uSocketClient::uReadFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketClient::uReadFailure::defaultTerminate


uSocketClient::uReadTimeout::uReadTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::uReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketClient::uReadTimeout::uReadTimeout

void uSocketClient::uReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketClient::uReadTimeout::defaultTerminate


uSocketClient::uWriteFailure::uWriteFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketClient::uWriteFailure::uWriteFailure

void uSocketClient::uWriteFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketClient::uWriteFailure::defaultTerminate


uSocketClient::uWriteTimeout::uWriteTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::uWriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketClient::uWriteTimeout::uWriteTimeout

void uSocketClient::uWriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketClient::uWriteTimeout::defaultTerminate


uInitEvent(uSocketClient::uFailure);
uInitEvent(uSocketClient::uOpenFailure);
uInitEvent(uSocketClient::uOpenTimeout);
uInitEvent(uSocketClient::uCloseFailure);
uInitEvent(uSocketClient::uReadFailure);
uInitEvent(uSocketClient::uReadTimeout);
uInitEvent(uSocketClient::uWriteFailure);
uInitEvent(uSocketClient::uWriteTimeout);


// Local Variables: //
// compile-command: "gmake install" //
// End: //
