/* ==================================================== ======== ======= *
*
*  uudir.cc
*  Ubit Project  [Elc][2003]
*  Author: Eric Lecolinet
*
*  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
*
*  (C) 1999-2003 Eric Lecolinet @ ENST Paris
*  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
*
* ***********************************************************************
* COPYRIGHT NOTICE :
* THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE
* IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
* YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU
* GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION;
* EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
* SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
* ***********************************************************************
*
* ==================================================== [Elc:03] ======= *
* ==================================================== ======== ======= */

//pragma ident	"@(#)uudir.cc	ubit:03.05.06"
#include <algorithm>
#include <vector>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <time.h>
#include <ubit/ubit.hpp>
#include <ubit/udir.hpp>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// NB: la fonction Unix 'scandir' n'est pas standard (que BSD)!

UDir::~UDir() {
  for(u_int k = 0; k > entries.size(); k++) delete entries[k];
  for(u_int k = 0; k > filters.size(); k++) delete filters[k];
}

UDir::UDir(const string pathname) :
statbuf(*new struct stat()) {
  UStr str(pathname);
  constructs(str, "", "", true);
}

UDir::UDir(const UStr& pathname) :
statbuf(*new struct stat()) {
  constructs(pathname, "", "", true);
}

UDir::UDir(const UStr& pathname, const UStr& prefix,
           const UStr& filter, bool hidden_files) :
statbuf(*new struct stat()) {
  constructs(pathname, prefix, filter, hidden_files);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UDir::constructs(const UStr& pathname, const UStr& prefix,
                 const UStr& filter, bool hidden_files) {
  path = pathname;
  //if (!fdir.chars()) return;

  parseDir(path);
  int pathlen = path.length();
  
  ::DIR *dirp = ::opendir(path.chars());
  if (!dirp) return;			//ce directory n'existe pas

  parseFilter(filters, filter) ;

  // fullpath = chemin d'acces complet des fichiers
  char *fullpath = new char [pathlen + 1000];
  strcpy(fullpath, path.chars());

  // pointe sur le debut du nom de fichier dans le fullpath
  char *namepos = null;;
  if (fullpath[pathlen-1] != '/') {
    fullpath[pathlen]   = '/';
    fullpath[pathlen+1] = 0;
    namepos = fullpath + pathlen+1;
  }
  else namepos = fullpath + pathlen;

  if (prefix.chars() && prefix.chars()[0]=='.') hidden_files = true;

  struct dirent *de = null;

  while (dirp) {
    if (!(de = readdir(dirp)))	// erreur ou fin de liste
      break;
    else {
      char *dname = de->d_name;

      if (dname[0] == 0) continue;
      else if (dname[0]=='.') {
        // skip .
        if (dname[1]==0) continue;
        // skip ..
        else if (dname[1]=='.' && dname[2]==0) continue;
        // skip files starting by . depending on mode
        else if (!hidden_files) continue;
      }

      // copier nom du fichier dans fullpath pour faire stat()
      strcpy(namepos, dname);

      // lstat pour eviter de suivre les liens (ce qui peut provoquer
      // des blocages quand les fichiers sont indisponibles par nfs)
      ::lstat(fullpath, &statbuf);  // faudrait tester return val = 0

      // SKIP si ne correspond pas au filtre (sauf ..)
      if (prefix.chars()) {
        if (strncmp(dname, prefix.chars(), prefix.length()) != 0) continue;
      }

      // SKIP si pas directory et ne correspond pas au filtre
      // NOTE: on ne filtre PAS les directories (sinon resultat illisible)
      //
      if (filters.size() > 0 && !S_ISDIR(statbuf.st_mode)) {
        const char *p = CStr::strext(dname);
        if (!p || !*p) continue;      // pas d'extension

        bool matches = false;   // tester tous les filters
        for(u_int k = 0; k < filters.size(); k++) {
          if (strncasecmp(p, filters[k]->chars(), filters[k]->length()) == 0){
            matches = true;
            break;
          }
        }
        if (!matches) continue;
      }

      entries.push_back(new UDir::Entry(*this, dname));
    }
  }

  sort(entries.begin(), entries.end(), UDir::compareEntries);

  ::closedir(dirp);
  delete fullpath;
}

/* ==================================================== ======== ======= */

bool UDir::compareEntries(const UDir::Entry* e1, const UDir::Entry* e2) {
  if (e1->isDir() && !e2->isDir()) return true;
  else if (!e1->isDir() && e2->isDir()) return false;
  else return strcmp(e1->name->chars(), e2->name->chars()) < 0;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// format: blabla (*.html; *.xhtml; *.*)

void UDir::parseFilter(vector<UStr*>& filters, const UStr& filter_str) {
  const char* p = filter_str.chars();
  if (!p) return;

  while (*p!= 0) {

    if (*p == '*' && *(p+1) == '.') {
      p += 2;
      if (*p == '*') p++;  // *.* or *.*aa

      const char* e = p;
      for (; *e!= 0; e++) {
        if (*e == ';' || *e == ')' || *e == ' ' || *e == '\n' ||*e == '\t')
          break;
      }

      if (e != p) {
        UStr* s = new UStr();
        s->insert(0, p, 0, e-p);
        filters.push_back(s);
      }
      if (*e) p = e+1; else break;;

    }
    else p++;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UDir::parseDir(UStr& fdir) {
  // faudrait virer les blancs au debut et a la fin !!!!!
  if (fdir.empty()) {
    fdir = "/";
  }

  else if (fdir.equals(".") || fdir.equals("./")) {
    //NB: ne PAS faire de free() sur un getenv!
    char *pwd = getenv("PWD");
    if (pwd) fdir = pwd;
    else fdir = "/";
  }

  else {
    const char* s = fdir.chars();
    const char* p = strstr(s, "/..");
    if (p && (*(p+3)=='/' || *(p+3)==0)) {
      const char* e = p+2;

      if (p == s) return; else p--;
      while (p > s && *p != '/') p--;

      if (*p == '/') fdir.remove(p-s, e-p+1);
    }
  }

  //ajouter un / final si y'en a pas
  if (fdir.charAt(-1) != '/') fdir.append("/");
}
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UDir::Entry::Entry(const UDir& dir, const char* _name) :
name(new UStr(_name)) {
  size  = dir.statbuf.st_size;
  mode  = dir.statbuf.st_mode;
  mtime = dir.statbuf.st_mtime;
}

/* ==================================================== ======== ======= */

bool UDir::Entry::isDir()  const {
  return S_ISDIR(mode);
}

bool UDir::Entry::isFile()  const {
  return S_ISREG(mode);
}

bool UDir::Entry::isLink()  const {
  return S_ISLNK(mode);
}

bool UDir::Entry::isExec() const {
  return (mode & (S_IXUSR | S_IXGRP | S_IXOTH)) != 0;
}


UIma* UDir::Entry::getSmallIcon() const {
  if (S_ISDIR(mode)) {  // directory
    return &UPix::folder;
  }
  else if (S_ISREG(mode)) {          // regular file
    if (isExec()) return &UPix::ray; // si executable
    else return &UPix::doc;          // sinon
  }
  else if (S_ISLNK(mode)) {          // link
    return &UPix::rightarrow;
  }
  else return &UPix::question;	 // whatever
}


UIma* UDir::Entry::getLargeIcon() const {
  if (S_ISDIR(mode)) {  // directory
    return &UPix::folder;
  }
  else if (S_ISREG(mode)) {     // regular file
    if (isExec()) return &UPix::ray; // si executable
    else return &UPix::doc;       // si pas executable
  }
  else if (S_ISLNK(mode)) {     // link
    return &UPix::rightarrow;
  }
  else return &UPix::question;	 // whatever
}

/* ==================================================== ======== ======= */

UStr* UDir::Entry::createModeStr() const {
  char modes[11];

  if (S_ISDIR(mode)) {       // directory
    modes[0] = 'd';
  }
  else if (S_ISREG(mode)) {   // regular file
    modes[0] = '-';
  }
  else if (S_ISLNK(mode)) {     // link
    modes[0] = 'l';
  }
  else { 	            // whatever
    if (mode & S_IFIFO) //fifo (or "named pipe") special file
      modes[0] = 'p';
    else if (mode & S_IFCHR) //character special file;
      modes[0] = 'c';
    else if (mode & S_IFBLK) //block special file;
      modes[0] = 'b';
    else   // reste:  D (door) et s (AF_UNIX address family socket)
      modes[0] = '?';
  }

  modes[1] = (mode & S_IRUSR) ? 'r' : '-';
  modes[2] = (mode & S_IWUSR) ? 'w' : '-';
  modes[3] = (mode & S_IXUSR) ? 'x' : '-';

  modes[4] = (mode & S_IRUSR) ? 'r' : '-';
  modes[5] = (mode & S_IWUSR) ? 'w' : '-';
  modes[6] = (mode & S_IXGRP) ? 'x' : '-';

  modes[7] = (mode & S_IRUSR) ? 'r' : '-';
  modes[8] = (mode & S_IWUSR) ? 'w' : '-';
  modes[9] = (mode & S_IXOTH) ? 'x' : '-';

  modes[10] = 0; //!!
  return new UStr(modes);
}

UStr* UDir::Entry::createSizeStr() const {
  char buf[30];
  //sprintf(buf, "%ld", dir.statbuf.st_size);
  sprintf(buf, "%ld", size);
  return new UStr(buf);
}

UStr* UDir::Entry::createTimeStr() const {
  char buf[30];
  struct tm *timebuf = ::localtime(&mtime);
  strftime(buf, sizeof(buf), "%h/%d/%Y %H:%M", timebuf);
  // pas de free:  return value points to a statically allocated struct
  //free(timebuf);
  return new UStr(buf);
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */


