# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Small library for OAuth signing of urls."""

import dbus
import httplib2
import logging
import ubuntu_sso

from oauth import oauth
from dbus.mainloop.glib import DBusGMainLoop

DBusGMainLoop(set_as_default=True)

import socket
socket.setdefaulttimeout(5)

APP_NAME = "Ubuntu One"
HMAC_SHA1 = oauth.OAuthSignatureMethod_HMAC_SHA1()
PLAINTEXT = oauth.OAuthSignatureMethod_PLAINTEXT()


def _undbusify(value):
    """Convert dbus types back to native types."""
    for singleton in (None, True, False):
        if value == singleton:
            return singleton
    for val_type in (long, int, float, complex,
                     unicode, str,
                     list, tuple, dict, set):
        if isinstance(value, val_type):
            return val_type(value)
    raise TypeError(value)


def get_oauth_credentials():
    """Get dictionary of OAuth information from the keyring."""

    DBusGMainLoop(set_as_default=True)

    bus = dbus.SessionBus()
    proxy = bus.get_object(
        ubuntu_sso.DBUS_BUS_NAME, ubuntu_sso.DBUS_CRED_PATH,
        follow_name_owner_changes=True)
    logging.info(
        'get_oauth_data: asking for credentials from Ubuntu SSO. App name: %s',
        APP_NAME)
    oauth_data = dict(
        (_undbusify(k), _undbusify(v)) for k, v in
        proxy.find_credentials(APP_NAME).iteritems())
    if len(oauth_data) > 0:
        logging.info(
            'get_oauth_data: Got non emtpy credentials from Ubuntu SSO.')
    return dict([(key, oauth_data[key]) for key in (
        'consumer_key', 'consumer_secret', 'token', 'token_secret')])


def get_oauth_token(token, secret):
    """Turn token and secret into a Token."""
    token_string = 'oauth_token=%s&oauth_token_secret=%s' % (token, secret)
    return oauth.OAuthToken.from_string(token_string)


def get_oauth_request_header(consumer, access_token, http_url,
                             http_method='GET', signature_method=HMAC_SHA1,
                             parameters=None):
    """Get an oauth request header given the token and the url."""
    assert http_url.startswith("https")
    oauth_request = oauth.OAuthRequest.from_consumer_and_token(
        http_url=http_url,
        http_method=http_method,
        oauth_consumer=consumer,
        token=access_token,
        parameters=parameters)
    oauth_request.sign_request(signature_method, consumer, access_token)
    return oauth_request.to_header()


def request(url, sigmeth='HMAC_SHA1', http_method='GET', request_body=None,
            consumer_key=None, consumer_secret=None, access_token=None,
            token_secret=None):
    """Make an OAuth signed request."""
    # Set the signature method. This should be HMAC unless you have a jolly
    # good reason for it to not be.
    if sigmeth == "PLAINTEXT":
        signature_method = PLAINTEXT
    else:
        signature_method = HMAC_SHA1
    if access_token is None or consumer_key is None:
        credentials = get_oauth_credentials()
        access_token = credentials['token']
        token_secret = credentials['token_secret']
        consumer_key = credentials['consumer_key']
        consumer_secret = credentials['consumer_secret']
    token = get_oauth_token(access_token, token_secret)
    consumer = oauth.OAuthConsumer(consumer_key, consumer_secret)

    oauth_header = get_oauth_request_header(
        consumer, token, url, http_method, signature_method)
    http = httplib2.Http()
    return http.request(
        url, method=http_method, headers=oauth_header, body=request_body)
