/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotationSelection.h"

#include <core_api/DNATranslation.h>
#include <gobjects/AnnotationTableObject.h>
#include <util_text/TextUtils.h>


namespace GB2 {

static QList<Annotation*> emptyAnnotations;

int AnnotationSelectionData::getSelectedRegionsLen() const {
    if (locationIdx != -1) {
        return annotation->getLocation()[locationIdx].len;
    }
    int len = 0;
    foreach(const LRegion& r, annotation->getLocation()) {
        len+=r.len;
    }
    return len;
}

void AnnotationSelection::clear() {
    if (selection.isEmpty()) {
        return;
    }
	QList<Annotation*> tmpRemoved;
	foreach(const AnnotationSelectionData& asd, selection) {
		tmpRemoved.push_back(asd.annotation);
	}
	selection.clear();
	emit si_selectionChanged(this, emptyAnnotations, tmpRemoved);
}

void AnnotationSelection::removeObjectAnnotations(AnnotationTableObject* obj) {
	QList<Annotation*> removed;
	foreach(Annotation* a, obj->getAnnotations()) {
		for(int i=0, n = selection.size(); i<n; i++) {
			if (selection[i].annotation == a) {
				removed.append(a);
				selection[i].annotation = NULL;
				selection[i].locationIdx = 0;
			}
		}
	}
	selection.removeAll(AnnotationSelectionData(NULL, 0));
	emit si_selectionChanged(this, emptyAnnotations, removed);
}

void AnnotationSelection::addToSelection(Annotation* a, int locationIdx) {
	assert(locationIdx >=-1 && locationIdx < a->getLocation().size());
	bool replaced = false;
	bool changed = true;
	for(int i = 0, n = selection.size(); i<n; i++) {
		if (selection[i].annotation == a) {
			changed = selection[i].locationIdx != locationIdx;
			if (changed) {
				selection[i].locationIdx = locationIdx;
				replaced = true;
			}
			break;
		}
	}
	if (!changed) {
		return;
	}
	QList<Annotation*> tmp;
	tmp.append(a);
	if (replaced) {
		emit si_selectionChanged(this, tmp, tmp);
	} else {
		selection.append(AnnotationSelectionData(a, locationIdx));
		emit si_selectionChanged(this, tmp, emptyAnnotations);
	}
}


void AnnotationSelection::removeFromSelection(Annotation* a) {
	for(int i = 0, n = selection.size(); i<n; i++) {
		if (selection[i].annotation == a) {
			selection.removeAt(i);
			QList<Annotation*> tmp;
			tmp.append(a);
			emit si_selectionChanged(this, emptyAnnotations, tmp);
			break;
		}
	}
}

const AnnotationSelectionData* AnnotationSelection::getAnnotationData(const Annotation* a) const {
	foreach(const AnnotationSelectionData& asd, selection) {
		if (asd.annotation == a) {
			return &asd;
		}
	}
	return NULL;
}


void AnnotationSelection::getAnnotationSequence(QByteArray& res, const AnnotationSelectionData& sd, char gapSym, 
                                                const QByteArray& sequence, DNATranslation* complTT, DNATranslation* aminoTT) 
{
    int start = sd.locationIdx == -1 ? 0 : sd.locationIdx;
    int nLocations = sd.locationIdx == -1 ? sd.annotation->getLocation().size() : 1;
    LRegion sequenceRange(0, sequence.size());
    for (int j = start, last = start + nLocations; j < last; j++) {
        if (j!=start) {
            res.append(gapSym);
        }
        LRegion reg = sd.annotation->getLocation().at(j).intersect(sequenceRange);
        QByteArray partSeq(sequence.constData() + reg.startPos, reg.len);
        if (complTT!=NULL) {
            TextUtils::translate(complTT->getOne2OneMapper(), partSeq.data(), partSeq.size());
            TextUtils::reverse(partSeq.data(), partSeq.size());
        }
        if (aminoTT!=NULL) {
            aminoTT->translate(partSeq.data(), reg.len, partSeq.data(), reg.len);
            partSeq.resize(partSeq.size()/3);
        }
        res.append(partSeq);
    }
}



//////////////////////////////////////////////////////////////////////////
// Annotation group selection
static QList<AnnotationGroup*> emptyGroups;

void AnnotationGroupSelection::clear() {
    if (selection.isEmpty()) {
        return;
    }
    QList<AnnotationGroup*> tmpRemoved = selection;
    selection.clear();
    emit si_selectionChanged(this, emptyGroups, tmpRemoved);
}

void AnnotationGroupSelection::removeObjectGroups(AnnotationTableObject* obj) {
    QList<AnnotationGroup*> tmpRemoved;
    foreach(AnnotationGroup* g, selection) {
        if (g->getGObject() == obj) {
            tmpRemoved.append(g);
        }
    }
    foreach(AnnotationGroup* g, tmpRemoved) {
        selection.removeOne(g);
    }
    emit si_selectionChanged(this, emptyGroups, tmpRemoved);
}

void AnnotationGroupSelection::addToSelection(AnnotationGroup* g) {
    if (selection.contains(g)) {
        return;
    }
    selection.append(g);
    QList<AnnotationGroup*> tmpAdded;
    tmpAdded.append(g);
    emit si_selectionChanged(this, tmpAdded, emptyGroups);
}


void AnnotationGroupSelection::removeFromSelection(AnnotationGroup* g) {
    bool found = selection.removeOne(g);
    if (found) {
        QList<AnnotationGroup*> tmpRemoved;
        tmpRemoved.append(g);
        emit si_selectionChanged(this, emptyGroups, tmpRemoved);
    }
}


}//namespace
