/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtOpenGL>

#include "GraphicUtils.h"
#include "BioStruct3DColorScheme.h"
#include "TubeGLRenderer.h"


namespace GB2 { 

const QString TubeGLRenderer::ID(QObject::tr("Tubes"));

void TubeGLRenderer::drawTubes( const BioStruct3DColorScheme* colorScheme )
{
    GLUquadricObj *pObj;    // Quadric Object

    pObj = gluNewQuadric();
    gluQuadricNormals(pObj, GLU_SMOOTH);
    bool firstPass = true;
    float ribbonThickness = 0.3f;
    SharedAtom bufAtom;
    
    foreach(SharedAtom atom, tubeAtoms) {
        Color4f atomColor = colorScheme->getAtomColor(atom);
        Vector3D pos = atom.constData()->coord3d;
        glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, atomColor.getConstData());
        glDrawAtom(pObj, pos, ribbonThickness);
        
        if (!firstPass) {
            // Draw bonds only between atoms of the same molecular chain
            if (atom.constData()->chainIndex == bufAtom.constData()->chainIndex) {
                // ... and when they are sequential
                if (atom.constData()->residueIndex - bufAtom.constData()->residueIndex == 1) { 
                    Vector3D bufPos = bufAtom.constData()->coord3d;
					Color4f bufAtomColor = colorScheme->getAtomColor(bufAtom);
					glDrawHalfBond(pObj, bufPos, pos, ribbonThickness);
					glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, bufAtomColor.getConstData());
					glDrawHalfBond(pObj, pos, bufPos, ribbonThickness);
				 }
            }
        } else {
            firstPass = false;
        }

        bufAtom = atom;
    }       

}


void TubeGLRenderer::drawBioStruct3D()
{
	drawTubes(colorScheme);
}


TubeGLRenderer::TubeGLRenderer( const BioStruct3D& struc, const BioStruct3DColorScheme* s ) : BioStruct3DGLRenderer(struc,s)
{
    const char* alphaCarbonTag = "CA";
    const char* phosporTag = "P";
    
    foreach (const SharedMolecule mol, struc.moleculeMap) {
        foreach (const SharedAtom atom, mol->atomMap) {
            if ( (atom->name.trimmed() == alphaCarbonTag) || (atom->name.trimmed() == phosporTag)) {
                tubeAtoms.push_back(atom);
            }
        }
    }

}


} //namespace
