/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "HMMCalibrateDialogController.h"
#include "TaskLocalStorage.h"

#include <HMMIO.h>
#include <hmmer2/funcs.h>

#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>

#include <util_gui/DialogUtils.h>

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>


namespace GB2 {

HMMCalibrateDialogController::HMMCalibrateDialogController(QWidget* w) 
: QDialog(w)
{
    task = NULL;
    setupUi(this);

    connect(hmmFileButton, SIGNAL(clicked()), SLOT(sl_hmmFileButtonClicked()));
    connect(outFileButton, SIGNAL(clicked()), SLOT(sl_outFileButtonClicked()));
    connect(okButton, SIGNAL(clicked()), SLOT(sl_okButtonClicked()));
}


void HMMCalibrateDialogController::sl_hmmFileButtonClicked() {
    LastOpenDirHelper lod(HMMIO::HMM_ID);
    lod.url = QFileDialog::getOpenFileName(this, tr("select_file_with_hmm_model"), lod, HMMIO::getHMMFileFilter());
    if (lod.url.isEmpty()) {
        return;
    }
    hmmFileEdit->setText(QFileInfo(lod.url).absoluteFilePath());
}

void HMMCalibrateDialogController::sl_outFileButtonClicked() {
    LastOpenDirHelper lod(HMMIO::HMM_ID);
    lod.url= QFileDialog::getSaveFileName(this, tr("select_file_with_hmm_model"), lod, HMMIO::getHMMFileFilter());
    if (lod.url.isEmpty()) {
        return;
    }
    outFileEdit->setText(QFileInfo(lod.url).absoluteFilePath());
}

void HMMCalibrateDialogController::sl_okButtonClicked() {
    if (task != NULL) {
        accept(); //go to background
        return;
    }

    // try prepare model
   UHMMCalibrateSettings s;
   QString errMsg;

   QString inFile = hmmFileEdit->text();
   QString outFile = inFile;
   if (inFile.isEmpty() && !QFileInfo(inFile).exists()) {
       errMsg = tr("illegal_in_file_name");
       hmmFileEdit->setFocus();
   }

   if (expertGroupBox->isChecked() && errMsg.isEmpty()) {
       if (fixedBox->value() < 0) {
           errMsg = tr("illegal fixed value");
           fixedBox->setFocus();
       } else {
           s.fixedlen = fixedBox->value();
       }

       s.lenmean = meanBox->value();
       s.nsample = numBox->value();
       s.lensd  = (float)sdBox->value();
       if (seedBox->value()!=0) {
           s.seed = seedBox->value();
       }
   }

   if (outputGroupBox->isChecked() && errMsg.isEmpty()) {
        outFile = outFileEdit->text();        
        if (outFile.isEmpty()) {
            errMsg = tr("illegal_out_file_name");
            outFileEdit->setFocus();
        }
   }

   if (!errMsg.isEmpty())  {
       QMessageBox::critical(this, tr("error"), errMsg);
       return;
   }

   // run task
   task = new HMMCalibrateToFileTask(inFile, outFile, s);
   task->setReportingEnabled(true);
   connect(task, SIGNAL(si_stateChanged()), SLOT(sl_onStateChanged()));
   connect(task, SIGNAL(si_progressChanged()), SLOT(sl_onProgressChanged()));
   AppContext::getTaskScheduler()->registerTopLevelTask(task);
   statusLabel->setText(tr("starting_calibration_process"));

   //update buttons
   okButton->setText(tr("back_button"));
   cancelButton->setText(tr("cancel_button"));

   // new default behavior: hide dialog and use taskview to track the progress and results
   accept(); //go to background
}


void HMMCalibrateDialogController::sl_onStateChanged() {
    Task* t = qobject_cast<Task*>(sender());
    assert(task!=NULL);
    if (task!=t || t->getState() != Task::State_Finished) {
        return;
    }
    task->disconnect(this);
    const TaskStateInfo& si = task->getStateInfo();
    if (si.hasErrors()) {
        statusLabel->setText(tr("calibration_finished_with_errors_%1").arg(si.error));
    } else if (task->isCanceled()) {
        statusLabel->setText(tr("calibration_canceled"));
    } else {
        statusLabel->setText(tr("calibration_finished_successfuly"));
    }
    okButton->setText(tr("ok_button"));
    cancelButton->setText(tr("close_button"));

    task = NULL;
}

void HMMCalibrateDialogController::sl_onProgressChanged() {
    assert(task==sender());
    statusLabel->setText(tr("progress_%1%").arg(task->getProgress()));
}

void HMMCalibrateDialogController::reject() {
    if (task!=NULL) {
        task->cancel();
    }
    QDialog::reject();
}


//////////////////////////////////////////////////////////////////////////
// task

HMMCalibrateTask::HMMCalibrateTask(plan7_s* hmm, const UHMMCalibrateSettings& s) 
	: HMMCalibrateAbstractTask("", hmm, s)
{
    setTaskName(tr("HMM calibrate '%1'").arg(hmm->name));
    tpm = Task::Progress_Manual;
}


void HMMCalibrateTask::run() {
    TaskLocalData::initializeHMMContext(getTaskId());
    _run();
    TaskLocalData::freeHMMContext();
}

void HMMCalibrateTask::_run()  {
    try {
        UHMMCalibrate::calibrate(hmm, settings, stateInfo);
    } catch (HMMException e) {
        stateInfo.error = e.error;
    }
}

//parallel calibrate task
HMMCalibrateParallelTask::HMMCalibrateParallelTask(plan7_s* hmm, const UHMMCalibrateSettings& s) 
	: HMMCalibrateAbstractTask("", hmm, s)
{
    setTaskName(tr("HMM calibrate '%1'").arg(hmm->name));
    tpm = Task::Progress_Manual;
	wpool = new WorkPool_s();
	for(int i=0;i<settings.nThreads;i++){
		HMMCalibrateParallelSubTask *task = new HMMCalibrateParallelSubTask(wpool);
		addSubTask(task);
		calibrateSubtasks << task;
	}
}

void HMMCalibrateParallelTask::run() {
    TaskLocalData::initializeHMMContext(getTaskId());
    _run();
    TaskLocalData::freeHMMContext();
}

void HMMCalibrateParallelTask::_run() {
    //todo: clean resources on exception!

	//get HMMERTaskLocalData
	HMMERTaskLocalData *tls = getHMMERTaskLocalData();
	alphabet_s &al = tls->al;

	SetAlphabet(hmm->atype);
	sre_srandom(settings.seed);
	

    //   calibrate
    assert(hmm!=NULL);
	hist = AllocHistogram(-200, 200, 100);
	assert(wpool!=NULL);
	assert(wpool->al!=NULL);
	wpool->al = &al;
	wpool->fixedlen = settings.fixedlen;
	wpool->hist = hist;
	wpool->hmm = hmm;
	wpool->lenmean = settings.lenmean;
	wpool->lensd = settings.lensd;
	wpool->nsample = settings.nsample;
	wpool->nseq = 0;
	wpool->randomseq = new float[MAXABET];
	assert(wpool->randomseq!=NULL);
	wpool->max_score = -FLT_MAX;
	wpool->progress = &(stateInfo.progress);
	
	float  p1;
	P7Logoddsify(hmm, TRUE);
    P7DefaultNullModel(wpool->randomseq, &p1);
	
    if (stateInfo.hasErrors() || stateInfo.cancelFlag) {
        FreeHistogram(hist);
        delete[] wpool->randomseq;
        delete wpool;
        return;
    }
	wpool->sem.release(settings.nThreads);
	subtaskWaitSem.acquire();

	if (!stateInfo.hasErrors()) {
		if (! ExtremeValueFitHistogram(hist, TRUE, 9999.)) {
			stateInfo.error = "fit failed; num sequences may be set too small?\n";
		} else {
            hmm->flags |= PLAN7_STATS;
			hmm->mu     = hist->param[EVD_MU];
			hmm->lambda = hist->param[EVD_LAMBDA];
		}
	}

	FreeHistogram(hist);
    delete[] wpool->randomseq;
	delete wpool;
}

QList<Task*> HMMCalibrateParallelTask::onSubTaskFinished(Task* subTask)
{
	QList<Task*> res;
	
	calibrateSubtasks.removeOne((HMMCalibrateParallelSubTask*)subTask);
	if(subTask->hasErrors()){
		stateInfo.error = subTask->getError();
		return res;
	}
	if(calibrateSubtasks.isEmpty()){
		subtaskWaitSem.release();
	}
	return res;
}

//subtasks
HMMCalibrateParallelSubTask::HMMCalibrateParallelSubTask(WorkPool_s *_wpool) 
:Task(tr("calibrate_hmm_model_subtask"), TaskFlag_DeleteWhenFinished), wpool(_wpool)
{
    tpm = Task::Progress_Manual;
}


void HMMCalibrateParallelSubTask::run() {
    assert(getParentTask()!=NULL);
    wpool->sem.acquire();
    TaskLocalData::bindToHMMContext(getParentTask()->getTaskId());
    _run();
    TaskLocalData::detachFromHMMContext();
}

void HMMCalibrateParallelSubTask::_run() {
	TaskStateInfo subInfo;
	assert(wpool->hmm!=NULL);
    try {
        UHMMCalibrate::calibrateParallelWorker(wpool, stateInfo);
    } catch (HMMException e) {
        stateInfo.error = e.error;
    }
}

HMMCalibrateToFileTask::HMMCalibrateToFileTask(const QString& _inFile, const QString& _outFile, const UHMMCalibrateSettings& s)
: Task("", TaskFlags_NR_DWF_SSSOR | TaskFlags_FAIL_OSCOE), hmm(NULL), inFile(_inFile), outFile(_outFile), settings(s) 
{
    setVerboseLogMode(true);
    QString tn = tr("HMM calibrate '%1'").arg(QFileInfo(inFile).fileName());
    setTaskName(tn);
    readTask = NULL;
    calibrateTask = NULL;
}


void HMMCalibrateToFileTask::prepare() {
    readTask = new HMMReadTask(inFile);
    readTask->setSubtaskProgressWeight(0);
    addSubTask(readTask);
}

QList<Task*> HMMCalibrateToFileTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    
    if (hasErrors()) {
        return res;
    }
    if (subTask == readTask) {
        hmm = readTask->getHMM();
        assert(hmm!=NULL);
        if (settings.nThreads == 1) {
            calibrateTask = new HMMCalibrateTask(hmm, settings);
        } else {
            calibrateTask = new HMMCalibrateParallelTask(hmm, settings);
        }
        res.append(calibrateTask);
    } else if (subTask == calibrateTask) {
        Task* t = new HMMWriteTask(outFile, hmm);
        t->setSubtaskProgressWeight(0);
        res.append(t);
    }
    return res;
}

QString HMMCalibrateToFileTask::generateReport() const {
    QString res;
    res+="<table>";
    res+="<tr><td width=200><b>" + tr("Source profile") + "</b></td><td>" + QFileInfo(inFile).absoluteFilePath() + "</td></tr>";

    if (hasErrors() || isCanceled()) {
        res+="<tr><td width=200><b>" + tr("Task was not finished") + "</b></td><td></td></tr>";
        res+="</table>";
        return res;
    }

    res+="<tr><td><b>" + tr("Result profile") + "</b></td><td>" + QFileInfo(outFile).absoluteFilePath() + "</td></tr>";
    res+="<tr><td><b>" + tr("Expert options") + "</b></td><td></td></tr>";
    
    res+="<tr><td><b>" + tr("Number of random sequences to sample") + "</b></td><td>"+QString::number(settings.nsample)+"</td></tr>";
    res+="<tr><td><b>" + tr("Random number seed") + "</b></td><td>"+QString::number(settings.seed)+"</td></tr>";
    res+="<tr><td><b>" + tr("Mean of length distribution") + "</b></td><td>"+QString::number(settings.lenmean)+"</td></tr>";
    res+="<tr><td><b>" + tr("Standard deviation of length distribution") + "</b></td><td>"+QString::number(settings.lensd)+"</td></tr>";
    
    res+="<tr><td><b>" + tr("Calculated evidence (mu , lambda)") + "</b></td><td>"+QString::number(hmm->mu, 'f', 6) + ", " + QString::number(hmm->lambda, 'f', 6) + "</td></tr>";
    
    res+="</table>";
    return res;
}

}//namespace
