/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MDIManagerImpl.h"

#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <core_api/Log.h>

#include <QtCore/QSet>
#include <QtGui/QAction>
#include <QtGui/QMenu>
#include <QtGui/QHBoxLayout>
#include <QtGui/QToolBar>

namespace GB2 {

static LogCategory log(ULOG_CAT_USER_INTERFACE);

#define SETTINGS_DIR QString("main_window/mdi/")

MWMDIManagerImpl::~MWMDIManagerImpl() {
}

void MWMDIManagerImpl::prepareGUI() {
	mdiContentOwner = NULL;

	connect(mdiArea, SIGNAL(subWindowActivated(QMdiSubWindow*)), SLOT(sl_onSubWindowActivated(QMdiSubWindow*)));
	
	windowMapper = new QSignalMapper(this);
	connect(windowMapper, SIGNAL(mapped(QWidget *)), this, SLOT(sl_setActiveSubWindow(QWidget *)));


	//prepare Window menu
	closeAct = new QAction(tr("close_active_view"), this);
    closeAct->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_W));
	closeAct->setStatusTip(tr("close_active_view_statustip"));
	connect(closeAct, SIGNAL(triggered()), mdiArea, SLOT(closeActiveSubWindow()));

	closeAllAct = new QAction(tr("close_all_mdi_windows"), this);
	closeAllAct->setStatusTip(tr("close_all_mdi_windows_statustip"));
	connect(closeAllAct, SIGNAL(triggered()), mdiArea, SLOT(closeAllSubWindows()));

    tileAct = new QAction(QIcon(":ugene/images/window_tile.png"), tr("tile_mdi_windows"), this);
	tileAct->setStatusTip(tr("tile_mdi_windows_statustip"));
	connect(tileAct, SIGNAL(triggered()), mdiArea, SLOT(tileSubWindows()));

	cascadeAct = new QAction(QIcon(":ugene/images/window_cascade.png"), tr("cascade_mdi_windows"), this);
	cascadeAct->setStatusTip(tr("cascade_mdi_windows_statustip"));
	connect(cascadeAct, SIGNAL(triggered()), mdiArea, SLOT(cascadeSubWindows()));

	nextAct = new QAction(QIcon(":ugene/images/window_next.png"), tr("next_mdi_window"), this);
	nextAct->setStatusTip(tr("next_mdi_window_statustip"));
	nextAct->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_Tab));
	connect(nextAct, SIGNAL(triggered()), mdiArea, SLOT(activateNextSubWindow()));

	previousAct = new QAction(QIcon(":ugene/images/window_prev.png"), tr("previous_mdi_window"), this);
	previousAct->setStatusTip(tr("previous_mdi_window_statustip"));
	previousAct->setShortcut(QKeySequence(Qt::CTRL + Qt::SHIFT + Qt::Key_Tab));
	connect(previousAct, SIGNAL(triggered()),mdiArea, SLOT(activatePreviousSubWindow()));

	separatorAct = new QAction("-", this);
	separatorAct->setSeparator(true);

	defaultIsMaximized = AppContext::getSettings()->getValue(SETTINGS_DIR + "maximized", true).toBool();

	QMenu* windowMenu = mw->getMenuManager()->getTopLevelMenu(MWMENU_WINDOW);
	connect(windowMenu, SIGNAL(aboutToShow()), this, SLOT(sl_updateWindowMenu()));

	updateState();
    clearMDIContent(true);
}

bool MWMDIManagerImpl::eventFilter(QObject *obj, QEvent *event) {
	QEvent::Type t = event->type();
	if ( t == QEvent::Close) {
		QMdiSubWindow* qw = qobject_cast<QMdiSubWindow*>(obj);
		assert(qw);
		MDIItem* i = getMDIItem(qw);
        emit si_windowClosing(i->w);
        if (!onCloseEvent(i->w)) {
            event->ignore();
            return true;
        }
        log.details(tr("Closing window %1").arg(i->w->getWindowName()));

        items.removeAll(i);
		delete i;
		updateState();
		clearMDIContent(true);
	} else if (t == QEvent::WindowStateChange) {
		QMdiSubWindow* qw = qobject_cast<QMdiSubWindow*>(obj);
		assert(qw);
		defaultIsMaximized = qw->isMaximized();
	}
	return QObject::eventFilter(obj, event);
}


void MWMDIManagerImpl::updateState() {
	updateActions();
	sl_updateWindowMenu();
	
	AppContext::getSettings()->setValue(SETTINGS_DIR + "maximized", defaultIsMaximized);
}

void MWMDIManagerImpl::updateActions() {
	bool hasMDIWindows = !items.empty();

	closeAct->setEnabled(hasMDIWindows);
	closeAllAct->setEnabled(hasMDIWindows);
	tileAct->setEnabled(hasMDIWindows);
	cascadeAct->setEnabled(hasMDIWindows);
	nextAct->setEnabled(hasMDIWindows);
	previousAct->setEnabled(hasMDIWindows);
	separatorAct->setVisible(hasMDIWindows);
}

void MWMDIManagerImpl::sl_updateWindowMenu() {
	QMenu* windowMenu = mw->getMenuManager()->getTopLevelMenu(MWMENU_WINDOW);
	windowMenu->clear();//TODO: avoid cleaning 3rd party actions
	windowMenu->addAction(closeAct);
	windowMenu->addAction(closeAllAct);
	windowMenu->addSeparator();
	windowMenu->addAction(tileAct);
	windowMenu->addAction(cascadeAct);
	windowMenu->addSeparator();
	windowMenu->addAction(nextAct);
	windowMenu->addAction(previousAct);
	windowMenu->addAction(separatorAct);

	separatorAct->setVisible(!items.isEmpty());

	MDIItem* currentItem = getCurrentMDIItem();
	for (int i = 0; i < items.size(); ++i) {
		MDIItem* item = items.at(i);
		QString text;
		if (i < 9) {
			text = QString("&%1 %2").arg(i + 1).arg(item->w->getWindowName());
		} else {
			text = QString("%1 %2").arg(i + 1).arg(item->w->getWindowName());
		}
		QAction *action  = windowMenu->addAction(text);
		action->setCheckable(true);
		action->setChecked(item == currentItem);
		connect(action, SIGNAL(triggered()), windowMapper, SLOT(map()));
		windowMapper->setMapping(action, item->qw);
	}
}

MDIItem* MWMDIManagerImpl::getCurrentMDIItem() const {
	if (QMdiSubWindow *currentSubWindow = mdiArea->currentSubWindow()) {
		return getMDIItem(currentSubWindow);
	}
	return NULL;
}


void MWMDIManagerImpl::addMDIWindow(MWMDIWindow* w) {
	bool contains = getWindowById(w->getId())!=NULL;
	if (contains) {
        assert(0); //must never happen
		return;
	}
	QMdiSubWindow* qw = mdiArea->addSubWindow(w);
	qw->setWindowTitle(w->getWindowName());
    qw->setWindowIcon(QIcon(":/ugene/images/ugene_16.png")); //todo: get icon from w
	MDIItem* i = new MDIItem(w, qw);
	items.append(i);
	qw->installEventFilter(this);

    log.details(tr("Adding window: %1").arg(w->getWindowName()));

	updateState();

	emit si_windowAdded(w);

	if (items.count() == 1 && defaultIsMaximized) {
		qw->showMaximized();
	} else {
		qw->show();	
	}
	qw->raise();
}

QList<MWMDIWindow*> MWMDIManagerImpl::getWindows() const {
	QList<MWMDIWindow*> res;
	foreach(MDIItem* i, items) {
		res.append(i->w);
	}
	return res;
}


bool MWMDIManagerImpl::closeMDIWindow(MWMDIWindow* w) {
	MDIItem* i = getMDIItem(w);
	assert(i);
	return i->qw->close();
}

MWMDIWindow* MWMDIManagerImpl::getWindowById(int id) const {
	MDIItem* i = getMDIItem(id);
	return i == NULL ? NULL: i->w;
}

MDIItem* MWMDIManagerImpl::getMDIItem(int id) const {
	foreach(MDIItem* i, items) {
		if (i->w->getId() == id) {
			return i;
		}
	}
	return NULL;
}

MDIItem* MWMDIManagerImpl::getMDIItem(MWMDIWindow* w) const {
	foreach(MDIItem* i, items) {
		if (i->w == w) {
			return i;
		}
	}
	return NULL;
}

MDIItem* MWMDIManagerImpl::getMDIItem(QMdiSubWindow* qw) const {
	foreach(MDIItem* i, items) {
		if (i->qw == qw) {
			return i;
		}
	}
	return NULL;
}

Qt::WindowStates MWMDIManagerImpl::getWindowState(MWMDIWindow* w) const {
	MDIItem* i = getMDIItem(w);
	return i==NULL ? Qt::WindowNoState : i->qw->windowState();
}

void MWMDIManagerImpl::activateWindow(MWMDIWindow* w)  {
	MDIItem* i = getMDIItem(w);
	assert(i);
	if (i==0) {
		return;
	}
	mdiArea->setActiveSubWindow(i->qw);
	updateState();
}

void MWMDIManagerImpl::sl_setActiveSubWindow(QWidget *w) {
	if (!w) {
		return;
	}
	mdiArea->setActiveSubWindow(qobject_cast<QMdiSubWindow *>(w));
}

void MWMDIManagerImpl::clearMDIContent(bool addCloseAction) {
	//clear toolbar
	mdiContentOwner = NULL;

    QToolBar* tb = mw->getToolBarManager()->getToolbar(MWTOOLBAR_ACTIVEMDI);
    QMenu* m = mw->getMenuManager()->getTopLevelMenu(MWMENU_ACTIONS);

    //delete submenus inserted to toolbar and menu
    //todo: provide a flag to enable/disable this behavior for MDI window
    QList<QAction*> allMDIActions = tb->actions() + m->actions();
    QSet<QMenu*> toDel;
    foreach(QAction* ma, allMDIActions) {
        QMenu* am = ma->menu();
        if (am!=NULL) {
            toDel.insert(am);
        }
    }
    
	tb->clear();
	m->clear();

    foreach(QMenu* mtd, toDel) {
        delete mtd;
    }

    if (addCloseAction) {
        m->addAction(closeAct);
    }
}

void MWMDIManagerImpl::sl_onSubWindowActivated(QMdiSubWindow *w) {
	MDIItem* mdiItem = getMDIItem(w);
	if (mdiItem == mdiContentOwner || mdiItem == NULL) {
		return;
	}
	clearMDIContent(false);

	QToolBar* tb = mw->getToolBarManager()->getToolbar(MWTOOLBAR_ACTIVEMDI);
	mdiContentOwner = mdiItem;
	mdiContentOwner->w->setupMDIToolbar(tb);
	
	QMenu* m = mw->getMenuManager()->getTopLevelMenu(MWMENU_ACTIONS);
	mdiContentOwner->w->setupViewMenu(m);
	m->addAction(closeAct);

}


MWMDIWindow* MWMDIManagerImpl::getActiveWindow() const {
	MDIItem* i = getCurrentMDIItem();
	if (i == NULL) {
		return NULL;
	}
	return i->w;
}

}//namespace
