/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ProjectServiceImpl.h"

#include "ProjectImpl.h"
#include "ProjectLoaderImpl.h"
#include "ProjectTasks.h"

#include "../AppContextImpl.h"
#include "../CorePlugin.h"

#include <core_api/MainWindow.h>
#include <core_api/Settings.h>
#include <core_api/Log.h>
#include <util_gui/GUIUtils.h>

#include <QtGui/QMenu>
#include <QtGui/QToolBar>

namespace GB2 {

#define SETTINGS_DIR QString("project_loader/")

static LogCategory log(ULOG_CAT_CORE_SERVICES);

ProjectServiceImpl::ProjectServiceImpl(Plugin* p, Project* _pr) 
: ProjectService(p, tr("project_sname"), tr("project_sdesc"))
{
    saveAction = NULL;
    saveAsAction = NULL;
	closeProjectAction = NULL;

	pr = _pr;
	assert(pr!=NULL);
}

ProjectServiceImpl::~ProjectServiceImpl() {
    //TODO: redesign real-time service registration/unregistration-> synchronize with plugin list
    CorePlugin* corePlug = (CorePlugin*)getPlugin();
    if (!corePlug->shutdown) {
        corePlug->removeService(this); 
    }
	delete pr;
}

Task* ProjectServiceImpl::saveProjectTask(SaveProjectTaskKind k) {
	return new SaveProjectTask(k);
}

Task* ProjectServiceImpl::closeProjectTask() {
	return new CloseProjectTask();	
}

Task* ProjectServiceImpl::createServiceEnablingTask() {
	return new ProjectServiceEnableTask(this);
}

Task* ProjectServiceImpl::createServiceDisablingTask() {
	return new ProjectServiceDisableTask(this);
}

void ProjectServiceImpl::sl_closeProject() {
	AppContext::getTaskScheduler()->registerTopLevelTask(closeProjectTask());
}

void ProjectServiceImpl::sl_save() {
	AppContext::getTaskScheduler()->registerTopLevelTask(saveProjectTask(SaveProjectTaskKind_SaveProjectAndDocuments));
}

void ProjectServiceImpl::sl_saveAs() {
    ProjectDialogController d(ProjectDialogController::Save_Project);
    int rc = d.exec();
    AppContext::getSettings()->setValue(SETTINGS_DIR + "last_dir",d.projectFolderEdit->text());

    if (rc == QDialog::Rejected) {
        return;
    }
    AppContext::getProject()->setProjectName(d.projectNameEdit->text());
    
    QString fileName = d.projectFolderEdit->text() + "/" + d.projectFileEdit->text();
    if (!fileName.endsWith(PROJECTFILE_EXT)) {
        fileName.append(PROJECTFILE_EXT);
    }
    AppContext::getProject()->setProjectURL(fileName);

    AppContext::getTaskScheduler()->registerTopLevelTask(saveProjectTask(SaveProjectTaskKind_SaveProjectAndDocuments));
}

//////////////////////////////////////////////////////////////////////////
/// Service tasks

ProjectServiceEnableTask::ProjectServiceEnableTask(ProjectServiceImpl* _psi) 
: Task(tr("project_enable_task"), TaskFlags_NR_DWF), psi(_psi)
{}

Task::ReportResult ProjectServiceEnableTask::report() {
    AppContextImpl::getApplicationContext()->setProjectService(psi);
	AppContextImpl::getApplicationContext()->setProject(psi->getProject());

	assert(psi->saveAction == NULL && psi->closeProjectAction == NULL);

    psi->saveAction = new QAction(QIcon(":ugene/images/project_save.png"), tr("project_save"), psi);
	psi->saveAction->setObjectName(ACTION_PROJECTSUPPORT__SAVE_PROJECT);
	psi->saveAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_S));
	psi->saveAction->setShortcutContext(Qt::WindowShortcut);
	connect(psi->saveAction, SIGNAL(triggered()), psi, SLOT(sl_save()));

    psi->saveAsAction = new QAction(tr("project_save_as"), psi);
    connect(psi->saveAsAction, SIGNAL(triggered()), psi, SLOT(sl_saveAs()));

	psi->closeProjectAction = new QAction(tr("project_close"), psi);
	psi->closeProjectAction->setObjectName(ACTION_PROJECTSUPPORT__CLOSE_PROJECT);
	psi->closeProjectAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_Q));
	psi->closeProjectAction->setShortcutContext(Qt::WindowShortcut);
	connect(psi->closeProjectAction, SIGNAL(triggered()), psi, SLOT(sl_closeProject()));

	MainWindow* mw = AppContext::getMainWindow();

    MWMenuManager* mm = mw->getMenuManager();
    QMenu* fileMenu = mm->getTopLevelMenu(MWMENU_FILE);
    QAction* beforeAction= GUIUtils::findActionAfter(fileMenu->actions(), ACTION_PROJECTSUPPORT__OPEN_PROJECT);
    fileMenu->insertAction(beforeAction,  psi->saveAction);
    fileMenu->insertAction(beforeAction,  psi->saveAsAction);
    fileMenu->insertAction(beforeAction,  psi->closeProjectAction);
	
	MWToolBarManager* tm = mw->getToolBarManager();
	QToolBar* tb = tm->getToolbar(MWTOOLBAR_MAIN);
    beforeAction= GUIUtils::findActionAfter(tb->actions(), ACTION_PROJECTSUPPORT__OPEN_PROJECT);
    tb->insertAction(beforeAction,  psi->saveAction);
	
	return ReportResult_Finished;
}


ProjectServiceDisableTask::ProjectServiceDisableTask(ProjectServiceImpl* _psi) 
: Task(tr("project_disable_task"), TaskFlags_NR_DWF), psi(_psi)
{}

Task::ReportResult ProjectServiceDisableTask::report() {
	AppContextImpl::getApplicationContext()->setProject(NULL);
	AppContextImpl::getApplicationContext()->setProjectService(NULL);

	delete psi->saveAction;
	psi->saveAction = NULL;
	
	delete psi->closeProjectAction;
	psi->closeProjectAction = NULL;

	return ReportResult_Finished;
}

} //namespace
