/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SubstMatrixRegistry.h"

#include <QDir>
#include <QFile>
#include <QTextStream>
#include <QApplication>

#include <core_api/Log.h>
#include <util_text/TextUtils.h>
#include <document_format/SubstMatrixFile.h>

#include <QObject>

namespace GB2 {

static LogCategory log(ULOG_CAT_SW);

SubstMatrixRegistry::SubstMatrixRegistry(QObject* own) :
	QObject(own)
{
    QString wmDirName = QDir::searchPaths( PATH_PREFIX_DATA ).first() + "/weight_matrix";
    QDir wmDir(wmDirName);
	QStringList ls = wmDir.entryList(QStringList("*.txt"));
	for (int i = 0; i < ls.size(); i++) {
		registerSubstMatrix(wmDirName + "/" + ls.at(i));
	}
}

SubstMatrixRegistry::~SubstMatrixRegistry() {
	mtx_list_iterator i = mtx_list.begin();
	while (mtx_list.end() != i) {
		delete (*i).mtx;
		i = mtx_list.erase(i);
	}
}

const SubstMatrix* SubstMatrixRegistry::getSubstMatrix(const QString& mtx_id) {
	const SubstMatrix* mtx = 0;
	mtx_list_iterator i = findSubstMatrix(mtx_id);
	if (mtx_list.end() != i) {
		mtx = (*i).mtx;
	}
	return mtx;
}

QStringList SubstMatrixRegistry::findMatricesByAlphabet(DNAAlphabet* alph) {
	QStringList res;
	if (0 == alph) {
		return res;
	}
	
	QByteArray alphArr = alph->getAlphabetChars();

	foreach (const NamedSubstMatrix& m, mtx_list) {
		const SubstMatrix* mtx = m.mtx;
		const DNAAlphabet* mtxAlph = mtx->getAlphabet();
		
		const QBitArray& mtxAlphMap = mtxAlph->getMap();
		bool contain = TextUtils::fits(mtxAlphMap, alphArr.constData(), alphArr.length());
		if (contain && (mtxAlph->getType() == alph->getType())) {
			res.append(m.mtx_id);
		}
	}
	return res;
}

bool SubstMatrixRegistry::registerSubstMatrix(const QString& fileName) {
	SubstMatrix* mtx = SubstMatrixFile::read(fileName);
	if (0 == mtx) {
		return false;
	}
	int index = 0;
	QString mtx_id;
	if (-1 == (index = fileName.lastIndexOf("/"))) {
		mtx_id = fileName;
	} else {
		mtx_id = fileName.mid(index+1);
	}

	QMutexLocker locker(&mutex);
	if (hadRegistered(mtx_id)) {
		log.error(tr("Matrix %1 already registered.").arg(mtx_id));
		return false;
	}
	
	NamedSubstMatrix m;
	m.mtx	 = mtx;
	m.mtx_id = mtx_id;
	mtx_list.append(m);
	return true;
}

mtx_list_iterator SubstMatrixRegistry::findSubstMatrix(const QString& mtx_id) {
	mtx_list_iterator i;
	for (i = mtx_list.begin(); i != mtx_list.end(); ++i) {
		if ((*i).mtx_id == mtx_id) {
			return i;
		}
	}
	return mtx_list.end();
}

bool SubstMatrixRegistry::hadRegistered(const QString& mtx_id) {
	return mtx_list.end() != findSubstMatrix(mtx_id);
}

void SubstMatrixRegistry::dump() const {
	mtx_list_const_iterator i;
	for (i = mtx_list.constBegin(); i != mtx_list.constEnd(); ++i) {
		log.trace(QString("Matrix name: %1\n").arg((*i).mtx_id.toAscii().data()));
		(*i).mtx->dump();
	}
}

QStringList SubstMatrixRegistry::getListMatrixNames() {
	QStringList res;
	foreach (const NamedSubstMatrix m, mtx_list) {
		res.append(m.mtx_id);
	}
	return res;
}

} // namespace
