/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotatedDNAViewTasks.h"
#include "AnnotatedDNAView.h"
#include "AnnotatedDNAViewFactory.h"
#include "AnnotatedDNAViewState.h"

#include "GSequenceLineView.h"

#include <gobjects/DNASequenceObject.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/GObjectRelationRoles.h>
#include <gobjects/GObjectUtils.h>

#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentModel.h>

#include <QtCore/QSet>

namespace GB2 {

/* TRANSLATOR GB2::AnnotatedDNAView */
/* TRANSLATOR GB2::ObjectViewTask */

static LogCategory log(ULOG_CAT_ADV);

//////////////////////////////////////////////////////////////////////////
/// open new view

//opens a single view for all sequence object in the list of sequence objects related to the objects in the list
OpenAnnotatedDNAViewTask::OpenAnnotatedDNAViewTask(const QList<GObject*>& objects) 
: ObjectViewTask(AnnotatedDNAViewFactory::ID)
{
    //  remember only sequence objects -> other added automatically
    //  load all objects

    QSet<Document*> docsToLoadSet;
    QSet<GObject*>  refsAdded;
    QList<GObject*> allSequenceObjects = GObjectUtils::findAllObjects(UOF_LoadedAndUnloaded, GObjectTypes::DNA_SEQUENCE);
    foreach(GObject* obj, objects) {
        log.trace("Object to open sequence view: '" + obj->getGObjectName()+"'");
        Document* doc = obj->getDocument();
        if (!doc->isLoaded()) {
            docsToLoadSet.insert(doc);
        }
        if (GObjectUtils::hasType(obj, GObjectTypes::DNA_SEQUENCE)) {
            sequenceObjectRefs.append(GObjectReference(doc->getURL(), obj->getGObjectName(), GObjectTypes::DNA_SEQUENCE));
            refsAdded.insert(obj);
            continue;
        }
        
    
        //look for sequence object using relations
        QSet<GObject*> objWithSeqRelation = GObjectUtils::selectRelations(obj, GObjectTypes::DNA_SEQUENCE, 
                                        GObjectRelationRole::SEQUENCE, allSequenceObjects, UOF_LoadedAndUnloaded);

        foreach(GObject* robj, objWithSeqRelation) {
            if (!GObjectUtils::hasType(robj, GObjectTypes::DNA_SEQUENCE)) {
                continue;
            }
            if (refsAdded.contains(robj)) {
                continue;
            }
            Document* rdoc = robj->getDocument();
            if (!rdoc->isLoaded()) {
                docsToLoadSet.insert(rdoc);
            }
            refsAdded.insert(robj);
            sequenceObjectRefs.append(GObjectReference(rdoc->getURL(), robj->getGObjectName(), GObjectTypes::DNA_SEQUENCE));

        }
    }
    foreach(Document* doc, docsToLoadSet) {
        log.trace("Document to load: '" + doc->getURL()+"'");
        documentsToLoad.append(doc);
    }
}

#define MAX_SEQ_OBJS_PER_VIEW 20

void OpenAnnotatedDNAViewTask::open() {
	if (stateInfo.hasErrors() || sequenceObjectRefs.isEmpty()) {
		return;
	}
    QList<DNASequenceObject*> seqObjects;
    QList<GObject*> allSequenceObjects = GObjectUtils::findAllObjects(UOF_LoadedOnly, GObjectTypes::DNA_SEQUENCE);
    foreach(const GObjectReference& r, sequenceObjectRefs) {
        GObject* obj = GObjectUtils::selectObjectByReference(r, allSequenceObjects, UOF_LoadedOnly);
        DNASequenceObject* seqObj = qobject_cast<DNASequenceObject*>(obj);
        if (seqObj!=NULL) {
            seqObjects.append(seqObj);
            if (seqObjects.size() > MAX_SEQ_OBJS_PER_VIEW) {
                log.details(tr("Maximum number of objects per view reached: %1").arg(MAX_SEQ_OBJS_PER_VIEW));
                break;
            }
        } else {
            log.details(tr("Sequence object not available! URL %1, name %2").arg(r.docUrl).arg(r.objName));
        }
    }
    if (seqObjects.isEmpty()) { //object was removed asynchronously with the task
        stateInfo.setError(tr("No sequence objects found"));
        return;
    }
    viewName = GObjectViewUtils::genUniqueViewName(seqObjects.first()->getDocument(), seqObjects.first());
    AnnotatedDNAView* v = new AnnotatedDNAView(viewName, seqObjects);
	GObjectViewWindow* w = new GObjectViewWindow(v, viewName, false);
	MWMDIManager* mdiManager = 	AppContext::getMainWindow()->getMDIManager();
	mdiManager->addMDIWindow(w);

}

//////////////////////////////////////////////////////////////////////////
// open view from state
static QSet<Document*> selectDocuments(Project* p, const QList<GObjectReference>& refs) {
	QSet<Document*> res;
	foreach(const GObjectReference& r, refs) {
		Document* doc = p->findDocumentByURL(r.docUrl);
		if (doc!=NULL) {
			res.insert(doc);
		}
	}
	return res;
}


OpenSavedAnnotatedDNAViewTask::OpenSavedAnnotatedDNAViewTask(const QString& viewName, const QVariantMap& stateData) 
: ObjectViewTask(AnnotatedDNAViewFactory::ID, viewName, stateData)
{
	AnnotatedDNAViewState state(stateData);
    QList<GObjectReference> refs = state.getSequenceObjects();
    if (refs.isEmpty()) {
        stateIsIllegal = true;
        stateInfo.setError(ObjectViewTask::tr("No sequence info found!"));
        return;
    }
 	foreach(const GObjectReference& ref, refs) {
        Document* doc = AppContext::getProject()->findDocumentByURL(ref.docUrl);
	    if (doc == NULL) {
		    stateIsIllegal = true;
		    stateInfo.setError(ObjectViewTask::tr("document_not_found_%1").arg(ref.docUrl));
		    return;
	    }
	    if (!doc->isLoaded()) {
		    documentsToLoad.append(doc);
	    }
    }
	
	QSet<Document*> adocs = selectDocuments(AppContext::getProject(), state.getAnnotationObjects());
	foreach(Document* adoc, adocs) {
		if (!adoc->isLoaded()) {
			documentsToLoad.append(adoc);
		}
	}
}

void OpenSavedAnnotatedDNAViewTask::open() {
	if (stateInfo.hasErrors()) {
		return;
	}
	AnnotatedDNAViewState state(stateData);
    QList<DNASequenceObject*> sequenceObjects;
    foreach(const GObjectReference& ref, state.getSequenceObjects()) {
        Document* doc = AppContext::getProject()->findDocumentByURL(ref.docUrl);
    	if (doc == NULL) {
	    	stateIsIllegal = true;
		    stateInfo.setError(ObjectViewTask::tr("document_not_found_%1").arg(ref.docUrl));
		    return;
	    }
	    GObject* obj = doc->findGObjectByName(ref.objName);
	    if (obj == NULL || obj->getGObjectType() != GObjectTypes::DNA_SEQUENCE) {
		    stateIsIllegal = true;
		    stateInfo.setError(tr("dna_object_not_found_%1").arg(ref.objName));
		    return;
	    }
        DNASequenceObject* dnaObj= qobject_cast<DNASequenceObject*>(obj);
        sequenceObjects.append(dnaObj);
    }
    AnnotatedDNAView* v = new AnnotatedDNAView(viewName, sequenceObjects);
    GObjectViewWindow* w = new GObjectViewWindow(v, viewName, true);
    MWMDIManager* mdiManager = 	AppContext::getMainWindow()->getMDIManager();
    mdiManager->addMDIWindow(w);
    v->updateState(state);
}


//////////////////////////////////////////////////////////////////////////
// update
UpdateAnnotatedDNAViewTask::UpdateAnnotatedDNAViewTask(AnnotatedDNAView* v, const QString& stateName, const QVariantMap& stateData) 
: ObjectViewTask(v, stateName, stateData)
{
}

void UpdateAnnotatedDNAViewTask::update() {
	if (view.isNull()) {
		return; //view was closed;
	}

	AnnotatedDNAView* aview = qobject_cast<AnnotatedDNAView*>(view.data());
    assert(aview!=NULL);
	
    AnnotatedDNAViewState state(stateData);
    aview->updateState(state);
}



} // namespace
