/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "IntegralBusModel.h"
#include "IntegralBusType.h"

/* TRANSLATOR GB2::Workflow::BusPort */
namespace GB2 {
namespace Workflow {

const QString BusPort::BUS_MAP = "bus-map";

BusPort::BusPort(const PortDescriptor& d, Actor* p) : Port(d,p), recursing(false) {
    params[BUS_MAP] = new Attribute(Descriptor(BUS_MAP), DataTypePtr());
}

DataTypePtr BusPort::getType() const {
    return isInput() ? type : getBusType();
}

DataTypePtr BusPort::getBusType() const {
    if (recursing) {
        return DataTypePtr(new IntegralBusType(Descriptor(), QMap<Descriptor, DataTypePtr>()));
    }
    recursing = true;
    IntegralBusType* t = new IntegralBusType(Descriptor(*this), QMap<Descriptor, DataTypePtr>());
    foreach (Port* p, owner()->getInputPorts()) {
        if ((p->getFlags()&BLIND_INPUT) == 0){
            t->addInputs(p);
        }
    }
    t->addOutput(type, this);
    recursing = false;
    return DataTypePtr(t);
}

static Actor* getLinkedActor(ActorId id, Port* output) {
    if (output->owner()->getId() == id) {
        return output->owner();
    }
    foreach(Port* transit, output->owner()->getInputPorts()) {
        foreach(Port* p, transit->getLinks().uniqueKeys()) {
            Actor* a = getLinkedActor(id,p);
            if (a) return a;
        }
    }
    return NULL;
}

Actor* BusPort::getProducer(const QString& slot) {
    QList<Actor*> l = getProducers(slot);
    if (l.size() == 1) {
        return l.first();
    } else {
        return NULL;
    }
}

QList<Actor*> BusPort::getProducers(const QString& slot) {
    QList<Actor*> res;
    Attribute* at = getParameter(BUS_MAP);
    if (at) {
        QStrStrMap busMap = at->value.value<QStrStrMap>();
        QStringList vals = busMap.value(slot).split(";");
        foreach(QString val, vals) {
            ActorId id = IntegralBusType::parseSlotDesc(val);
            foreach(Port* peer, getLinks().uniqueKeys()) {
                Actor* ac = getLinkedActor(id,peer);
                if (ac) {
                    res << ac;
                }
            }
        }
    }
    return res;
}

void BusPort::remap(const QMap<ActorId, ActorId>& m) {
    Attribute* a = getParameter(BUS_MAP);
    if (a) {
        QStrStrMap busMap = a->value.value<QStrStrMap>();
        IntegralBusType::remap(busMap, m);
        setParameter(BUS_MAP, qVariantFromValue<QStrStrMap>(busMap));
    }
}

static QStringList findMatchingSlots(DataTypePtr set, DataTypePtr el) {
    QStringList result;
    foreach(const Descriptor& d, set->getElements()) {
        if (set->getElement(d) == el) {
            result.append(d.getId());
        }
    }
    return result;
}

static void filterAmbiguousSlots(QList<Descriptor>& keys, const QMap<Descriptor, DataTypePtr>& map, QStrStrMap& result) {
    foreach(DataTypePtr val, map) {
        const QList<Descriptor> lst = map.keys(val);
        if (lst.size() != 1) {
            foreach(Descriptor d, lst) {
                result.insert(d.getId(), "");
                keys.removeOne(d);
            }
        }
    }
}

void BusPort::setupBusMap() {
    if (isInput() && getWidth() == 1) {
        DataTypePtr to = getType();
        if (!to->isMap()) {
            QMap<Descriptor, DataTypePtr> map;
            map.insert(*this, to);
            to = new DataTypeSet(Descriptor(), map);
        }
        DataTypePtr from = bindings.uniqueKeys().first()->getType();
        QList<Descriptor> keys = to->getElements();
        QStrStrMap busMap = getParameter(BusPort::BUS_MAP)->value.value<QStrStrMap>();
        filterAmbiguousSlots(keys, to->getElementsMap(), busMap);
        foreach(Descriptor key, keys) {
            DataTypePtr eldt = to->getElement(key);
            QStringList cands = findMatchingSlots(from, eldt);
            if (eldt->isList()) {
                cands += findMatchingSlots(from, eldt->getElement());
                QString res = cands.join(";");
                busMap.insert(key.getId(), res);
            } else if (cands.size() != 1) {
                //no unambiguous match, reset
                busMap.insert(key.getId(), "");
            } else {
                busMap.insert(key.getId(), cands.first());
                /*QString old = busMap.value(key.getId());
                cands.append(Descriptor(""));
                int idx = busMap.contains(key.getId()) ? cands.indexOf(old) : 0;
                Descriptor actual = (idx >= 0) ? cands.at(idx) : cands.first();
                busMap.insert(key.getId(), actual.getId());*/
            }
        }

        setParameter(BUS_MAP, qVariantFromValue<QStrStrMap>(busMap));
    }
}

static QMap<QString, QStringList> getListMappings(const QStrStrMap& bm, const Port* p) {
    assert(p->isInput());    
    DataTypePtr dt = p->getType();
    QMap<QString, QStringList> res;
    if (dt->isList()) {
        QString val = bm.value(p->getId());
        if (!val.isEmpty()) {
            res.insert(p->getId(), val.split(";"));
        }
    } else if (dt->isMap()) {
        foreach(Descriptor d, dt->getElements()) {
            QString val = bm.value(d.getId());
            if (dt->getElement(d)->isList() && !val.isEmpty()) {
                res.insert(d.getId(), val.split(";"));
            }
        }
    }
    return res;
}

bool BusPort::validate(QStringList& l) const {
    bool good = Configuration::validate(l);
    if (isInput() && !validator) {
        good &= ScreenedSlotValidator::validate(QStringList(), this, l);
    }
    return good;
}

bool ScreenedSlotValidator::validate( const QStringList& screenedSlots, const BusPort* vport, QStringList& l) 
{
    bool good = true;
    {
        if (vport->getWidth() == 0) {
            l.append(BusPort::tr("No input data supplied"));
            return false;
        }
        QStrStrMap bm = vport->getParameter(BusPort::BUS_MAP)->value.value<QStrStrMap>();
        int busWidth = bm.size();
        QMap<QString, QStringList> listMap = getListMappings(bm, vport);
        // iterate over all producers and exclude valid mappings from bus bindings
        foreach(Port* p, vport->getLinks().uniqueKeys()) {
            assert(qobject_cast<BusPort*>(p));//TBD?
            DataTypePtr t = p->getType();
            assert(t->isMap());
            {
                foreach(Descriptor d, t->getElements()) {
                    foreach(QString key, bm.keys(d.getId())) {
                        //log.debug("reducing bus from key="+ikey+" to="+rkey);
                        assert(!key.isEmpty());
                        bm.remove(key);
                    }
                    foreach(QString key, listMap.uniqueKeys()) {
                        QStringList& l = listMap[key];
                        l.removeAll(d.getId());
                        if (l.isEmpty()) {
                            listMap.remove(key);
                            bm.remove(key);
                        }
                    }
                }
            }
        }
        if (busWidth == bm.size()) {
            l.append(BusPort::tr("No input data supplied"));
            good = false;
        }
        {
            QMapIterator<QString,QString> it(bm);
            while (it.hasNext())
            {
                it.next();
                const QString& slot = it.key();
                QString slotName = vport->getType()->getElementDescriptor(slot).getDisplayName();
                //assert(!slotName.isEmpty());
                if (it.value().isEmpty()) {
                    if (!screenedSlots.contains(slot)) {
                        l.append(BusPort::tr("Warning, empty input slot: %1").arg(slotName));
                    }
                } else {
                    l.append(BusPort::tr("Bad slot binding: %1 to %2").arg(slotName).arg(it.value()));
                    good = false;
                }
            }
        }
        {
            QMapIterator<QString,QStringList> it(listMap);
            while (it.hasNext())
            {
                it.next();
                const QString& slot = it.key();
                QString slotName = vport->getType()->getElementDescriptor(slot).getDisplayName();
                assert(!slotName.isEmpty());
                assert(!it.value().isEmpty());
                l.append(BusPort::tr("Bad slot binding: %1 to %2").arg(slotName).arg(it.value().join(",")));
                good = false;
            }
        }
    }
    return good;
}

bool ScreenedSlotValidator::validate( const Configuration* cfg, QStringList& output ) const {
    return validate(screenedSlots, static_cast<const BusPort*>(cfg), output);
}

}//Workflow namespace
}//GB2namespace
