/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_WORKFLOW_MODEL_H_
#define _GB2_WORKFLOW_MODEL_H_

#include <workflow/Datatype.h>
#include <workflow/Attribute.h>

#include <QtCore/QMimeData>
#include <QtGui/QIcon>
#include <QtGui/QTextDocument>

#include <assert.h>

namespace GB2 {

namespace Workflow {

class Actor;
class ActorDocument;
class CompositeActor;
class PortDescriptor;
class Port;
class Link;
class Schema;

class WCOREAPI_EXPORT VisualDescriptor : public Descriptor {
public:
    VisualDescriptor(const Descriptor& d, const QString & _iconPath = QString()) : Descriptor(d), iconPath(_iconPath) {}
    void setIconPath( const QString & iconPath_ ) { iconPath = iconPath_; }

    //getIcon will create icon from its path (if path exists) on the first call. 
    //This is compatibility issue for congene.
    QIcon getIcon();

    void setIcon(QIcon icon_);
private:
    QString iconPath;
    QIcon icon;
};

class WCOREAPI_EXPORT Prompter {
public:
    virtual ActorDocument* createDescription(Actor*) = 0;
    virtual ~Prompter() {};
};

class WCOREAPI_EXPORT ActorPrototype : public VisualDescriptor {
public:
    ActorPrototype(const Descriptor& desc, 
        const QList<PortDescriptor*>& ports = QList<PortDescriptor*>(), 
        const QList<Attribute*>& attrs = QList<Attribute*>());
    virtual ~ActorPrototype();
    QList<PortDescriptor*> getPortDesciptors() const {return ports;}
    QList<Attribute*> getAttributes() const {return attrs;}

    void setEditor(ConfigurationEditor* e) {ed = e;}
    void setValidator(ConfigurationValidator* v) {val = v;}
    void setPrompter(Prompter* p) {prompter = p;}
    void setPortValidator(const QString& id, ConfigurationValidator* v) {portValidators[id] = v;}

    virtual bool isAcceptableDrop(const QMimeData*, QVariantMap* = NULL) const {return false;}

    typedef QMap<QString,QVariant> ParamsMap;
    virtual Actor* createInstance(const QMap<QString, QVariant>& params = ParamsMap());

protected:
    virtual Port* createPort(const PortDescriptor& d, Actor* p);

protected:
    QList<Attribute*> attrs;
    QList<PortDescriptor*> ports; 
    ConfigurationEditor* ed;
    ConfigurationValidator* val;
    Prompter* prompter;
    QMap<QString, ConfigurationValidator*> portValidators;
};

class WCOREAPI_EXPORT Peer {
public:
    Peer() : peer(NULL) {}
    virtual ~Peer() {}
    template<typename T> inline T* castPeer() const {return (T*)(peer);}
    void* getPeer() const {return peer;}
    void setPeer(void* p) {peer = p;}
protected:
    void* peer;
};

class WCOREAPI_EXPORT Actor: public QObject, public Configuration, public Peer {
    Q_OBJECT
public:
    virtual ~Actor();
    ActorPrototype* getProto() const {return proto;}
    ActorId getId() const {return QString("%1").arg( (int) (size_t) this) ;}

    QString getLabel() const {return label.isEmpty() ? QString("%1 %2").arg(getProto()->getDisplayName()).arg(getId()) : label;}
    void setLabel(const QString& l) {label = l; emit si_labelChanged();}

    Port* getPort(const QString& id) const {return ports.value(id);}
    QList<Port*> getPorts() const { return ports.values();}
    QList<Port*> getInputPorts() const;
    QList<Port*> getOutputPorts() const;

    virtual void setParameter(const QString& name, const QVariant& val) {
        Configuration::setParameter(name, val);
        emit si_modified();
    }

    ActorDocument* getDescription() const {return doc;}
    void setDescription(ActorDocument* d) {doc = d;}

    //Schema* getContainer() const {return container;}
    //void setContainer(Schema* a);

    virtual void remap(const QMap<ActorId, ActorId>&);

signals:
    void si_labelChanged();
    void si_modified();
protected:
    friend class ActorPrototype;
    Actor(ActorPrototype* proto);
    Actor(const Actor&) : QObject(), Configuration(), Peer() {assert(0);}

protected:
    QString label;
    ActorPrototype* proto;
    //Schema* container;
    QMap<QString,Port*> ports; 
    ActorDocument* doc;
};

class WCOREAPI_EXPORT PortDescriptor : public Descriptor {
public:
    PortDescriptor(const Descriptor& desc, DataTypePtr type, bool input, bool multi = false, uint f = 0) 
        : Descriptor(desc), type(type), input(input), multi(multi), flags(f) {}
    virtual ~PortDescriptor() {}
    bool isInput() const {return input;}
    bool isOutput() const {return !input;}
    bool isMulti() const {return multi;}
    uint getFlags() const {return flags;}
    virtual DataTypePtr getType() const {return type;}
protected:
    DataTypePtr type;
    bool input, multi;
    uint flags;
};

class WCOREAPI_EXPORT Port : public QObject, public PortDescriptor, public Configuration,  public Peer {
    Q_OBJECT
public:
    Port(const PortDescriptor& d, Actor* p): PortDescriptor(d), proc(p) {}
    virtual ~Port() {}
    Actor* owner() const {return proc;}
    QMap<Port*,Link*> getLinks() const {return bindings;}
    int getWidth() const {return bindings.size();}
    virtual void addLink(Link*);
    virtual void removeLink(Link*);

    virtual void setParameter(const QString& name, const QVariant& val) {
        Configuration::setParameter(name, val);
        emit bindingChanged();
    }

    virtual bool canBind(const Port* other) const;
signals:
    void bindingChanged();
protected:
    Actor* proc;
    QMap<Port*,Link*> bindings;
};

class WCOREAPI_EXPORT Link : public QObject, public Peer {
    Q_OBJECT
public:
    Port* source() const {return src;}
    Port* destination() const {return dest;}

//    DataTypeConvertion* getDataMapper() const {return mapper;}
//    void setDataMapper(DataTypeConvertion* mapper);

    Link() : src(NULL), dest(NULL) {}
    Link(Port* p1, Port* p2) {connect(p1, p2);}
    virtual ~Link() {}
    void connect(Port* p1, Port* p2);

private:
    Port *src, *dest;
//    DataTypeConvertion* mapper;
};

class WCOREAPI_EXPORT ActorDocument : public QTextDocument {
public:
    ActorDocument(Actor* a) : QTextDocument(a), target(a) {}
    virtual ~ActorDocument() {}
    virtual void update(const QVariantMap& ) {}
protected:
    Actor* target;
};

}//Workflow namespace
}//GB2 namespace


#endif
