/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SWAlgorithmTests.h"
#include "SWAlgorithmTask.h"

#include <datatype/DNASequence.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/GObjectTypes.h>

#include <core_api/DocumentModel.h>
#include <core_api/DocumentFormats.h>
#include <core_api/GObject.h>

#include <util_smith_waterman/SubstMatrixFactory.h>
#include <util_smith_waterman/SmithWatermanSettings.h>
#include <util_smith_waterman/SmithWatermanTests.h>

#include <qbytearray.h>

#define FILE_SUBSTITUTION_MATRIX_ATTR "subst_f"
#define FILE_FASTA_CONTAIN_SEQUENCE_ATTR "seq_f"
#define FILE_FASTA_CONTAIN_PATTERN_ATTR "pattern_f"
#define GAP_OPEN_ATTR "g_o"
#define GAP_EXT_ATTR "g_e"
#define PERCENT_OF_SCORE_ATTR "percent_of_score"
//#define MINIMUM_SCORE_ATTR "min_scor"
#define COUNT_OF_RESULTS_ATTR "count_res"
#define EXPECTED_RESULT_ATTR "expected_res"



#include "iostream"

namespace GB2 {

	void GTest_SWAlgorithm::init(XMLTestFormat *tf, const QDomElement& el) {		
		Q_UNUSED(tf);

		searchSeqDocName = el.attribute(FILE_FASTA_CONTAIN_SEQUENCE_ATTR);
		if (searchSeqDocName.isEmpty()) {
			failMissingValue(FILE_FASTA_CONTAIN_SEQUENCE_ATTR);
			return;
		} 

		patternSeqDocName = el.attribute(FILE_FASTA_CONTAIN_PATTERN_ATTR);
		if (patternSeqDocName.isEmpty()) {
			failMissingValue(FILE_FASTA_CONTAIN_PATTERN_ATTR);
			return;
		} 

		pathToSubst = el.attribute(FILE_SUBSTITUTION_MATRIX_ATTR);
		if (pathToSubst.isEmpty()) {
			failMissingValue(FILE_SUBSTITUTION_MATRIX_ATTR);
			return;
		} 		

		QString buffer = el.attribute(GAP_OPEN_ATTR);
		bool ok = false;

		if (!buffer.isEmpty()) {
			ok=false;
			gapOpen = buffer.toInt(&ok);
			if(!ok) {
				failMissingValue(GAP_OPEN_ATTR);
				return;
			}
		}

		buffer = el.attribute(GAP_EXT_ATTR);
		if (!buffer.isEmpty()) {
			ok=false;
			gapExtension = buffer.toInt(&ok);
			if(!ok) {
				failMissingValue(GAP_EXT_ATTR);
				return;
			}
		}

		buffer = el.attribute(PERCENT_OF_SCORE_ATTR);
		if (!buffer.isEmpty()) {
			ok=false;
			percentOfScore = buffer.toFloat(&ok);
			if(!ok) {
				failMissingValue(PERCENT_OF_SCORE_ATTR);
				return;
			}
		}

		buffer = el.attribute(COUNT_OF_RESULTS_ATTR);
		if (!buffer.isEmpty()) {
			ok=false;
			countOfResults = buffer.toInt(&ok);
			if(!ok) {
				failMissingValue(COUNT_OF_RESULTS_ATTR);
				return;
			}
		}

		expected_res = el.attribute(EXPECTED_RESULT_ATTR);
		if (expected_res.isEmpty()) {
			failMissingValue(EXPECTED_RESULT_ATTR);
			return;
		}

		if (!parseExpected_res()) {
			stateInfo.setError(QString("value not correct %1").arg(EXPECTED_RESULT_ATTR));
		}					
	}


	void GTest_SWAlgorithm::prepare() {
//get search sequence		
		DNASequenceObject * searchSeqObj = getContext<DNASequenceObject>(this, searchSeqDocName);
		if(searchSeqObj==NULL){
			stateInfo.setError(QString("error can't cast to sequence from GObject"));
			return;
		}
		searchSeq = searchSeqObj->getDNASequence().seq;

//get pattern sequence
		DNASequenceObject * patternSeqObj = getContext<DNASequenceObject>(this, patternSeqDocName);
		if(patternSeqObj==NULL){
			stateInfo.setError(QString("error can't cast to sequence from GObject"));
			return;
		}
		patternSeq = patternSeqObj->getDNASequence().seq;

//set subst matrix

		QString pathToCommonData = getEnv()->getVar("COMMON_DATA_DIR");		
		if(patternSeqObj==NULL){
			stateInfo.setError(QString("error can't get path to common_data dir"));
			return;
		}
		QString fullPathToSubst = pathToCommonData + "/" + pathToSubst;
//TODO: matrix reader
		SubstitutionMatrix substMatr;
		substMatr.readMatrixFromFile(fullPathToSubst);

		int matLen = substMatr.alphabet.size();

		float * substMatrAr = new float [matLen * matLen];
		for (int i = 0; i < matLen; i++) 
			for (int j = 0; j < matLen; j++) 
				substMatrAr[matLen * i + j] = substMatr.getMatrix()[i][j];

		QByteArray alph;
		alph.append(substMatr.alphabet);

		SubstMatrix * mtx = SubstMatrixFactory::createSubstMatrix(alph, substMatrAr);		
		delete substMatrAr;
		if (0 == mtx) {
			stateInfo.setError(QString("value not set %1").arg(FILE_SUBSTITUTION_MATRIX_ATTR));
			return;
		}		

// 		SWAlgorithmTask(const SmithWatermanSettings& s,
// 			const QString& taskName);
		
		SmithWatermanSettings s;
		s.pSm = mtx;
		s.sqnc = searchSeq;
		s.ptrn = patternSeq;
		s.globalRegion.startPos = 0;
		s.globalRegion.len = searchSeq.length();
		s.gapModel.scoreGapOpen = gapOpen;
		s.gapModel.scoreGapExtd = gapExtension;
		s.percentOfScore = percentOfScore;
		s.resultFilter = NULL;
		s.resultCallback = NULL;
		s.complTT = NULL;
		s.aminoTT = NULL;

//		swAlgorithmTask = new SWAlgorithmTask(s, tr("Smith Waterman 2 test"), false);
// 		swAlgorithmTask = new SWAlgorithmTask(mtx, searchSeq, patternSeq, gapOpen, gapExtension, minScore, countOfResults);

		addSubTask(swAlgorithmTask);
		
	}

	bool GTest_SWAlgorithm::parseExpected_res() {
		PairSequences pairSequences;
		int score = 0;
		int sBegin = 0;
		int sEnd = 0;
		int pBegin = 0;
		int pEnd = 0;
		LRegion sInterval(0, 0) ;
		LRegion pInterval(0, 0);
		QByteArray s;		
		QByteArray p;		

		char separator = '|';		
		
		QString str1;
		QString str2;		
		for (int i = 0; i < expected_res.split("**").size() - 1; i++) {
			score = 0;
			sBegin = 0;
			sEnd = 0;
			s.clear();
			pBegin = 0;
			pEnd = 0;
			p.clear();

			str1 = expected_res.section("**", i, i);			
//score
			str2 = str1.section(separator, 0, 0);
			if (!toInt(str2, score)) return false;
			str2.clear();
//sBegin
			str2 = str1.section(separator, 1, 1);
			if (!toInt(str2, sBegin)) return false;
			str2.clear();
//sEnd
			str2 = str1.section(separator, 2, 2);
			if (!toInt(str2, sEnd)) return false;
			str2.clear();
//s
			str2 = str1.section(separator, 3, 3);
			if (str2.isEmpty()) return false;
			s.append(str2);
			str2.clear();

//////////////////////////////////////////////////////////////////////////

//pBegin
			str2 = str1.section(separator, 4, 4);
			if (!toInt(str2, pBegin)) return false;
			str2.clear();
//pEnd
			str2 = str1.section(separator, 5, 5);
			if (!toInt(str2, pEnd)) return false;
			str2.clear();
//p
			str2 = str1.section(separator, 6, 6);
			if (str2.isEmpty()) return false;
			p.append(str2);
			str2.clear();

//append to expected results			
			
			pairSequences.searchSeq = s;
			pairSequences.patterSeq = p;

			pairSequences.sInterval.startPos = sBegin;
			pairSequences.sInterval.len = sEnd - sBegin;

			pairSequences.pInterval.startPos = pBegin;
			pairSequences.pInterval.len = pEnd - pBegin;		

			pairSequences.score = score;

			expectedRes.append(pairSequences);

		}
		return true;

	}

bool GTest_SWAlgorithm::toInt(QString & str, int & num) {
		bool ok = false;		
		if (!str.isEmpty()) {			
			num = str.toInt(&ok);
		} 
		return ok;		
	}	


	Task::ReportResult GTest_SWAlgorithm::report() {		
		QList<PairAlignSequences> res = swAlgorithmTask->getResult();	

		//Print result
// 		for (int qq = 0; qq < res.size(); qq++) {
// 			cout <<"res " <<qq <<": " <<res.at(qq).score <<"|" <<res.at(qq).intervalSeq1.startPos 
// 				<<"|" <<res.at(qq).intervalSeq1.endPos() <<"|" <<res.at(qq).intervalSeq2.startPos
// 				<<"|" <<res.at(qq).intervalSeq2.endPos() <<endl;
// 		}		
		
		if (res.size() != expectedRes.size()) {
			stateInfo.setError(QString("Not expected result: count result not coincide"));
			return ReportResult_Finished;
		}

		QByteArray searchAlignSeq;
		QByteArray patternhAlignSeq;
		
		for (int i = 0; i < res.size(); i++) {

// /			res[i].getAlignSequences(searchSeq, patternSeq, searchAlignSeq, patternhAlignSeq);
// 
// 			if (res.at(i).intervalSeq1.startPos != expectedRes.at(i).sInterval.startPos
// 				||
// 				res.at(i).intervalSeq1.len != expectedRes.at(i).sInterval.len) {
// 				stateInfo.setError(QString("Not expected result: intervalSeq1"));
// 				return ReportResult_Finished;
// 			}
// 			if (res.at(i).intervalSeq2.startPos != expectedRes.at(i).pInterval.startPos
// 				||
// 				res.at(i).intervalSeq2.len != expectedRes.at(i).pInterval.len) {
// 					stateInfo.setError(QString("Not expected result: intervalSeq2"));
// 					return ReportResult_Finished;
// 			}

			if (res.at(i).score != expectedRes.at(i).score) {
				stateInfo.setError(QString("Not expected result: score"));
				return ReportResult_Finished;
			}
			if (searchAlignSeq != expectedRes.at(i).searchSeq) {
 				stateInfo.setError(QString("Not expected result: search seq"));
				return ReportResult_Finished;
			}
			if (patternhAlignSeq != expectedRes.at(i).patterSeq) {
				stateInfo.setError(QString("Not expected result: pattern seq"));
				return ReportResult_Finished;
			}
		}

		return ReportResult_Finished;
	}

}
