/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "PlainTextFormat.h"


#include <core_api/Task.h>
#include <gobjects/TextObject.h>
#include <util_text/TextUtils.h>

#include <core_api/IOAdapter.h>

#include <memory>

namespace GB2 {

/* TRANSLATOR GB2::IOAdapter */    

PlainTextFormat::PlainTextFormat(QObject* p) : DocumentFormat(p) 
{
	formatName = tr("plain_text_format");
}

QStringList PlainTextFormat::getSupportedDocumentFileExtensions() {
	QStringList l;
	l<<"txt";
	return l;
}

Document* PlainTextFormat::createNewDocument(IOAdapterFactory* io, const QString& url, const QVariantMap& fs) {
	Document* d = DocumentFormat::createNewDocument(io, url, fs);
	GObject* o = new TextObject("", "Text");
	d->addObject(o);
	return d;
}

#define BUFF_SIZE 1024

Document* PlainTextFormat::loadExistingDocument(IOAdapterFactory* iof, const QString& url, TaskStateInfo& ti, const QVariantMap& fs) {
	std::auto_ptr<IOAdapter> io(iof->createIOAdapter());
	
	if (!io->open(url, IOAdapterMode_Read)) {
		ti.setError(IOAdapter::tr("error_opening_url_for_read '%1'").arg(url));
		return NULL;
	}
	QString text;
    int size = io->left();
    if (size > 0) {
        text.reserve(size);
    }
	QByteArray block(BUFF_SIZE, '\0');
	int blockLen = 0;
	while ((blockLen = io->readBlock(block.data(), BUFF_SIZE)) > 0) {
        //todo: text.append can fail on realloc for large sizes
        text.append(QString::fromLocal8Bit(block.data(), blockLen));
        ti.progress = io->getProgress();
	}
	io->close();

    //todo: check file-readonly status?

    TextObject* to = new TextObject(text, "Text");
    QList<GObject*> objects;
    objects.append(to);
	Document* d = new Document(this, iof, url, objects, fs);
	return d;
}

void PlainTextFormat::storeDocument(Document* d, TaskStateInfo& ti, IOAdapterFactory* iof, const QString& newDocURL) {
	assert(d->getDocumentFormat() == this);
	assert(d->getObjects().size() ==1);
	
    if (iof == NULL) {
        iof = d->getIOAdapterFactory();
    }
	assert(iof);
	
	std::auto_ptr<IOAdapter> io(iof->createIOAdapter());
	
	GObject* obj = d->getObjects().first();
	assert(obj!=NULL);
	TextObject* to = (TextObject*)obj;
	const QString& text = to->getText();

	QString url = newDocURL.isEmpty() ? d->getURL() : newDocURL;
	if (!io->open(url, IOAdapterMode_Write)) {
		ti.setError(IOAdapter::tr("error_opening_url_for_write '%1'").arg(url));
		return;
	}
	QByteArray local8bit = text.toLocal8Bit();
	int nWritten = 0;
	int nTotal = text.size();
	while(nWritten < nTotal) {
		int l = io->writeBlock(local8bit.data() + nWritten, nTotal - nWritten);
		assert(l > 0);
		nWritten+= l;
	}
}
	

bool PlainTextFormat::isDataFormatSupported(const char* data, int size) const {
	return !TextUtils::contains(TextUtils::BINARY, data, size);
}

bool PlainTextFormat::isObjectOpSupported(const Document* d , DocumentFormat::DocObjectOp op, GObjectType t) const{
	return (t == GObjectTypes::TEXT && (op == DocObjectOp_Remove || d->getObjects().isEmpty()));
}

bool PlainTextFormat::checkConstraints(const DocumentFormatConstraints& c) const {
	foreach (GObjectType t, c.supportedObjectTypes) {
		if (t!=GObjectTypes::TEXT) {
			return false;
		}
	}
	if (c.checkRawData) {
		return isDataFormatSupported(c.rawData.constData(), c.rawData.size());
	}
    if( c.supportsStreamingRead ) {
        return false;
    }
    return true;
}


}//namespace
