/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorConsensusArea.h"
#include "MSAEditor.h"
#include "MSAEditorSequenceArea.h"
#include "MSAEditorConsensusCache.h"

#include <gobjects/MAlignmentObject.h>
#include <util_gui/GraphUtils.h>
#include <util_gui/GUIUtils.h>
#include <util_algorithm/MSAUtils.h>
#include <core_api/DNAAlphabet.h>

#include <QtGui/QPainter>
#include <QtGui/QApplication>
#include <QtGui/QClipboard>
#include <QtGui/QToolTip>

namespace GB2 {


MSAEditorConsensusArea::MSAEditorConsensusArea(MSAEditorUI* _ui) : editor(_ui->editor), ui(_ui) {
    completeRedraw = true;
    cachedView = new QPixmap();

    connect(ui->seqArea, SIGNAL(si_startChanged(const QPoint&, const QPoint&)), SLOT(sl_startChanged(const QPoint&, const QPoint&)));
    connect(ui->seqArea, SIGNAL(si_cursorMoved(const QPoint&, const QPoint&)), SLOT(sl_cursorMoved(const QPoint&, const QPoint&)));
    connect(ui->seqArea, SIGNAL(si_scaleChanged()), SLOT(sl_scaleChanged()));

    connect(editor->getMSAObject(), SIGNAL(si_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)), 
                                    SLOT(sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)));


    connect(editor, SIGNAL(si_buildStaticMenu(GObjectView*, QMenu*)), SLOT(sl_buildStaticMenu(GObjectView*, QMenu*)));
    copyConsensusAction = new QAction(tr("copy_consensus"), this);
    connect(copyConsensusAction, SIGNAL(triggered()), SLOT(sl_copyConsensusSequence()));
    connect(editor, SIGNAL(si_buildPopupMenu(GObjectView* , QMenu*)), SLOT(sl_buildContextMenu(GObjectView*, QMenu*)));

    setupFontAndHeight();
    
    setMouseTracking(true);

    consensusCache = new MSAEditorConsensusCache(this, editor->getMSAObject());
}

MSAEditorConsensusArea::~MSAEditorConsensusArea() {
    delete cachedView;
}

bool MSAEditorConsensusArea::event(QEvent* e) {
    if (e->type() == QEvent::ToolTip) {
        QHelpEvent* he = static_cast<QHelpEvent*>(e);
        QString tip = createToolTip(he);
        if (!tip.isEmpty()) {
            QToolTip::showText(he->globalPos(), tip);
        }
        return true;
    }
    return QWidget::event(e);
}

QString MSAEditorConsensusArea::createToolTip(QHelpEvent* he) const {
    int  x = he->pos().x();
    int pos = ui->seqArea->coordToPos(x);
    if (pos >= 0) {
        return MSAUtils::getConsensusPercentTip(editor->getMSAObject()->getMAlignment(), pos, 0, 5);
    }
    return QString();
}

void MSAEditorConsensusArea::resizeEvent(QResizeEvent *e) {
    completeRedraw = true;
    QWidget::resizeEvent(e);
}

void MSAEditorConsensusArea::paintEvent(QPaintEvent *e) {
    QSize s = size();
    QSize sas = ui->seqArea->size(); Q_UNUSED(sas);
    if (sas.width() != s.width()) { //this can happen due to the manual layouting performed by MSAEditor -> just wait for the next resize+paint
        return;
    }
    assert(s.width() == sas.width());
    if (cachedView->size() != s) {
        assert(completeRedraw);
        delete cachedView;
        cachedView = new QPixmap(s);
    }
    if (completeRedraw) {
        QPainter pCached(cachedView);
        pCached.fillRect(cachedView->rect(), Qt::white);
        drawConsensus(pCached);
        drawRuler(pCached);
        drawHistogram(pCached);
        completeRedraw = false;
    }
    QPainter p(this);
    p.drawPixmap(0, 0, *cachedView);
    drawCursor(p);

    QWidget::paintEvent(e);
}


void MSAEditorConsensusArea::drawCursor(QPainter& p) {
    int pos = ui->seqArea->getCursorPos().x();
    if (pos < ui->seqArea->getFirstVisibleBase() || pos > ui->seqArea->getLastVisibleBase(true)) {
        return;
    }
    QFont f = ui->seqArea->getSeqFont();
    f.setWeight(QFont::DemiBold);
    p.setFont(f);
    drawConsensusChar(p, pos, true);
}

void MSAEditorConsensusArea::drawConsensus(QPainter& p) {
    //draw consensus
    p.setPen(Qt::black);
    
    QFont f = ui->seqArea->getSeqFont();
    f.setWeight(QFont::DemiBold);
    p.setFont(f);

    int startPos = ui->seqArea->getFirstVisibleBase();
    int lastPos = ui->seqArea->getLastVisibleBase(true);
    for (int pos = startPos; pos <= lastPos; pos++) {
        drawConsensusChar(p, pos, false);
    }
}

void MSAEditorConsensusArea::drawConsensusChar(QPainter& p, int pos, bool selected) {
    LRegion yRange = getYRange(MSAEditorConsElement_CONSENSUS_TEXT);
    LRegion xRange= ui->seqArea->getBaseXRange(pos, false);
    QRect cr(xRange.startPos, yRange.startPos, xRange.len + 1, yRange.len);
    int w = width(), h = height();
    assert(xRange.endPos() <= w && yRange.endPos() <= h); Q_UNUSED(w); Q_UNUSED(h);
    if (selected) {
        QColor color(Qt::lightGray);
        color = color.lighter(115);
        p.fillRect(cr, color);
    }
    char c = consensusCache->getConsensusChar(pos);
    p.drawText(cr, Qt::AlignVCenter | Qt::AlignHCenter, QString(c));
}

#define RULER_NOTCH_SIZE 3

void MSAEditorConsensusArea::drawRuler(QPainter& p) {
    //draw ruler
    p.setPen(Qt::darkGray);

    int w = width();
    int startPos = ui->seqArea->getFirstVisibleBase();
    int lastPos = ui->seqArea->getLastVisibleBase(true);

    QFontMetrics rfm(rulerFont);
    LRegion rr = getYRange(MSAEditorConsElement_RULER);
    LRegion rrP = getYRange(MSAEditorConsElement_CONSENSUS_TEXT);
    int dy = rr.startPos - rrP.endPos();
    rr.len+=dy;
    rr.startPos-=dy;
    LRegion firstBaseXReg = ui->seqArea->getBaseXRange(startPos, false);
    LRegion lastBaseXReg = ui->seqArea->getBaseXRange(lastPos, false);
    int firstLastLen = lastBaseXReg.startPos - firstBaseXReg.startPos;
    int firstXCenter = firstBaseXReg.startPos + firstBaseXReg.len / 2;
    QPoint startPoint(firstXCenter, rr.startPos);
    
    GraphUtils::RulerConfig c;
    c.singleSideNotches = true;
    c.notchSize = RULER_NOTCH_SIZE;
    c.textOffset = (rr.len - rfm.ascent()) /2;
    c.extraAxisLenBefore = startPoint.x();
    c.extraAxisLenAfter = w - (startPoint.x() + firstLastLen);
    c.textBorderStart = -firstBaseXReg.len / 2;
    c.textBorderEnd = -firstBaseXReg.len / 2;
    GraphUtils::drawRuler(p, startPoint, firstLastLen, startPos + 1, lastPos + 1, rulerFont, c);

    startPoint.setY(rr.endPos());
    c.drawNumbers = false;
    c.textPosition = GraphUtils::LEFT;
    GraphUtils::drawRuler(p, startPoint, firstLastLen, startPos + 1, lastPos + 1, rulerFont, c);
}

void MSAEditorConsensusArea::drawHistogram(QPainter& p) {
    QColor c("#255060");
    p.setPen(c);
    LRegion yr = getYRange(MSAEditorConsElement_HISTOGRAM);
    yr.startPos++; yr.len-=2; //keep borders
    int nSeq = editor->getNumSequences();
    QBrush brush(c, Qt::Dense4Pattern);
    QVector<int> counts(256, 0);
    for (int pos = ui->seqArea->getFirstVisibleBase(), lastPos = ui->seqArea->getLastVisibleBase(true); pos <= lastPos; pos++) {
        LRegion xr = ui->seqArea->getBaseXRange(pos, true);
        int percent = consensusCache->getConsensusCharPercent(pos);
        assert(percent >= 0 && percent <= 100);
        int h = qRound(percent * yr.len / 100.0);
        QRect hr(xr.startPos + 1, yr.endPos() - h, xr.len - 2, h);
        p.drawRect(hr);
        p.fillRect(hr, brush);
    }
}

LRegion MSAEditorConsensusArea::getYRange(MSAEditorConsElement e) const {
    LRegion res;
    switch(e) {
        case MSAEditorConsElement_HISTOGRAM: 
                                res = LRegion(0, 50);
                                break;
        case MSAEditorConsElement_CONSENSUS_TEXT: 
                                res = LRegion(0, ui->seqArea->getRowHeight());
                                res.startPos += getYRange(MSAEditorConsElement(e-1)).endPos();
                                break;
        case MSAEditorConsElement_RULER: 
                                res = LRegion(0, rulerFontHeight + 2 * RULER_NOTCH_SIZE + 4);
                                res.startPos += getYRange(MSAEditorConsElement(e-1)).endPos();
                                break;
    }
    return res;
}

void MSAEditorConsensusArea::sl_startChanged(const QPoint& p, const QPoint& prev) {
    if (p.x() == prev.x()) {
        return;
    }
    completeRedraw = true;
    update();
}

void MSAEditorConsensusArea::sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&) {
    completeRedraw = true;
    update();
}

void MSAEditorConsensusArea::setupFontAndHeight() {
    rulerFont.setFamily("Arial");
    rulerFont.setPointSize(qMax(8, int(ui->seqArea->getSeqFont().pointSize() * 0.7)));
    rulerFontHeight = QFontMetrics(rulerFont).height();

    setFixedHeight(getYRange(MSAEditorConsElement_RULER).endPos() + 1);
}

void MSAEditorConsensusArea::sl_scaleChanged() {
    setupFontAndHeight();
}

void MSAEditorConsensusArea::sl_cursorMoved(const QPoint& pos, const QPoint& prev) {
    if (prev.x() == pos.x()) {
        return;
    }
    update();
}

void MSAEditorConsensusArea::sl_buildStaticMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorConsensusArea::sl_buildContextMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorConsensusArea::buildMenu(QMenu* m) {
    QMenu* copyMenu = GUIUtils::findSubMenu(m, MSAE_MENU_COPY);
    assert(copyMenu!=NULL);
    copyMenu->addAction(copyConsensusAction);
}

void MSAEditorConsensusArea::sl_copyConsensusSequence() {
    QApplication::clipboard()->setText(consensusCache->getConsensusLine());
}


}//namespace
