/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QMutableListIterator>
#include <QtGui/QHeaderView>

#include <core_api/LRegion.h>
#include <core_api/AppContext.h>
#include <core_api/SecStructPredictAlgRegistry.h>
#include <gobjects/DNASequenceObject.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_tasks/CreateAnnotationTask.h>
#include <util_gui/CreateAnnotationDialog.h>
#include <util_gui/CreateAnnotationWidgetController.h>
#include <selection/DNASequenceSelection.h>

#include "SecStructDialog.h"
#include "SecStructPredictTask.h"

namespace GB2 {

SecStructDialog::SecStructDialog( ADVSequenceObjectContext* _ctx, QWidget *p ) : QDialog(p), rangeStart(0), rangeEnd(0), ctx(_ctx), task(NULL)
{
    setupUi(this);
    sspr = AppContext::getSecStructPredictAlgRegistry();
    algorithmComboBox->addItems(sspr->getAlgNameList());
    saveAnnotationButton->setDisabled(true);
    
    LRegion initialSelection = ctx->getSequenceSelection()->isEmpty() ? LRegion() : ctx->getSequenceSelection()->getSelectedRegions().first();

    int seqLen = ctx->getSequenceLen();

    rangeStartSpinBox->setMinimum(1);
    rangeStartSpinBox->setMaximum(seqLen);

    rangeEndSpinBox->setMinimum(1);
    rangeEndSpinBox->setMaximum(seqLen);

    rangeStartSpinBox->setValue(initialSelection.isEmpty() ? 1 : initialSelection.startPos + 1);
    rangeEndSpinBox->setValue(initialSelection.isEmpty() ? seqLen : initialSelection.endPos());

    resultsTable->setColumnCount(2);
    QStringList headerNames;
    headerNames.append(tr("Region"));
    headerNames.append(tr("Structure Type"));
    resultsTable->setHorizontalHeaderLabels(headerNames);
    resultsTable->horizontalHeader()->setStretchLastSection(true);
    
    connect(AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_onTaskFinished(Task*)));
    connectGUI();

}

void SecStructDialog::connectGUI()
{
    connect(rangeStartSpinBox, SIGNAL(valueChanged(int)), this, SLOT(sl_spinRangeStartChanged(int)));
    connect(rangeEndSpinBox, SIGNAL(valueChanged(int)), this, SLOT(sl_spinRangeEndChanged(int)));
    connect(startButton, SIGNAL(clicked()), this, SLOT(sl_onStartPredictionClicked()));
    connect(saveAnnotationButton, SIGNAL(clicked()), this, SLOT(sl_onSaveAnnotations()));

    
}

void SecStructDialog::sl_spinRangeStartChanged( int val )
{
    if (val > rangeEndSpinBox->value()) {
        rangeEndSpinBox->setValue(val);
    }
    
}

void SecStructDialog::sl_spinRangeEndChanged( int val )
{
    if (val < rangeStartSpinBox->value()) {
        rangeStartSpinBox->setValue(val);
    }
}

void SecStructDialog::updateState()
{
    bool haveActiveTask = task!=NULL;
    bool haveResults = !results.isEmpty();

    algorithmComboBox->setEnabled(!haveActiveTask);
    startButton->setEnabled(!haveActiveTask);
    cancelButton->setEnabled(!haveActiveTask);
    saveAnnotationButton->setEnabled(haveResults);
    totalPredictedStatus->setText( QString("%1").arg(results.size()));
    showResults();

}

void SecStructDialog::sl_onStartPredictionClicked()
{
    assert(task == NULL);
    SecStructPredictTaskFactory* factory = sspr->getAlgorithm(algorithmComboBox->currentText());

    //prepare target sequence
    const QByteArray & seq = ctx->getSequenceData();
    rangeStart = rangeStartSpinBox->value();
    rangeEnd = rangeEndSpinBox->value();
    assert(rangeStart <= rangeEnd);
    assert(rangeStart >= 0 && rangeEnd <= seq.size() );

    task = factory->createTaskInstance(seq.mid(rangeStart, rangeEnd - rangeStart));
    AppContext::getTaskScheduler()->registerTopLevelTask(task);
    results.clear();
    
    updateState();
}

void SecStructDialog::sl_onTaskFinished( Task* t )
{
    if (t != task || t->getState()!= Task::State_Finished) {
        return;
    }
    results = task->getResults();

    //shifting results according to startPos
    for(QMutableListIterator<SharedAnnotationData> it_ad(results); it_ad.hasNext(); ) {
        AnnotationData * ad = it_ad.next().data();
        assert( !ad->complement );
        QList<LRegion> & regions = ad->location;
        for( QMutableListIterator<LRegion> it(regions); it.hasNext(); ) {
            it.next().startPos += rangeStart;
        }
    }
    task = NULL;
    rangeStart = 0;
    rangeEnd = 0;
    updateState();

}

void SecStructDialog::showResults()
{
    int rowIndex = 0;
    resultsTable->setRowCount(results.size());
    foreach(SharedAnnotationData data, results) {
        LRegion annRegion = data->location.first();
        QTableWidgetItem *locItem = new QTableWidgetItem( QString("[%1..%2]").arg(annRegion.startPos).
            arg(annRegion.endPos()) );
        resultsTable->setItem(rowIndex, 0, locItem);
        QTableWidgetItem* nameItem = new QTableWidgetItem( QString(data->name));
        resultsTable->setItem(rowIndex, 1, nameItem);
        ++rowIndex;
    }
    

}

#define SEC_STRUCT_ANNOTATION_GROUP_NAME "predicted"

void SecStructDialog::sl_onSaveAnnotations()
{
    CreateAnnotationModel m;
    m.sequenceObjectRef = ctx->getSequenceObject();
    m.hideLocation = true;
    m.hideAnnotationName = true;
    m.data->name = SEC_STRUCT_ANNOTATION_GROUP_NAME;
    m.sequenceLen = ctx->getSequenceObject()->getSequenceLen();
    CreateAnnotationDialog d(this, m);
    int rc = d.exec();
    if (rc != QDialog::Accepted) {
        return;
    }
    CreateAnnotationsTask* t = new CreateAnnotationsTask(m.getAnnotationObject(), m.groupName, results);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);

    QDialog::accept();

    
}
} // namespace


