/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SubstMatrix.h"

#include <core_api/Log.h>
#include <QObject>

namespace GB2 {

static LogCategory log(ULOG_CAT_SW);

SubstMatrix::SubstMatrix():
        dim(0), data(0), dnaAlphabet(0)
{
	for (int i = 0; i < 256; i++) {
		tr_map[i] = -1;
	}
}

SubstMatrix::~SubstMatrix() {
	if (0 != data) {
		delete[] data;
	}
}

bool SubstMatrix::create(
					DNAAlphabet* _dnaAlphabet, 
					const QByteArray& mtxAlph, 
					const float* rawMatrix,
					float defScore) {
	dnaAlphabet = _dnaAlphabet;
	if (0 == dnaAlphabet) {
		return false;
	}
	dim = dnaAlphabet->getNumAlphabetChars();
	data = new float[dim * dim];
	if (0 == data) {
		return false;
	}
	setMatrix(defScore);
	initTranslationMap(mtxAlph);
	initRawMatrix(mtxAlph, rawMatrix);
	return true;
}

void SubstMatrix::setMatrix(float defScore) {
	for (int i = 0; i < dim; i++) {
		for (int j = 0; j < dim; j++) {
			data[i * dim + j] = defScore;
		}
	}
}

void SubstMatrix::initTranslationMap(const QByteArray& mtxAlph) {

	for (int i = 0; i < 256; i++) {
		tr_map[i] = dim - 1;
	}

	int charIndex = 0;
	char defSymbol = dnaAlphabet->getDefaultSymbol();
	for (int i = 0, rawDim = mtxAlph.length(); i < rawDim; i++) {
                uchar ch = mtxAlph[i];
		if (defSymbol != ch) {
			tr_map[ch] = charIndex++;
		}
	}

	QByteArray dnaAlph = dnaAlphabet->getAlphabetChars();
	for (int i = 0; i < dim; i++) {
                uchar ch = dnaAlph[i];
		if (tr_map[ch] == dim-1 && defSymbol != ch) {
			tr_map[ch] = charIndex++;
		}
	}
	return;
}

void SubstMatrix::initRawMatrix(const QByteArray& mtxAlph, const float* rawMatrix) {
	int rawDim = mtxAlph.length();
	for (int row = 0; row < rawDim; row++) {
		for (int column = 0; column < rawDim; column++) {
                        uchar ch_row = mtxAlph[row];
                        uchar ch_column = mtxAlph[column];
			float score = rawMatrix[row * rawDim + column];
			setScore(ch_row, ch_column, score);
		}	
	}
	return;
}

float SubstMatrix::rawGet(int x, int y) const {
	return data[x * dim + y];
}
void  SubstMatrix::rawSet(int x, int y, float el) {
	data[x * dim + y] = el;
}

float SubstMatrix::getScore(unsigned char a, unsigned char b) const {
	int x = tr_map[a];
	int y = tr_map[b];
	return rawGet(x, y);
}
void SubstMatrix::setScore(unsigned char a, unsigned char b, float el) {
	int x = tr_map[a];
	int y = tr_map[b];
	rawSet(x, y, el);
}

int	 SubstMatrix::tr_mapGet(unsigned char ch) const {
	return tr_map[ch];
}
void SubstMatrix::tr_mapSet(unsigned char ch, int val) {
	tr_map[ch] = val;
}
bool SubstMatrix::tr_mapEmpty(unsigned char ch) const {
	return (-1 == tr_mapGet(ch));
}

int SubstMatrix::getDim() const {
	return dim;
}
DNAAlphabet* SubstMatrix::getAlphabet() const {
	return dnaAlphabet;
}
char SubstMatrix::getKthSimbol(int k) const {
	unsigned char res = 0;
	if (k == dim - 1) {
		res = dnaAlphabet->getDefaultSymbol();
	} else if (0 <= k && k < dim-1) {
		for (int i = 0; i < 256; i++) {
			if (k == tr_map[i]) {
				res = i;
				break;
			}
		}
	}
	return res;
}

void SubstMatrix::dump() const {
	log.trace("--------------------------------------\n");
	for (int i = 0; i < dim; i++) {
		for (int j = 0; j < dim; j++) {
			log.trace(QString("%1 ").arg(data[i * dim + j]));
		}
		log.trace("\n");
	}
	log.trace("\n");
}

} // namespace
