/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SaveDocumentTask.h"

#include <core_api/IOAdapter.h>
#include <core_api/DocumentModel.h>
#include <core_api/Log.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <gobjects/GObjectUtils.h>
#include <util_gui/DialogUtils.h>
#include <util_gui/GUIUtils.h>

#include <QtGui/QMessageBox>

#include <memory>

namespace GB2 {

static LogCategory log(ULOG_CAT_CORE_SERVICES);

SaveDocumentTask::SaveDocumentTask(Document* _doc, IOAdapterFactory* _io, const QString& _url, bool ensureExt)
: Task(tr("save_document_task_name"), TaskFlag_None), doc(_doc), iof(_io), url(_url), flags(SaveDoc_Overwrite)
{
	assert(doc!=NULL);
	if (iof == NULL) {
		iof = doc->getIOAdapterFactory();
	}
	if (url.isEmpty()) {
		url = doc->getURL();
	}
    if (ensureExt) {
        url = DialogUtils::ensureFileExt(url, doc->getDocumentFormat()->getSupportedDocumentFileExtensions());
    }
	lock = NULL;
}

SaveDocumentTask::SaveDocumentTask(Document* _doc, uint f)
: Task(tr("save_document_task_name"), TaskFlag_None), doc(_doc), iof(doc->getIOAdapterFactory()), url(doc->getURL()), flags(f)
{
    assert(doc!=NULL);
    lock = new StateLock(getTaskName());
}


void SaveDocumentTask::prepare() {
    if (doc.isNull()) {
        setError("Document was removed");
        return;
    }
    lock = new StateLock(getTaskName());
	doc->lockState(lock);
}

void SaveDocumentTask::run() {
    if (flags & SaveDoc_Roll && !DialogUtils::rollFile(url, stateInfo, &log)) {
        return;
    }

    log.info(tr("Saving document %1\n").arg(url));
    DocumentFormat* df = doc->getDocumentFormat();

    if (flags & SaveDoc_Append) {
        std::auto_ptr<IOAdapter> io(iof->createIOAdapter());
        if (!io->open(url, IOAdapterMode_Append)) {
            setError(Translations::errorOpeningFileWrite(url));
            return;
        }
        df->storeDocument(doc, stateInfo, io.get());
    } else {
	    df->storeDocument(doc, stateInfo, iof, url);
    }
}

Task::ReportResult SaveDocumentTask::report() {
    if (lock!=NULL) {
        assert(!doc.isNull());
        doc->unlockState(lock);
        delete lock;
        lock = NULL;
    }
    if (hasErrors() || doc.isNull()) {
        return ReportResult_Finished;
    }
    if (url == doc->getURL() && iof == doc->getIOAdapterFactory()) {
		doc->makeClean();
	}
    if (flags & SaveDoc_DestroyAfter) {
        doc->unload();
        delete doc;
    }
	return Task::ReportResult_Finished;
}


//////////////////////////////////////////////////////////////////////////
/// save multiple

SaveMiltipleDocuments::SaveMiltipleDocuments(const QList<Document*>& docs, bool askBeforeSave)
: Task(tr("save_multiple_documents_task_name"), TaskFlag_NoRun)
{
    bool saveAll = false;
    foreach(Document* doc, docs) {
        bool save=true;
        if (askBeforeSave) {
            QMessageBox::StandardButtons buttons = QMessageBox::StandardButtons(QMessageBox::Yes) | QMessageBox::No;
            if (docs.size() > 1) {
                buttons = buttons | QMessageBox::YesToAll | QMessageBox::NoToAll;
            }

            QMessageBox::StandardButton res = saveAll ? QMessageBox::YesToAll : QMessageBox::question(NULL,
                tr("save_doc_title"), tr("save_doc_text: %1").arg(doc->getURL()),
                buttons, QMessageBox::Yes);

            if (res == QMessageBox::NoToAll) {
                break;
            }
            if (res == QMessageBox::YesToAll) {
                saveAll = true;
            }
            if (res == QMessageBox::No) {
                save = false;
            }
        }
        if (save) {
            addSubTask(new SaveDocumentTask(doc));
        }
    }
}


QList<Document*> SaveMiltipleDocuments::findModifiedDocuments(const QList<Document*>& docs) {
	QList<Document*> res;
	foreach(Document* doc, docs) {
		if (doc->isTreeItemModified()) {
			res.append(doc);
		}
	}
	return res;
}

//////////////////////////////////////////////////////////////////////////
// unload document
UnloadDocumentTask::UnloadDocumentTask(Document* _doc, bool save)
: Task(tr("unload_document_%1").arg(_doc->getURL()), TaskFlag_NoRun), doc(_doc), saveTask(NULL)
{
    if (save) {
        saveTask = new SaveDocumentTask(doc);
        addSubTask(saveTask);
    }
}

Task::ReportResult UnloadDocumentTask::report() {
    if (doc.isNull() || !doc->isLoaded()) {
        return Task::ReportResult_Finished;
    }
    propagateSubtaskError();
    QString errPrefix = tr("Document '%1' can't be unloaded: ").arg(doc->getName());
    if (hasErrors()) {
        assert(saveTask!=NULL);
        log.error(errPrefix +  tr("save failed!"));
        return Task::ReportResult_Finished;
    }
    QString error = checkSafeUnload(doc);
    if (!error.isEmpty()) {
        stateInfo.setError(errPrefix + error);
        log.error(stateInfo.getError());
        return Task::ReportResult_Finished;
    }
    bool ok = doc->unload();
    if (!ok) {
        stateInfo.setError(errPrefix + tr("unexpected error"));
        log.error(stateInfo.getError());
    }
    return Task::ReportResult_Finished;
}

void UnloadDocumentTask::runUnloadTaskHelper(const QList<Document*>& docs, UnloadDocumentTask_SaveMode sm) {
    QMap<Document*, QString> failedToUnload;

    // document can be unloaded if there are no active view with this doc + it's not state locked by user
    TriState saveAll = sm == UnloadDocumentTask_SaveMode_Ask ? TriState_Unknown :
        (sm == UnloadDocumentTask_SaveMode_NotSave ? TriState_No : TriState_Yes);

    foreach(Document* doc, docs) {
        QString err = checkSafeUnload(doc);
        if (!err.isEmpty()) {
            failedToUnload[doc] = err;
            continue;
        }
        bool saveCurrentDoc = doc->isModified() && saveAll == TriState_Yes;
        if (doc->isModified() && saveAll == TriState_Unknown) {

            QMessageBox::StandardButtons buttons = QMessageBox::StandardButtons(QMessageBox::Yes) | QMessageBox::No;
            if (docs.size() > 1) {
                buttons = buttons | QMessageBox::YesToAll | QMessageBox::NoToAll;
            }

            QMessageBox::StandardButton res = saveAll ? QMessageBox::YesToAll : QMessageBox::question(NULL,
                tr("save_doc_title"), tr("save_doc_text: %1").arg(doc->getURL()),
                buttons, QMessageBox::Yes);

            if (res == QMessageBox::NoToAll) {
                saveAll = TriState_No;
            } else  if (res == QMessageBox::YesToAll) {
                saveAll = TriState_Yes;
                saveCurrentDoc = true;
            } else if (res == QMessageBox::No) {
                saveCurrentDoc = false;
            } else {
                assert(res == QMessageBox::Yes);
                saveCurrentDoc = true;
            }
        }
        AppContext::getTaskScheduler()->registerTopLevelTask(new UnloadDocumentTask(doc, saveCurrentDoc));
    }

    if (!failedToUnload.isEmpty()) {
        QString text = tr("document_failed_to_unload:")+"<br>";
        foreach(Document* doc, failedToUnload.keys()) {
            text+=doc->getName()+" : " + failedToUnload[doc] + "<br>";
        }
        QMessageBox::warning(NULL, tr("Warning"), text);
    }
}

QString UnloadDocumentTask::checkSafeUnload(Document* doc) {
    bool hasViews = !GObjectViewUtils::findViewsWithAnyOfObjects(doc->getObjects()).isEmpty();
    if (hasViews) {
        return tr("document has active views");
    }

    bool liveLocked = doc->hasLocks(StateLockableTreeFlags_ItemAndChildren, StateLockFlag_LiveLock);
    if (liveLocked) {
        return tr("document is live-locked");
    }

    return QString();
}

//////////////////////////////////////////////////////////////////////////
// save a copy and add to project
SaveCopyAndAddToProjectTask::SaveCopyAndAddToProjectTask(Document* doc, IOAdapterFactory* iof, const QString& url, bool ensureExt)
: Task (tr("Save a copy %1").arg(url), TaskFlags_NR_FOSCOE), url(url)
{
    origURL = doc->getURL();
    df = doc->getDocumentFormat();
    hints = doc->getGHintsMap();
    saveTask = new SaveDocumentTask(doc, iof, url, ensureExt);
    addSubTask(saveTask);

    foreach(GObject* obj, doc->getObjects()) {
        info.append(UnloadedObjectInfo(obj));
    }
}

Task::ReportResult SaveCopyAndAddToProjectTask::report() {
    if (hasErrors() || isCanceled()) {
        return ReportResult_Finished;
    }
    Project* p = AppContext::getProject();
    if (p == NULL) {
        setError(tr("No active project found"));
        return ReportResult_Finished;
    }
    if (p->isStateLocked()) {
        setError(tr("Project is locked"));
        return ReportResult_Finished;
    }
    const QString& url = saveTask->getURL();
    if (p->findDocumentByURL(url)) {
        setError(tr("Document is already added to the project %1").arg(url));
        return ReportResult_Finished;
    }
    Document* doc = new Document(df, saveTask->getIOAdapterFactory(), url, info, hints);
    foreach(GObject* o, doc->getObjects()) {
        GObjectUtils::updateRelationsURL(o, origURL, url);
    }
    doc->setModified(false);
    p->addDocument(doc);
    return ReportResult_Finished;
}

///////////////////////////////////////////////////////////////////////////
// relocate task

RelocateDocumentTask::RelocateDocumentTask(const QString& fu, const QString& tu)
: Task (tr("Relocate document %1 -> %2").arg(fu).arg(tu), TaskFlag_NoRun), fromURL(fu), toURL(tu)
{
}

Task::ReportResult RelocateDocumentTask::report() {
    Project* p = AppContext::getProject();
    if (p == NULL) {
        setError(tr("No active project found"));
        return ReportResult_Finished;
    }
    if (p->isStateLocked()) {
        setError(tr("Project is locked"));
        return ReportResult_Finished;
    }
    Document* d = p->findDocumentByURL(fromURL);
    if (d == NULL) {
        setError(tr("Document not found: %1").arg(fromURL));
        return ReportResult_Finished;
    }
    if (d->isLoaded()) {
        setError(tr("Only unloaded objects can be relocated"));
        return ReportResult_Finished;
    }

    d->setURL(toURL);
    if (fromURL.endsWith(d->getName())) { // if document name is default -> update it too
        d->setName(QFileInfo(toURL).fileName());
    }

    //update relations to new url
    foreach(Document* d, p->getDocuments()) {
        foreach(GObject* o, d->getObjects()) {
            GObjectUtils::updateRelationsURL(o, fromURL, toURL);
        }
    }

    return ReportResult_Finished;
}

}//namespace
