/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "LocalDomain.h"

#include <workflow/Schema.h>
#include <workflow_support/WorkflowSettings.h>
#include <core_api/Log.h>

namespace GB2 {
namespace LocalWorkflow {

const QString LocalDomainFactory::ID("domain.local.bio");

static LogCategory log(ULOG_CAT_WD);

BaseWorker::BaseWorker(Actor* a, bool autoTransitBus) : actor(a) {
    foreach(Port* p, a->getPorts()) {
        if (qobject_cast<BusPort*>(p)) {
            IntegralBus* bus = new IntegralBus(p);
            ports.insert(p->getId(), bus);
            p->setPeer(bus);
        }
    }
    if (autoTransitBus) {
        foreach(Port* p, a->getInputPorts()) {
            IntegralBus* bus = p->castPeer<IntegralBus>();
            foreach(Port* op, a->getOutputPorts()) {
                if (p->isInput() != op->isInput()) {
                    IntegralBus* ob = op->castPeer<IntegralBus>();
                    ob->addComplement(bus);
                    bus->addComplement(ob);
                }
            }
        }
    }
    a->setPeer(this);
    failFast = WorkflowSettings::failFast();
}

bool BaseWorker::addCommunication(const QString& id, CommunicationChannel* ch) {
    Q_UNUSED(id);
    Q_UNUSED(ch);
    assert(0);
    return false;
}

BaseWorker::~BaseWorker()
{
    foreach(Port* p, actor->getPorts()) {
        if (qobject_cast<BusPort*>(p)) {
            p->setPeer(NULL);
        }
    }
    qDeleteAll(ports.values());
    actor->setPeer(NULL);
}

void SimplestSequentialScheduler::init() {
    foreach(Actor* a, schema->procs) {
        a->castPeer<BaseWorker>()->init();
    }
}

bool SimplestSequentialScheduler::isReady() {
    foreach(Actor* a, schema->procs) {
        if (a->castPeer<BaseWorker>()->isReady()) {
            return true;
        }
    }
    return false;
}

Task* SimplestSequentialScheduler::tick() {
    foreach(Actor* a, schema->procs) {
        if (a->castPeer<BaseWorker>()->isReady()) {
            lastWorker = a->castPeer<BaseWorker>();
            return lastTask = lastWorker->tick();
        }
    }
    assert(0);
    return NULL;
}

bool SimplestSequentialScheduler::isDone() {
    foreach(Actor* a, schema->procs) {
        if (!a->castPeer<BaseWorker>()->isDone()) {
            return false;
        }
    }
    return true;
}

void SimplestSequentialScheduler::cleanup() {
    foreach(Actor* a, schema->procs) {
        a->castPeer<BaseWorker>()->cleanup();
    }
}

SimplestSequentialScheduler::~SimplestSequentialScheduler()
{
}

GB2::Workflow::WorkerState SimplestSequentialScheduler::getWorkerState( ActorId id)
{
    Actor* a = schema->actorById(id);
    assert(a);
    BaseWorker* w = a->castPeer<BaseWorker>();
    if (lastWorker == w) {
        Task* t = lastTask;
        if (w->isDone() && t && t->isFinished()) {
            return WorkerDone;
        }
        return WorkerRunning;
    }
    if (w->isDone()) {
        return WorkerDone;
    } else if (w->isReady()) {
        return WorkerReady;
    }
    return WorkerWaiting;
}

Worker* LocalDomainFactory::createWorker(Actor* a) {
    Worker* w = NULL;
    DomainFactory* f = getById(a->getProto()->getId());
    if (f) {
        w = f->createWorker(a);
#ifdef _DEBUG
        assert(w);
        BaseWorker* bw = dynamic_cast<BaseWorker*>(w);
        assert(qobject_cast<BaseWorker*>(bw));
        assert(bw == a->getPeer());
#endif
    }
    
    return w;    
}

static CommunicationSubject* setupBus(Port* p) {
    QString id = p->getId();
    BaseWorker* worker = p->owner()->castPeer<BaseWorker>();
    assert(worker);
    CommunicationSubject* subj = worker;
    IntegralBus* bus = qobject_cast<IntegralBus*>(p->castPeer<QObject>());
    if (bus) {
        assert(subj->getCommunication(id) == dynamic_cast<CommunicationChannel*>(bus));
        subj = bus;
    } else if (subj) {
        assert(0);
        bus = new IntegralBus(p);
        p->setPeer(bus);
        subj->addCommunication(id, bus);
        subj = bus;
        foreach(Port* op, p->owner()->getPorts()) {
            if (p->isInput() != op->isInput()) {
                IntegralBus* ob = qobject_cast<IntegralBus*>(op->castPeer<QObject>());
                if (ob) {
                    ob->addComplement(bus);
                    bus->addComplement(ob);
                }
            }
        }
    }
    return subj;
}

CommunicationChannel* LocalDomainFactory::createConnection(Link* l) {
    SimpleQueue* cc = NULL;
    QString srcId = l->source()->getId();
    QString dstId = l->destination()->getId();
    CommunicationSubject* src = setupBus(l->source());
    CommunicationSubject* dst = setupBus(l->destination());
    if (src && dst) {
        cc = new SimpleQueue();
        src->addCommunication(srcId, cc);
        dst->addCommunication(dstId, cc);
    }
    l->setPeer(cc);
    return cc;
}

Scheduler* LocalDomainFactory::createScheduler(Schema* sh, Configuration*) {
    SimplestSequentialScheduler* sc = new SimplestSequentialScheduler(sh);
    //sh->setPeer(sc);
    return sc;
}

void LocalDomainFactory::destroy( Scheduler* sh, Schema* schema)
{
    foreach(Link* l, schema->flows) {
        delete l->castPeer<SimpleQueue>();
        l->setPeer(NULL);
    }

    foreach(Actor* a, schema->procs) {
        delete a->castPeer<BaseWorker>();
    }

    delete sh;
}

} // Workflow namespace
} // GB2 namespace
