/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SWAlgorithmPlugin.h"

#include "SWAlgorithmTask.h"
#include "SWTaskFactory.h"

#include <util_gui/GUIUtils.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVConstants.h>
#include <gobjects/AnnotationTableObject.h>

#include <test_framework/xmltest/XMLTestFormat.h>
#include <test_framework/GTest.h>
#include <test_framework/GTestFrameworkComponents.h>
#include <util_algorithm/GAutoDeleteList.h>

#include <core_api/AppContext.h>
#include <core_api/AppResources.h>
#include <core_api/CudaGpuRegistry.h>
#include <core_api/AtiStreamGpuRegistry.h>

#include <util_smith_waterman/SmithWatermanTests.h>
#include <util_ov_annotated_dna/ADVConstants.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVUtils.h>

#include <core_api/SmithWatermanTaskFactoryRegistry.h>
#include <core_api/Log.h>

namespace GB2 {

    static LogCategory log(ULOG_CAT_SW);

extern "C" Q_DECL_EXPORT Plugin* GB2_PLUGIN_INIT_FUNC() {
#ifdef SW2_BUILD_WITH_SSE2
    bool runningOnSSEProc = AppResourcePool::isSSE2Enabled();
    if( !runningOnSSEProc ) {
        return 0;
    }
#endif
    SWAlgorithmPlugin * plug = new SWAlgorithmPlugin();
    return plug;
}

SWAlgorithmPlugin::SWAlgorithmPlugin() 
: Plugin(tr("Smith-Waterman "), tr("An implementation of Smith-Waterman algorithm"))
{
    //initilazing ADV context
    if (AppContext::getMainWindow()) {
        ctxADV = new SWAlgorithmADVContext(this);
        ctxADV->init();
    }

    //Smith-Waterman algorithm tests
    GTestFormatRegistry * tfr = AppContext::getTestFramework()->getTestFormatRegistry();
    XMLTestFormat *xmlTestFormat = qobject_cast<XMLTestFormat*>(tfr->findFormat("XML"));
    assert(xmlTestFormat!=NULL);    

    GB2::GAutoDeleteList<GB2::XMLTestFactory>* l = new GB2::GAutoDeleteList<GB2::XMLTestFactory>(this);
    l->qlist = SWAlgorithmTests::createTestFactories();

    foreach(XMLTestFactory* f, l->qlist) { 
        bool res = xmlTestFormat->registerTestFactory(f);
        Q_UNUSED(res);
        assert(res);
    }    
    SmithWatermanTaskFactoryRegistry* swar = AppContext::getSmithWatermanTaskFactoryRegistry();

    log.details("registration classic impl of smith-waterman");        
    swar->registerAlgorithm(new SWTaskFactory(classic), QString("Classic 2"));    

#ifdef SW2_BUILD_WITH_SSE2
    log.details("registration SSE2 impl of smith-waterman");
    swar->registerAlgorithm(new SWTaskFactory(sse2), QString("SSE2"));
#endif    

    this->connect(AppContext::getPluginSupport(), SIGNAL(si_allStartUpPluginsLoaded()), SLOT(regDependedIMPLFromOtherPlugins()));    
}

QList<XMLTestFactory*> SWAlgorithmTests::createTestFactories() {
    QList<XMLTestFactory*> res;
    res.append(GTest_SmithWatermnan::createFactory());
    return res;
}

//SLOT
void SWAlgorithmPlugin::regDependedIMPLFromOtherPlugins() {
    log.details("run SLOT regDependedIMPLFromOtherPlugins()");

    SmithWatermanTaskFactoryRegistry* swar = AppContext::getSmithWatermanTaskFactoryRegistry();

#ifdef SW2_BUILD_WITH_CUDA
    if ( !AppContext::getCudaGpuRegistry()->empty() ) {
        log.details("registration CUDA impl of smith-waterman");        
        swar->registerAlgorithm(new SWTaskFactory(cuda), QString("CUDA"));
    }
#endif

#ifdef SW2_BUILD_WITH_ATISTREAM
    if ( !AppContext::getAtiStreamGpuRegistry()->empty() ) {
        log.details("registration ATIStream impl of smith-waterman");        
        swar->registerAlgorithm(new SWTaskFactory(atistream), QString("ATI Stream"));
    }
#endif

}

SWAlgorithmADVContext::SWAlgorithmADVContext(QObject* p) : 
GObjectViewWindowContext(p, ANNOTATED_DNA_VIEW_FACTORY_ID)
{
}

void SWAlgorithmADVContext::initViewContext(GObjectView* view) {
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(view);
    assert(av != NULL);
    ADVGlobalAction* a = new ADVGlobalAction(av, QIcon(":core/images/sw.png"), tr("Find pattern [Smith-Waterman]"), 15);

    a->setShortcut(QKeySequence(Qt::CTRL | Qt::SHIFT | Qt::Key_F));
    a->setShortcutContext(Qt::WindowShortcut);
    av->getWidget()->addAction(a);

    connect(a, SIGNAL(triggered()), SLOT(sl_search()));
}


void SWAlgorithmADVContext::sl_search() {
    GObjectViewAction* action = qobject_cast<GObjectViewAction*>(sender());
    assert(0 != action);

    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(action->getObjectView());
    assert(av != NULL);

    ADVSequenceObjectContext* seqCtx = av->getSequenceInFocus();
    SmithWatermanDialog smv(av->getWidget(), seqCtx, &dialogConfig);
    smv.exec();
}

} //namespace

