/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <cassert>
#include <QtNetwork/QHostInfo>

#include <core_api/AppContext.h>
#include <core_api/Log.h>

#include "LocalTask.h"
#include "LocalTaskManager.h"

#define LOCAL_TASK_MANAGER_LOG_CAT "Local task manager"

using namespace GB2;

template< class T >
void setIfYouCan( T* to, const T & val ) {
    if( NULL != to ) {
        *to = val;
    }
}

namespace GB2 {

static LogCategory log( LOCAL_TASK_MANAGER_LOG_CAT );

const QUuid LocalTaskManager::uuid = QUuid::createUuid();
const QString LocalTaskManager::TASK_WITH_GIVEN_ID_NOT_FOUND = LocalTaskManager::tr( "No Local task found with '%1' id" );
const int LocalTaskManager::TASK_SECONDS_TO_LIVE = 120;
const int LocalTaskManager::DELETE_TASK_UPDATE_TIME = 1000;

LocalTaskManager::LocalTaskManager() {
    log.trace( tr( "Local task manager created" ) );
    connect( &deleteTasksTimer, SIGNAL( timeout() ), SLOT( sl_deleteTaskTimerUpdate() ) );
    deleteTasksTimer.start( DELETE_TASK_UPDATE_TIME );
}

LocalTaskManager::~LocalTaskManager() {
    log.trace( tr( "Local task manager destroyed" ) );
}

RemoteTaskError LocalTaskManager::runTask( const QString & taskFactoryId, const QVariant & settings, qint64 * taskId ) {
    assert( !taskFactoryId.isEmpty() );
    
    LocalTaskFactoryRegistry * ltfr = AppContext::getLocalTaskFactoryRegistry();
    assert( NULL != ltfr );
    const LocalTaskFactory * factory = ltfr->getLocalTaskFactory( taskFactoryId );
    if( NULL == factory ) {
        return RemoteTaskError( false, tr( "Task factory '%1' not found" ).arg( taskFactoryId ) );
    }
    
    LocalTask * task = factory->createInstance( settings );
    if( NULL == task ) {
        return RemoteTaskError( false, tr( "'%1' factory cannot create task instance" ).arg( taskFactoryId ) );
    }
    
    task->setNoAutoDelete( true );
    assert( task->hasFlags( TaskFlag_NoAutoDelete ) );
    allTasks.insert( task->getTaskId(), task );
    
    connect( task, SIGNAL( si_stateChanged() ), SLOT( sl_taskStateChanged() ) );
    TaskScheduler * tskScheduler = AppContext::getTaskScheduler();
    assert( NULL != tskScheduler );
    tskScheduler->registerTopLevelTask( task );
    
    log.trace( tr( "Local task manager: %1 task was started" ).arg( taskFactoryId ) );
    
    setIfYouCan<qint64>( taskId, task->getTaskId() );
    return RemoteTaskError( true );
}

RemoteTaskError LocalTaskManager::cancelTask( qint64 id ) {
    Task * task =  getTaskById( id );
    if( NULL == task ) {
        return RemoteTaskError( false, TASK_WITH_GIVEN_ID_NOT_FOUND.arg( id ) );
    }
    task->cancel();
    
    log.trace( tr( "Local task manager: task with %1 id was canceled" ).arg( QString::number( id ) ) );
    
    return RemoteTaskError( true );
}

RemoteTaskError LocalTaskManager::deleteTask( qint64 id ) {
    Task * task = getTaskById( id );
    if( NULL == task ) {
        return RemoteTaskError( false, TASK_WITH_GIVEN_ID_NOT_FOUND.arg( QString::number( id ) ) );
    }
    if( Task::State_Finished != task->getState() ) {
        return RemoteTaskError( false, tr( "Cannot delete running task" ) );
    }
    deleteTask( task );
    
    log.trace( tr( "Local task manager: task with %1 id was deleted" ).arg( QString::number( id ) ) );
    
    return RemoteTaskError( true );
}

void LocalTaskManager::deleteTask( Task * task ) {
    assert( NULL != task );
    if( deletedTasks.contains( task ) ) {
        return;
    }
    assert( tasksToDel.contains( task ) );
    {
        assert( 1 == tasksToDel.remove( task ) );
    }
    {
        assert( 1 == allTasks.remove( task->getTaskId() ) );
    }
    deletedTasks.append( task );
    delete task;
}

RemoteTaskError LocalTaskManager::getTaskCancelFlag( qint64 id, bool * cancelFlag ) const {
    Task * task = getTaskById( id );
    if( NULL == task ) {
        return RemoteTaskError( false, TASK_WITH_GIVEN_ID_NOT_FOUND.arg( QString::number( id ) ) );
    }
    setIfYouCan<bool>( cancelFlag, task->isCanceled() );
    return RemoteTaskError( true );
}

RemoteTaskError LocalTaskManager::getTaskState( qint64 id, Task::State * state ) const {
    Task * task = getTaskById( id );
    if( NULL == task ) {
        return RemoteTaskError( false, TASK_WITH_GIVEN_ID_NOT_FOUND.arg( QString::number( id ) ) );
    }
    setIfYouCan<Task::State>( state, task->getState() );
    return RemoteTaskError( true );
}

RemoteTaskError LocalTaskManager::getTaskProgress( qint64 id, int * progress ) const {
    Task * task = getTaskById( id );
    if( NULL == task ) {
        return RemoteTaskError( false, TASK_WITH_GIVEN_ID_NOT_FOUND.arg( QString::number( id ) ) );
    }
    setIfYouCan<int>( progress, task->getProgress() );
    return RemoteTaskError( true );
}

RemoteTaskError LocalTaskManager::getTaskResult( qint64 id, QVariant * taskResult ) const {
    Task * task = getTaskById( id );
    if( NULL == task ) {
        return RemoteTaskError( false, TASK_WITH_GIVEN_ID_NOT_FOUND.arg( QString::number( id ) ) );
    }
    LocalTask * localTask = dynamic_cast< LocalTask * >( task );
    if( NULL == localTask ) {
        return RemoteTaskError( false, tr( "Cannot convert task with %1 id to LocalTask" ).arg( QString::number( id ) ) );
    }
    if( Task::State_Finished != localTask->getState() ) {
        return RemoteTaskError( false, tr( "Cannot get result from unfinished task" ) );
    }
    
    const LocalTaskResult * result = localTask->getResult();
    if( NULL == result ) {
        return RemoteTaskError( false, tr( "LocalTask didn't produced result" ) );
    }
    setIfYouCan<QVariant>( taskResult, result->serialize() );
    return RemoteTaskError( true );
}

RemoteTaskError LocalTaskManager::getTaskError( qint64 id, QString * errMsg ) const {
    Task * task = getTaskById( id );
    if( NULL == task ) {
        return RemoteTaskError( false, TASK_WITH_GIVEN_ID_NOT_FOUND.arg( id ) );
    }
    setIfYouCan<QString>( errMsg, task->getError() );
    return RemoteTaskError( true );
}

QUuid LocalTaskManager::getUuid() {
    return LocalTaskManager::uuid;
}

QList<qint64> LocalTaskManager::getTasks()const {
    return allTasks.keys();
}

void LocalTaskManager::sl_taskStateChanged() {
    Task * task = qobject_cast< Task* >( sender() );
    if( NULL == task ) {
        return;
    }
    
    if( Task::State_Finished == task->getState() ) {
        tasksToDel.insert( task, TASK_SECONDS_TO_LIVE );
    }
}

Task * LocalTaskManager::getTaskById( qint64 id ) const {
    return allTasks.value( id, NULL );
}

QStringList LocalTaskManager::getServicesList() {
    LocalTaskFactoryRegistry * ltfr = AppContext::getLocalTaskFactoryRegistry();
    assert( NULL != ltfr );
    return ltfr->getLocalTaskFactoryIds();
}

QString LocalTaskManager::getHostName() {
    return QHostInfo::localHostName();
}

void LocalTaskManager::sl_deleteTaskTimerUpdate() {
    QMap< Task*, int >::iterator it = tasksToDel.begin();
    while( tasksToDel.end() != it ) {
        int & val = it.value();
        --val;
        ++it;
    }
    
    QList< Task* > tasks = tasksToDel.keys();
    foreach( Task * task, tasks ) {
        assert( NULL != task );
        if( 0 == tasksToDel.value( task ) ) {
            deleteTask( task );
        }
    }
}

} // GB2
