/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtCore/QTimer>

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/Counter.h>

#include "RemoteTask.h"

#define REMOTE_TASK_LOG_CAT "Remote task"

namespace GB2 {

static LogCategory log( REMOTE_TASK_LOG_CAT );

/*********************************************
 * RemoteTask
 *********************************************/
RemoteTask::RemoteTask( const QString & ltfId, const LocalTaskSettings & set, RemoteMachine * m ) :
Task( "", TaskFlag_None ), taskFactoryId( ltfId ), settings( set.serialize() ), machine( m ),
stateOnRemoteMachine( State_New ), canceledOnRemoteMachine( false ),
result( NULL ), networkError( false ), eventLoop( NULL ), changer( NULL ) {
    GCOUNTER( cvar, tvar, "REMOTE_TASK" );
    if( taskFactoryId.isEmpty() ) {
        setTaskName( tr( "Remote task" ) );
        setError( tr( "No task given to remote task" ) );
        return;
    }
    setTaskName( tr( "Remote task: %1" ).arg( taskFactoryId ) );
    if( NULL == machine ) {
        setError( tr( "No machine given to remote task" ) );
        return;
    }
    
    log.message( LogLevel_TRACE, tr( "Remote task created" ) );
}

RemoteTask::~RemoteTask() {
    delete result;
}

void RemoteTask::run() {
    if( hasErrors() ) {
        return;
    }
    
    RemoteTaskError err = machine->runRemoteTask( taskFactoryId, settings, &taskIdOnRemoteMachine );
    if( !err.getOk() ) {
        setRemoteMachineError( tr( "Cannot run remote task: '%1'" ).arg( err.getMsg() ) );
        return;
    }
    log.message( LogLevel_TRACE, tr( "Remote %1 task started" ).arg( taskFactoryId ) );
    
    changer = new RemoteTaskChanger( this );
    eventLoop = new QEventLoop();
    QTimer::singleShot( TIMER_UPDATE_TIME, changer, SLOT( sl_RemoteTaskTimerUpdate() ) );
    QTimer::singleShot( TASK_ON_REMOTE_MACHINE_STARTED_TIMEOUT, changer, SLOT( sl_isStartedOnRemoteMachine() ) );
    eventLoop->exec();
    delete eventLoop;
    delete changer;
}

Task::ReportResult RemoteTask::report() {
    if( State_Finished != stateOnRemoteMachine ) {
        return ReportResult_CallMeAgain;
    }
    return ReportResult_Finished;
}

void RemoteTask::deleteRemoteMachineTask() {
    RemoteTaskError err = machine->deleteRemoteTask( taskIdOnRemoteMachine );
    if( !err.getOk() ) {
        log.message( LogLevel_TRACE, tr( "Cannot delete task on remote machine" ) );
    }
}

LocalTaskResult * RemoteTask::getResult() {
    return result;
}

bool RemoteTask::hasNetworkErrorOccurred() const {
    return networkError;
}

void RemoteTask::setRemoteMachineError( const QString & msg ) {
    setError( msg );
    networkError = true;
    finilize();
}

void RemoteTask::finilize() {
    stateOnRemoteMachine = State_Finished; /* to make RemoteTask finished after it got error */
    if( NULL != eventLoop && eventLoop->isRunning() ) {
        eventLoop->exit();
    }
}

void RemoteTask::deleteRemoteTaskAndFinilize() {
    deleteRemoteMachineTask();
    finilize();
}

RemoteMachine * RemoteTask::getRemoteMachine() const {
    return machine;
}

/*********************************************
* RemoteTaskChanger
*********************************************/
RemoteTaskChanger::RemoteTaskChanger( RemoteTask * t ) : task( t ) {
    assert( NULL != task );
}

void RemoteTaskChanger::sl_RemoteTaskTimerUpdate() {
    log.message( LogLevel_TRACE, tr( "Remote task with %1 id update started" ).arg( QString::number( task->taskIdOnRemoteMachine ) ) );
    
    if( task->hasErrors() ) {
        task->finilize();
        return;
    }
    RemoteTaskError err;
    
    // work with cancel flag
    if( task->isCanceled() ) {
        if( !task->canceledOnRemoteMachine ) {
            err = task->machine->cancelRemoteTask( task->taskIdOnRemoteMachine );
            if( !err.getOk() ) {
                task->setRemoteMachineError( RemoteTask::tr( "Cannot cancel remote task: '%1'" ).arg( err.getMsg() ) );
                return;
            }
            task->canceledOnRemoteMachine = true;
        }
        task->deleteRemoteTaskAndFinilize();
        return;
    } else { // !isCanceled()
        assert( !task->canceledOnRemoteMachine );
        bool canceled = false;
        err = task->machine->getRemoteTaskCancelFlag( task->taskIdOnRemoteMachine, &canceled );
        if( !err.getOk() ) {
            task->setRemoteMachineError( RemoteTask::tr( "Cannot get remote task cancel flag: '%1'" ).arg( err.getMsg() ) );
            return;
        }
        task->canceledOnRemoteMachine = canceled;
        if( task->canceledOnRemoteMachine ) {
            task->deleteRemoteTaskAndFinilize();
            return;
        }
    }
    
    // work with task state
    err = task->machine->getRemoteTaskState( task->taskIdOnRemoteMachine, &task->stateOnRemoteMachine ); // TODO: synchronize this state with current task state
    if( !err.getOk() ) {
        task->setRemoteMachineError( RemoteTask::tr( "Cannot get remote task state: '%1'" ).arg( err.getMsg() ) );
        return;
    }
    if( Task::State_Finished == task->stateOnRemoteMachine ) {
        QString taskErrMsg;
        err = task->machine->getRemoteTaskError( task->taskIdOnRemoteMachine, &taskErrMsg );
        if( !err.getOk() ) {
            task->setRemoteMachineError( RemoteTask::tr( "Cannot get remote task error: '%1'" ).arg( err.getMsg() ) );
            return;
        }
        if( !taskErrMsg.isEmpty() ) {
            task->setError( taskErrMsg );
            task->deleteRemoteTaskAndFinilize();
            return;
        }
        
        QVariant serializedResult;
        err = task->machine->getRemoteTaskResult( task->taskIdOnRemoteMachine, &serializedResult );
        if( !err.getOk() ) {
            task->setRemoteMachineError( RemoteTask::tr( "Cannot get remote task result: '%1'" ).arg( err.getMsg() ) );
            return;
        }
        LocalTaskFactoryRegistry * taskFactoryRegistry = AppContext::getLocalTaskFactoryRegistry();
        assert( NULL != taskFactoryRegistry );
        const LocalTaskFactory * taskFactory = taskFactoryRegistry->getLocalTaskFactory( task->taskFactoryId );
        assert( NULL != taskFactory );
        task->result = taskFactory->createResults( serializedResult );
        if( NULL == task->result ) {
            task->setError( RemoteTask::tr( "Cannot create task result from remote task" ) );
            task->deleteRemoteTaskAndFinilize();
            return;
        }
        
        task->deleteRemoteTaskAndFinilize();
        return;
    }
    
    // work with task progress
    int progress = 0;
    err = task->machine->getRemoteTaskProgress( task->taskIdOnRemoteMachine, &progress );
    if( !err.getOk() ) {
        task->setRemoteMachineError( RemoteTask::tr( "Cannot get remote task progress: '%1'" ).arg( err.getMsg() ) );
        return;
    }
    task->stateInfo.progress = 0 <= progress? progress : 0;
    
    QTimer::singleShot( RemoteTask::TIMER_UPDATE_TIME, this, SLOT( sl_RemoteTaskTimerUpdate() ) );
}

void RemoteTaskChanger::sl_isStartedOnRemoteMachine() {
    log.message( LogLevel_TRACE, tr( "is remote task started timeout triggered" ) );
    
    RemoteTaskError err = task->machine->getRemoteTaskState( task->taskIdOnRemoteMachine, &task->stateOnRemoteMachine );
    if( !err.getOk() || Task::State_New == task->stateOnRemoteMachine ) {
        task->machine->cancelRemoteTask( task->taskIdOnRemoteMachine );
        task->setError( RemoteTask::tr( "Task on remote machine was not started for %1 seconds. Aborted" ).
            arg( RemoteTask::TASK_ON_REMOTE_MACHINE_STARTED_TIMEOUT / 1000 ) );
        task->deleteRemoteTaskAndFinilize();
    }
}

} // GB2
