/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_TASK_SCHEDULER_IMPL_H_
#define _GB2_TASK_SCHEDULER_IMPL_H_

#include <core_api/Task.h>

#include <QtCore/QThread>
#include <QtCore/QTimer>

namespace GB2 {

class TaskInfo;
class AppResourcePool;
class AppResource;

class TaskThread : public QThread {
public:
    TaskThread(TaskInfo* _ti) : ti(_ti),finishEventListener(NULL) {}
    void run();

    TaskInfo* ti;
    QObject*  finishEventListener;
};


class TaskInfo {
public:
    TaskInfo(Task* t, TaskInfo* p) 
        : task(t), parentTaskInfo(p), wasPrepared(false), subtasksWereCanceled(false), selfRunFinished(false),
        hasLockedPrepareResources(false), hasLockedRunResources(false),
        prevProgress(0), numPreparedSubtasks(0), numRunningSubtasks(0), numFinishedSubtasks(0),  thread(NULL) {}
    
    virtual ~TaskInfo();

    //true if task state >= RUN && thread is finished or not used at all
    
    Task*           task;
    TaskInfo*       parentTaskInfo;
    QList<Task*>    newSubtasks;

    bool            wasPrepared;            // 'true' if prepare() was called for the task
    bool            subtasksWereCanceled;   // 'true' if canceled task has called cancel() on its subtasks
    bool            selfRunFinished;        // indicates that the 'run' method of this task was finished
    bool            hasLockedPrepareResources;  //true if there were resource locks for 'prepare' stage
    bool            hasLockedRunResources;      //true if there were resource locks for 'run' stage
    

    int             prevProgress;   //used for TaskProgress_Manual
    QString         prevDesc;

    int             numPreparedSubtasks;
    int             numRunningSubtasks;
    int             numFinishedSubtasks;

    TaskThread*     thread;

    inline int numActiveSubtasks() const {
        return numPreparedSubtasks+numRunningSubtasks;
    }


};

class GB2_COREAPI_EXPORT TaskSchedulerImpl : public TaskScheduler {
    Q_OBJECT
public:
    TaskSchedulerImpl(AppResourcePool* rp);
    ~TaskSchedulerImpl();

    virtual void registerTopLevelTask(Task* t);

    virtual void unregisterTopLevelTask(Task* t);

    const QList<Task*>& getTopLevelTasks() const {return topLevelTasks;}

    Task * getTopLevelTaskById( qint64 id ) const;

    QDateTime estimatedFinishTime(Task*) const;
    
    virtual void cancelTask(Task* t);
    
    virtual void cancelAllTasks();
    
    virtual QString getStateName(Task* t) const;

private slots:
    void update();
    void sl_threadFinished();

private:
    bool processFinishedTasks();
    void unregisterFinishedTopLevelTasks();
    void processNewSubtasks();
    void prepareNewTasks();
    void runReady();

    bool readyToFinish(TaskInfo* ti);
    bool addToPriorityQueue(Task* t, TaskInfo* parentInfo); //return true if added. Failure can be caused if a task requires resources
    void runThread(TaskInfo* pi);
    void stopTask(Task* t);
    void updateTaskProgressAndDesc(TaskInfo* ti);
    void promoteTask(TaskInfo* ti, Task::State newState);
    void deleteTask(Task* t);
    
    QString tryLockResources(Task* task, bool prepareStage, bool& hasLockedResourcesAfterCall); //returns error message
    void releaseResources(TaskInfo* ti, bool prepareStage);

    void propagateStateToParent(Task* t);
    void updateOldTasksPriority();
    void checkSerialPromotion(TaskInfo* pti, Task* subtask);

private:
    QTimer                  timer;
    QList<Task*>            topLevelTasks;
    QList<TaskInfo*>        priorityQueue;
    QList<TaskInfo*>        tasksWithNewSubtasks;
    QList<Task*>            newTasks;
    QStringList             stateNames;
    
    AppResourcePool*        resourcePool;
    AppResource*            threadsResource;
    bool                    stateChangesObserved;

};

} //namespace
#endif
