/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CreateSubalignimentTask.h"

#include <core_api/DocumentModel.h>
#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>
#include <core_api/ProjectModel.h>
#include <util_tasks/LoadDocumentTask.h>
#include <util_tasks/MultiTask.h>
#include <util_tasks/AddDocumentTask.h>

#include <core_services/DocumentFormatRegistryImpl.h>

#include <util_tasks/SaveDocumentTask.h>

#include <gobjects/GObjectUtils.h>


namespace GB2{

static LogCategory log(ULOG_CAT_USER_INTERFACE);

CreateSubalignimentTask::CreateSubalignimentTask(MAlignmentObject* _maObj, LRegion _window, 
											  const QStringList& _seqNames, const GUrl& _url, 
											    bool _addToProject, bool _saveImmediately )
: Task(tr("Create subaligniment task"), TaskFlags_NR_FOSCOE), 
maObj(_maObj), window(_window), seqNames(_seqNames), url(_url), 
addToProject(_addToProject), saveImmediately(_saveImmediately)
{
    curDoc = maObj->getDocument();
    if(url == curDoc->getURL() || url.isEmpty()){
        saveToAnother = false;
    }
}

void CreateSubalignimentTask::prepare(){
	
    newDoc = curDoc;
	QString ext = url.completeFileSuffix();

	if(curDoc->isStateLocked()){
		log.error(tr("Document is locked"));
		return;
	}

	DocumentFormat *dfd = NULL;
	DocumentFormatRegistry *dfr = AppContext::getDocumentFormatRegistry();
	QList<DocumentFormatId> dfIdList = dfr->getRegisteredFormats(), supportedFormats;
	foreach(DocumentFormatId dfId, dfIdList){
		DocumentFormat *df = AppContext::getDocumentFormatRegistry()->getFormatById(dfId);
		foreach(QString dfExt, df->getSupportedDocumentFileExtensions()){
			if(ext.contains(dfExt)){
				dfd = df; 
			}
		}
	}

	if(dfd == NULL){
		log.error(tr("Unable to detect format by file extension"));
		delete newDoc;
		return;
	}

	IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
	if(saveToAnother){
		QList<GObject*> GObjList = curDoc->getObjects();
		newDoc = dfd->createNewDocument(iof, url, curDoc->getGHintsMap());
		foreach(GObject* go, GObjList){
			GObject *cl = go->clone();
			newDoc->addObject(cl);
			if(go == maObj){
				maObj = static_cast<MAlignmentObject*> (cl);
			}
		}
		foreach(GObject* o, newDoc->getObjects()) {
			GObjectUtils::updateRelationsURL(o, curDoc->getURL(), url);
		}
	}
	
    //TODO: add "remove empty rows and columns" flag to crop function
	QSet<QString> rowNames;
    foreach ( const QString& name, seqNames) {
        rowNames.insert(name);
    }
    maObj->crop(window, rowNames);
    
	QList<Task*> tasks;
	if(saveImmediately){
		tasks.append(new SaveDocumentTask(newDoc, iof, url));
	}
	if(addToProject){
		tasks.append(new LoadUnloadedDocumentAndOpenViewTask(newDoc));
		tasks.append(new AddDocumentTask(newDoc));
	}
	if (tasks.size() != 0){
		addSubTask(new MultiTask(tr("Save document and open view"), tasks));
	}
}

Task::ReportResult CreateSubalignimentTask::report(){
    return ReportResult_Finished;
}

}
