/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GraphicsBranchItem.h"
#include "GraphicsButtonItem.h"
#include "TreeViewerUtils.h"

#include <QtGui/QGraphicsView>
#include <QtGui/QGraphicsScene>
#include <QtCore/QStack>

namespace GB2 {

const int GraphicsBranchItem::TEXT_SPACE = 8;
const int GraphicsBranchItem::SELECTED_PEN_WIDTH = 2;

void GraphicsBranchItem::collapse() {
    collapsed = !collapsed;
    QList<QGraphicsItem*> items = childItems();
    if (collapsed) {
        for (int i = 0, s = items.size(); i < s; ++i) {
            if (dynamic_cast<GraphicsBranchItem*>(items[i])) {
                items[i]->hide();
            }
        }
        QPen pen1(QColor(0, 0, 0));
        pen1.setWidth(SELECTED_PEN_WIDTH);
        pen1.setCosmetic(true);
        QGraphicsRectItem *r = new QGraphicsRectItem(0, -4, 16, 8, this);
        r->setPen(pen1);
    } else {
        for (int i = 0, s = items.size(); i < s; ++i) {
            if (dynamic_cast<QGraphicsRectItem*>(items[i])) {
                items[i]->setParentItem(NULL);
                scene()->removeItem(items[i]);
            } else {
                if (items[i] != getDistanceText() && items[i] != getNameText()) {
                    items[i]->show();
                }
            }
        }
    }
}

void GraphicsBranchItem::setSelected(bool sel) {
    QStack<GraphicsBranchItem*> stack;
    stack.push(this);
    while (!stack.empty()) {
        GraphicsBranchItem *top = stack.pop();
        if (top->button) {
            top->button->setSelected(sel);
        }
        QList<QGraphicsItem*> items = top->childItems();
        for (int i = 0, s = items.size(); i < s; ++i) {
            GraphicsBranchItem *item = dynamic_cast<GraphicsBranchItem*>(items[i]);
            if (item) {
                QPen pen1 = item->pen();
                pen1.setWidth(sel ? SELECTED_PEN_WIDTH : 0);
                item->setPen(pen1);
                stack.push(item);
            }
        }
    }
}

void GraphicsBranchItem::initText(qreal d) {
    QString str = QString::number(d, 'f', 3);
    int i = str.length() - 1;
    for (; i >= 0 && str[i] == '0'; --i) ;
    if (str[i] == '.')
        --i;
    str.truncate(i + 1);

    distanceText = new QGraphicsSimpleTextItem(str);
    distanceText->setFont(TreeViewerUtils::getFont());
    distanceText->setBrush(Qt::darkGray);
    QRectF rect = distanceText->boundingRect();
    distanceText->setPos(-rect.width(), 0);
    distanceText->setParentItem(this);
    distanceText->setZValue(1);
}

GraphicsBranchItem::GraphicsBranchItem(bool withButton)
: button(withButton ? new GraphicsButtonItem() : NULL), distanceText(NULL), nameText(NULL), collapsed(false) {
    setBrush(QColor(0, 0, 0));
    QPen pen1(QColor(0, 0, 0));
    pen1.setCosmetic(true);
    setPen(pen1);
    if (withButton) {
        button->setParentItem(this);
    }
}

GraphicsBranchItem::GraphicsBranchItem(const QString& name): button(NULL), distanceText(NULL), collapsed(false) {
    QPen pen1(Qt::darkGray);
    pen1.setStyle(Qt::DotLine);
    pen1.setCosmetic(true);
    setPen(pen1);

    width = 0;
    nameText = new QGraphicsSimpleTextItem(name);
    nameText->setFont(TreeViewerUtils::getFont());
    nameText->setBrush(Qt::darkGray);
    QRectF rect = nameText->boundingRect();
    nameText->setPos(GraphicsBranchItem::TEXT_SPACE, -rect.height() / 2);
    nameText->setParentItem(this);
    nameText->setZValue(1);
}

GraphicsBranchItem::GraphicsBranchItem(qreal d, bool withButton)
: button(withButton ? new GraphicsButtonItem() : NULL), distanceText(NULL), nameText(NULL), collapsed(false) {
    initText(d);
    QPen pen1(QColor(0, 0, 0));
    pen1.setCosmetic(true);
    if (d < 0)
        pen1.setStyle(Qt::DashLine);
    setPen(pen1);
    setBrush(QColor(0, 0, 0));
    if (withButton) {
        button->setParentItem(this);
    }
}

void GraphicsBranchItem::setWidth(qreal w) {
    if (width == w)
        return;

    setPos(pos().x() - width + w, pos().y());
    if (getDistanceText() != NULL) {
        QPointF pos = getDistanceText()->pos();
        getDistanceText()->setPos(pos.x() + (width - w) * 0.5, pos.y());
    }

    prepareGeometryChange();
    width = w;
}

}
