/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TreeViewer.h"
#include "TreeViewerFactory.h"
#include "GraphicsBranchItem.h"
#include "GraphicsButtonItem.h"
#include "TreeViewerUtils.h"
#include "CreateBranchesTask.h"
#include "CreateCircularBranchesTask.h"
#include "CreateUnrootedBranchesTask.h"
#include "GraphicsRectangularBranchItem.h"

#include <core_api/AppContext.h>

#include <core_api/DocumentModel.h>
#include <core_api/DocumentFormats.h>
#include <core_api/IOAdapter.h>
#include <core_api/ProjectModel.h>
#include <core_api/L10n.h>

#include <util_gui/DialogUtils.h>
#include <phyltree/CreatePhyTreeDialogController.h>
#include <phyltree/PhyTreeGeneratorRegistry.h>

#include <QtCore/QStack>

#include <QtGui/QVBoxLayout>
#include <QtGui/QMouseEvent>
#include <QtGui/QPrinter>
#include <QtGui/QPrintDialog>
#include <QtGui/QPainter>
#include <QtXml/QtXml>
#include <QtSvg/QSvgGenerator>
#include <QtGui/QGraphicsSimpleTextItem>
#include <QtGui/QGraphicsLineItem>
#include <QtGui/QMessageBox>

namespace GB2 {

TreeViewer::TreeViewer(const QString& viewName, GObject* obj, GraphicsRectangularBranchItem* _root, qreal s):
    GObjectView(TreeViewerFactory::ID, viewName),
    printAction(NULL),
    contAction(NULL),
    nameLabelsAction(NULL),
    distanceLabelsAction(NULL),
    captureTreeAction(NULL),
    exportAction(NULL),
    rectangularLayoutAction(NULL),
    circularLayoutAction(NULL),
    unrootedLayoutAction(NULL),
    labelsMenu(NULL),
    captureMenu(NULL),
    layoutMenu(NULL),
    ui(NULL),
    root(_root),
    scale(s)
{
    phyObject = qobject_cast<PhyTreeObject*>(obj);
    objects.append(phyObject);
    requiredObjects.append(phyObject);
}

void TreeViewer::createActions() {
    labelsMenu = new QMenu();
    nameLabelsAction = labelsMenu->addAction(tr("Show sequence names"));
    nameLabelsAction->setCheckable(true);
    nameLabelsAction->setChecked(true);
    distanceLabelsAction = labelsMenu->addAction(tr("Show distance labels"));
    distanceLabelsAction->setCheckable(true);
    distanceLabelsAction->setChecked(true);

    printAction = new QAction(QIcon(":/core/images/printer.png"), tr("Print tree"), ui);

    captureMenu = new QMenu();
    captureTreeAction = captureMenu->addAction(tr("Capture tree"));
    exportAction = captureMenu->addAction(tr("Export tree in SVG"));

    contAction = new QAction(QIcon(":core/images/align_tree_labels.png"), tr("Align name labels"), ui);
    contAction->setCheckable(true);

    layoutMenu = new QMenu();
    rectangularLayoutAction = layoutMenu->addAction(tr("Rectangular layout"));
    rectangularLayoutAction->setCheckable(true);
    rectangularLayoutAction->setChecked(true);
    circularLayoutAction = layoutMenu->addAction(tr("Circular layout"));
    circularLayoutAction->setCheckable(true);
    unrootedLayoutAction = layoutMenu->addAction(tr("Unrooted layout"));
    unrootedLayoutAction->setCheckable(true);

    QActionGroup* layoutGroup = new QActionGroup(ui);
    rectangularLayoutAction->setActionGroup(layoutGroup);
    circularLayoutAction->setActionGroup(layoutGroup);
    unrootedLayoutAction->setActionGroup(layoutGroup);
}

void TreeViewer::buildStaticToolbar(QToolBar* tb) {
    QToolButton* button = new QToolButton();
    button->setPopupMode(QToolButton::InstantPopup);
    QAction* defaultAction = new QAction(QIcon(":/core/images/text_ab.png"), "", button);
    button->setDefaultAction(defaultAction);
    button->setMenu(labelsMenu);
    tb->addWidget(button);

    tb->addAction(printAction);

    button = new QToolButton();
    button->setPopupMode(QToolButton::InstantPopup);
    defaultAction = new QAction(QIcon(":/core/images/cam2.png"), "", button);
    button->setDefaultAction(defaultAction);
    button->setMenu(captureMenu);
    tb->addWidget(button);

    tb->addAction(contAction);

    button = new QToolButton();
    button->setPopupMode(QToolButton::InstantPopup);
    defaultAction = new QAction(tr("Layout"), button);
    button->setDefaultAction(defaultAction);
    button->setMenu(layoutMenu);
    tb->addWidget(button);
}

QWidget* TreeViewer::createWidget() {
    assert(ui == NULL);
    ui = new TreeViewerUI(this);
    return ui;
}


////////////////////////////
// TreeViewerUI

const qreal TreeViewerUI::ZOOM_COEF = 1.2;
const qreal TreeViewerUI::MINIMUM_ZOOM = 1.0;
const qreal TreeViewerUI::MAXIMUM_ZOOM = 10.0; // TODO: make it depends on tree size
const int TreeViewerUI::MARGIN = 10;

TreeViewerUI::TreeViewerUI(TreeViewer* treeViewer): phyObject(treeViewer->getPhyObject()), root(treeViewer->getRoot()), rectRoot(treeViewer->getRoot()), layout(TreeLayout_Rectangular), selectedButton(NULL) {
    contEnabled = false;
    showDistanceLabels = true;
    showNameLabels = true;
    maxNameWidth = 0;
    zoom = 1.0;

    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    setFrameShape(QFrame::NoFrame);
    setTransformationAnchor(QGraphicsView::AnchorUnderMouse);
    setScene(new QGraphicsScene());
    scene()->addItem(root);
    addLegend(treeViewer->getScale());
    updateRect();

    treeViewer->createActions();
    connect(treeViewer->getNameLabelsAction(), SIGNAL(triggered(bool)), SLOT(sl_showNameLabelsTriggered(bool)));
    connect(treeViewer->getDistanceLabelsAction(), SIGNAL(triggered(bool)), SLOT(sl_showDistanceLabelsTriggered(bool)));
    connect(treeViewer->getPrintAction(), SIGNAL(triggered()), SLOT(sl_printTriggered()));
    connect(treeViewer->getCaptureTreeAction(), SIGNAL(triggered()), SLOT(sl_captureTreeTriggered()));
    connect(treeViewer->getExportAction(), SIGNAL(triggered()), SLOT(sl_exportTriggered()));
    connect(treeViewer->getContAction(), SIGNAL(triggered(bool)), SLOT(sl_contTriggered(bool)));
    connect(treeViewer->getRectangularLayoutAction(), SIGNAL(triggered(bool)), SLOT(sl_rectangularLayoutTriggered()));
    connect(treeViewer->getCircularLayoutAction(), SIGNAL(triggered(bool)), SLOT(sl_circularLayoutTriggered()));
    connect(treeViewer->getUnrootedLayoutAction(), SIGNAL(triggered(bool)), SLOT(sl_unrootedLayoutTriggered()));

    buttonPopup = new QMenu(this);
    QAction* chrootAction = buttonPopup->addAction(QObject::tr("set root"));
    connect(chrootAction, SIGNAL(triggered(bool)), SLOT(sl_chrootTriggered()));
    chrootAction->setEnabled(false);
    QAction* swapAction = buttonPopup->addAction(QObject::tr("swap siblings"));
    connect(swapAction, SIGNAL(triggered(bool)), SLOT(sl_swapTriggered()));
    swapAction->setEnabled(false);
    QAction* zoomAction = buttonPopup->addAction(QObject::tr("zoom to"));
    connect(zoomAction, SIGNAL(triggered(bool)), SLOT(sl_zoomTriggered()));
    QAction* collapseAction = buttonPopup->addAction(QObject::tr("collapse"));
    connect(collapseAction, SIGNAL(triggered(bool)), SLOT(sl_collapseTriggered()));
}

TreeViewerUI::~TreeViewerUI() {
    delete scene();
}

void TreeViewerUI::addLegend(qreal scale) {
    static const qreal WIDTH = 30.0;
    qreal d = WIDTH / scale;
    QString str = QString::number(d, 'f', 3);
    int i = str.length() - 1;
    for (; i >= 0 && str[i] == '0'; --i) ;
    if (str[i] == '.')
        --i;
    str.truncate(i + 1);

    legend = new QGraphicsLineItem(0, 0, WIDTH, 0);
    QGraphicsItem* text = new QGraphicsSimpleTextItem(str, legend);
    QRectF rect = text->boundingRect();
    text->setPos(0.5 * (WIDTH - rect.width()), -rect.height());
    scene()->addItem(legend);
}

void TreeViewerUI::wheelEvent(QWheelEvent *we) {
    qreal zoom1 = pow(ZOOM_COEF, we->delta() / 120.0);
    qreal zoom2 = zoom * zoom1;
    zoom2 = qMax(MINIMUM_ZOOM, zoom2);
    zoom2 = qMin(MAXIMUM_ZOOM, zoom2);
    scale(zoom2 / zoom, zoom2 / zoom);
    zoom = zoom2;
    QWidget::wheelEvent(we);
}

void TreeViewerUI::mousePressEvent(QMouseEvent *e) {
    if (!(e->modifiers() & Qt::ShiftModifier)) {
        root->setSelected(false);
        selectedButton = dynamic_cast<GraphicsButtonItem*>(itemAt(e->pos()));
        if (selectedButton != NULL) {
            if (e->button() == Qt::RightButton) {
                buttonPopup->popup(e->globalPos());
            }
        } else {
            if (e->button() == Qt::LeftButton) {
                setDragMode(QGraphicsView::ScrollHandDrag);
            }
        }
    }
    QGraphicsView::mousePressEvent(e);
}

void TreeViewerUI::mouseReleaseEvent(QMouseEvent *e) {
    if (root) {
        setDragMode(QGraphicsView::NoDrag);
    }
    QGraphicsView::mouseReleaseEvent(e);
}

void TreeViewerUI::resizeEvent(QResizeEvent *e) {
    QRectF rect = scene()->sceneRect();
    rect.setWidth(rect.width() / zoom);
    rect.setHeight(rect.height() / zoom);
    rect.moveCenter(scene()->sceneRect().center());
    fitInView(rect, Qt::KeepAspectRatio);
    QGraphicsView::resizeEvent(e);
}

void TreeViewerUI::paint(QPainter &painter) {
    painter.setBrush(QColor(0, 0, 0));
    painter.setFont(TreeViewerUtils::getFont());
    scene()->render(&painter);
}

void TreeViewerUI::updateRect() {
    QRectF rect = root->mapToScene(root->childrenBoundingRect().united(root->boundingRect())).boundingRect();
    rect.setLeft(rect.left() - MARGIN);
    rect.setRight(rect.right() - (showNameLabels ? 0 : maxNameWidth) + MARGIN);
    rect.setTop(rect.top() - MARGIN);
    rect.setBottom(rect.bottom() + legend->childrenBoundingRect().height() + MARGIN);
    legend->setPos(0, rect.bottom() - MARGIN);
    scene()->setSceneRect(rect);
}

void TreeViewerUI::sl_chrootTriggered() {

}

void TreeViewerUI::sl_swapTriggered() {

}

void TreeViewerUI::sl_zoomTriggered() {
    if (selectedButton != NULL) {
        QGraphicsItem* item = selectedButton->parentItem();
        QRectF rect = item->mapRectToScene(item->childrenBoundingRect());
        QRectF rect1 = scene()->sceneRect();
        qreal zoom1 = qMin(rect1.width() / rect.width(), rect1.height() / rect.height());
        zoom *= zoom1;
        fitInView(rect, Qt::KeepAspectRatio);
    }
}

void TreeViewerUI::sl_collapseTriggered() {
    if (selectedButton != NULL) {
        selectedButton->collapse();
    }
}

void TreeViewerUI::sl_captureTreeTriggered() {
    QString format = QString(TreeViewerUtils::IMAGE_FILTERS).section(";;", 4, 4);
    QString fileName = phyObject->getDocument()->getName();
    TreeViewerUtils::saveImageDialog(TreeViewerUtils::IMAGE_FILTERS, fileName, format);
    if (!fileName.isEmpty()) {
        QImage image(scene()->sceneRect().toRect().size(), QImage::Format_RGB32);
        image.fill(0xFFFFFFFF);
        QPainter painter(&image);
        paint(painter);
        bool result = image.save(fileName, format.toAscii().constData());
        if (!result) {
            QMessageBox::critical(this, L10N::errorTitle(), L10N::errorImageSave(fileName, format));
            return;
        }
    }
}

void TreeViewerUI::sl_exportTriggered() {
    QString fileName = phyObject->getDocument()->getName();
    QString format = "SVG - Scalable Vector Graphics (*.svg)";
    TreeViewerUtils::saveImageDialog(format, fileName, format);
    if (!fileName.isEmpty()) {
        QRect rect = scene()->sceneRect().toRect();
        rect.moveTo(0, 0);
        QSvgGenerator generator;
        generator.setFileName(fileName);
        generator.setSize(rect.size());
        generator.setViewBox(rect);

        QPainter painter;
        painter.begin(&generator);
        paint(painter);
        painter.end();
    }
}

void TreeViewerUI::sl_contTriggered(bool on) {
    if (on != contEnabled) {
        contEnabled = on;
        QStack<GraphicsBranchItem*> stack;
        stack.push(root);
        if (root != rectRoot) {
            stack.push(rectRoot);
        }
        while (!stack.empty()) {
            GraphicsBranchItem* item = stack.pop();
            if (item->getNameText() == NULL) {
                foreach (QGraphicsItem* citem, item->childItems()) {
                    GraphicsBranchItem* gbi = dynamic_cast<GraphicsBranchItem*>(citem);
                    if (gbi != NULL) {
                        stack.push(gbi);
                    }
                }
            } else {
                item->setWidth(on ? scene()->sceneRect().width() + scene()->sceneRect().left() - item->getNameText()->scenePos().x() - (showNameLabels ? item->getNameText()->boundingRect().width() + GraphicsBranchItem::TEXT_SPACE : 0) : 0);
            }
        }
        updateRect();
    }
}

void TreeViewerUI::sl_rectangularLayoutTriggered() {
    if (layout != TreeLayout_Rectangular) {
        layout = TreeLayout_Rectangular;
        scene()->removeItem(root);
        root = rectRoot;
        scene()->addItem(root);
        updateRect();
    }
}

void TreeViewerUI::sl_circularLayoutTriggered() {
    if (layout != TreeLayout_Circular) {
        layout = TreeLayout_Circular;
        layoutTask = new CreateCircularBranchesTask(rectRoot);
        connect(layoutTask, SIGNAL(si_stateChanged()), SLOT(sl_layoutRecomputed()));
        TaskScheduler* scheduler = AppContext::getTaskScheduler();
        scheduler->registerTopLevelTask(layoutTask);
    }
}

void TreeViewerUI::sl_unrootedLayoutTriggered() {
    if (layout != TreeLayout_Unrooted) {
        layout = TreeLayout_Unrooted;
        layoutTask = new CreateUnrootedBranchesTask(rectRoot);
        connect(layoutTask, SIGNAL(si_stateChanged()), SLOT(sl_layoutRecomputed()));
        TaskScheduler* scheduler = AppContext::getTaskScheduler();
        scheduler->registerTopLevelTask(layoutTask);
    }
}

void TreeViewerUI::sl_layoutRecomputed() {
    if (layoutTask->getState() != Task::State_Finished || layoutTask->hasErrors()) {
        return;
    }

    scene()->removeItem(root);
    root = layoutTask->getResult();
    scene()->addItem(root);
    updateRect();

    if (!showNameLabels || !showDistanceLabels) {
        LabelTypes lt;
        if (!showDistanceLabels) {
            lt |= LabelType_Distance;
        }
        if (!showNameLabels) {
            lt |= LabelType_SequnceName;
        }
        showLabels(lt);
    }
}

void TreeViewerUI::showLabels(LabelTypes labelTypes) {
    QStack<GraphicsBranchItem*> stack;
    stack.push(root);
    if (root != rectRoot) {
        stack.push(rectRoot);
    }
    while (!stack.isEmpty()) {
        GraphicsBranchItem *node = stack.pop();
        if (labelTypes.testFlag(LabelType_SequnceName)) {
            if (node->getNameText() != NULL) {
                node->getNameText()->setVisible(showNameLabels);
            }
        }
        if (labelTypes.testFlag(LabelType_Distance)) {
            if (node->getDistanceText() != NULL) {
                node->getDistanceText()->setVisible(showDistanceLabels);
            }
        }
        foreach (QGraphicsItem* item, node->childItems()) {
            GraphicsBranchItem *bitem = dynamic_cast<GraphicsBranchItem*>(item);
            if (bitem != NULL) {
                stack.push(bitem);
            }
        }
    }
}

void TreeViewerUI::sl_showNameLabelsTriggered(bool on) {
    if (on != showNameLabels) {
        QRectF rect = sceneRect();
        rect.setWidth(rect.width() + (on ? 1 : -1) * maxNameWidth);
        scene()->setSceneRect(rect);
        showNameLabels = on;
        showLabels(LabelType_SequnceName);

        if (contEnabled) {
            QStack<GraphicsBranchItem*> stack;
            stack.push(root);
            if (root != rectRoot) {
                stack.push(rectRoot);
            }
            while (!stack.empty()) {
                GraphicsBranchItem* item = stack.pop();
                if (item->getNameText() == NULL) {
                    foreach (QGraphicsItem* citem, item->childItems()) {
                        GraphicsBranchItem* gbi = dynamic_cast<GraphicsBranchItem*>(citem);
                        if (gbi != NULL) {
                            stack.push(gbi);
                        }
                    }
                } else {
                    item->setWidth(item->getWidth() + (on ? 1 : -1) * (maxNameWidth - item->getNameText()->boundingRect().width() - 2 * GraphicsBranchItem::TEXT_SPACE));
                }
            }
        }
    }
}

void TreeViewerUI::sl_showDistanceLabelsTriggered(bool on) {
    if (on != showDistanceLabels) {
        showDistanceLabels = on;
        showLabels(LabelType_Distance);
    }
}

void TreeViewerUI::sl_printTriggered() {
    QPrinter printer;
    QPrintDialog dialog(&printer, this);
    if (dialog.exec() != QDialog::Accepted)
        return;

    QPainter painter(&printer);
    paint(painter);
}


}//namespace
