/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DotPlotDialog.h"

#include <util_gui/CreateAnnotationWidgetController.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>

namespace GB2 {

DotPlotDialog::DotPlotDialog(QWidget *parent, const QList<ADVSequenceObjectContext *> &seq, int minLen, int identity, ADVSequenceObjectContext *sequenceX, ADVSequenceObjectContext *sequenceY)
: QDialog(parent), sequences(seq)
{
    setupUi(this);

	if (sequences.size() <= 0) {
		return;
	}

	// set algorithms
	Q_ASSERT(algoCombo);
    algoCombo->addItem(tr("Auto"), RFAlgorithm_Auto);
    algoCombo->addItem(tr("Suffix index"), RFAlgorithm_Suffix);
    algoCombo->addItem(tr("Diagonals"), RFAlgorithm_Diagonal);

	Q_ASSERT(xAxisCombo);
	Q_ASSERT(yAxisCombo);

	int xSeqIndex=-1, ySeqIndex=-1;
	int curIndex = 0;
	foreach (ADVSequenceObjectContext *s, sequences) {
		Q_ASSERT(s);

		xAxisCombo->addItem(s->getSequenceGObject()->getGObjectName());
		yAxisCombo->addItem(s->getSequenceGObject()->getGObjectName());

		if (sequenceX == s) {
			xSeqIndex = curIndex;
		}
		if (sequenceY == s) {
			ySeqIndex = curIndex;
		}

		curIndex++;
	}

	// choose the second sequence for Y axis
	if (sequences.size() > 1) {
		yAxisCombo->setCurrentIndex(1);
	}

	if (xSeqIndex >= 0) {
		xAxisCombo->setCurrentIndex(xSeqIndex);
	}
	if (ySeqIndex >= 0) {
		yAxisCombo->setCurrentIndex(ySeqIndex);
	}

	Q_ASSERT(minLenBox);
	Q_ASSERT(identityBox);
	minLenBox->setValue(minLen);
	identityBox->setValue(identity);

    connect(minLenHeuristicsButton, SIGNAL(clicked()), SLOT(sl_minLenHeuristics()));
	connect(hundredPercentButton, SIGNAL(clicked()), SLOT(sl_hundredPercent()));
}

void DotPlotDialog::accept() {

	xSeq = sequences.at(xAxisCombo->currentIndex());
	ySeq = sequences.at(yAxisCombo->currentIndex());

	Q_ASSERT(xSeq);
	Q_ASSERT(ySeq);

    QDialog::accept();
}

void DotPlotDialog::sl_minLenHeuristics() {
	Q_ASSERT(identityBox);
    identityBox->setValue(100);

    // formula used here: nVariations / lenVariations = wantedResCount (==1000)
    // where nVariations == area size
    // lenVariations = 4^len where len is result
    // so we have len = ln(nVariations/wantedResCount)/ln(4)

	Q_ASSERT(xAxisCombo);
	Q_ASSERT(yAxisCombo);
	ADVSequenceObjectContext *xSequence = sequences.at(xAxisCombo->currentIndex());
	ADVSequenceObjectContext *ySequence = sequences.at(yAxisCombo->currentIndex());

	Q_ASSERT(xSequence);
	Q_ASSERT(ySequence);
	qint64 xSeqLen = xSequence->getSequenceLen();
	qint64 ySeqLen = ySequence->getSequenceLen();

    double nVariations = xSeqLen*ySeqLen;
    double resCount = 1000;
	Q_ASSERT(resCount);
    double len = log(nVariations / resCount) / log(double(4));

	Q_ASSERT(minLenBox);
    minLenBox->setValue((int)len);
}

void DotPlotDialog::sl_hundredPercent() {

	Q_ASSERT(identityBox);
	identityBox->setValue(100);
}

int DotPlotDialog::getMismatches() const {

	Q_ASSERT(identityBox);
	Q_ASSERT(minLenBox);
	return (100-identityBox->value()) * minLenBox->value()/ 100;
}

// which algorithm
RFAlgorithm DotPlotDialog::getAlgo() const {

	Q_ASSERT(algoCheck);
	if (algoCheck->isChecked()) {

		Q_ASSERT(algoCombo);
		int index = algoCombo->currentIndex();
		return RFAlgorithm(algoCombo->itemData(index).toInt());
	}

	return RFAlgorithm_Auto;
}

int DotPlotDialog::getMinLen() const {

	Q_ASSERT(minLenBox);
	return minLenBox->value();
}

}//namespace
